/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * 
 */
package jp.mosp.time.bean.impl;

import java.sql.Connection;
import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.MospException;
import jp.mosp.framework.base.MospParams;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.base.PlatformBean;
import jp.mosp.platform.bean.human.EntranceReferenceBeanInterface;
import jp.mosp.platform.bean.human.RetirementReferenceBeanInterface;
import jp.mosp.platform.bean.workflow.WorkflowReferenceBeanInterface;
import jp.mosp.platform.constant.PlatformConst;
import jp.mosp.platform.constant.PlatformMessageConst;
import jp.mosp.platform.dto.workflow.WorkflowDtoInterface;
import jp.mosp.time.base.TimeBean;
import jp.mosp.time.bean.ApplicationReferenceBeanInterface;
import jp.mosp.time.bean.AttendanceReferenceBeanInterface;
import jp.mosp.time.bean.AttendanceRegistBeanInterface;
import jp.mosp.time.bean.DifferenceRequestReferenceBeanInterface;
import jp.mosp.time.bean.HolidayRequestReferenceBeanInterface;
import jp.mosp.time.bean.TimeSettingReferenceBeanInterface;
import jp.mosp.time.bean.TotalTimeEmployeeTransactionReferenceBeanInterface;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.constant.TimeMessageConst;
import jp.mosp.time.dao.settings.AttendanceDaoInterface;
import jp.mosp.time.dao.settings.HolidayRequestDaoInterface;
import jp.mosp.time.dao.settings.OvertimeRequestDaoInterface;
import jp.mosp.time.dao.settings.SubHolidayRequestDaoInterface;
import jp.mosp.time.dao.settings.SubstituteDaoInterface;
import jp.mosp.time.dto.settings.ApplicationDtoInterface;
import jp.mosp.time.dto.settings.AttendanceDtoInterface;
import jp.mosp.time.dto.settings.DifferenceRequestDtoInterface;
import jp.mosp.time.dto.settings.HolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.OvertimeRequestDtoInterface;
import jp.mosp.time.dto.settings.SubHolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.SubstituteDtoInterface;
import jp.mosp.time.dto.settings.TimeSettingDtoInterface;
import jp.mosp.time.dto.settings.impl.TmdAttendanceDto;

/**
 * 勤怠データ登録クラス。
 */
public class AttendanceRegistBean extends TimeBean implements AttendanceRegistBeanInterface {
	
	/**
	 * 勤怠データマスタDAOクラス。<br>
	 */
	private AttendanceDaoInterface								dao;
	/**
	 * 勤怠データ参照インターフェース。<br>
	 */
	private AttendanceReferenceBeanInterface					attendanceReference;
	/**
	 * 設定適用管理参照インターフェース。<br>
	 */
	private ApplicationReferenceBeanInterface					applicationReference;
	/**
	 * 勤怠設定参照インターフェース。<br>
	 */
	private TimeSettingReferenceBeanInterface					timeSettingReference;
	/**
	 * 休暇申請参照インターフェース。<br>
	 */
	private HolidayRequestReferenceBeanInterface				holidayRequestReference;
	/**
	 * ワークフロー参照インターフェース。<br>
	 */
	private WorkflowReferenceBeanInterface						workflowReference;
	/**
	 * 残業申請DAOインターフェース。<br>
	 */
	private OvertimeRequestDaoInterface							overtimeDao;
	/**
	 * 時差出勤申請参照インターフェース。<br>
	 */
	private DifferenceRequestReferenceBeanInterface				differenceReference;
	/**
	 * 休暇申請参照インターフェース。<br>
	 */
	private HolidayRequestDaoInterface							holidayDao;
	/**
	 * 振替休日データDAO。
	 */
	private SubstituteDaoInterface								substituteDao;
	/**
	 * 代休申請参照インターフェース。
	 */
	private SubHolidayRequestDaoInterface						subHolidayDao;
	/**
	 * 社員勤怠集計管理参照クラス。
	 */
	private TotalTimeEmployeeTransactionReferenceBeanInterface	totalTimeEmployeeReference;
	/**
	 * 人事入社情報参照インターフェース
	 */
	private EntranceReferenceBeanInterface						entranceReference;
	/**
	 * 人事退職情報参照クラス。<br>
	 */
	private RetirementReferenceBeanInterface					retirementReference;
	

	/**
	 * 勤怠設定参照インターフェース。<br>
	 */
	
	/**
	 * {@link PlatformBean#PlatformBean()}を実行する。<br>
	 */
	public AttendanceRegistBean() {
		super();
	}
	
	/**
	 * {@link PlatformBean#PlatformBean(MospParams, Connection)}を実行する。<br>
	 * @param mospParams MosPパラメータクラス
	 * @param connection DBコネクション
	 */
	public AttendanceRegistBean(MospParams mospParams, Connection connection) {
		super(mospParams, connection);
	}
	
	@Override
	public void initBean() throws MospException {
		dao = (AttendanceDaoInterface)createDao(AttendanceDaoInterface.class);
		attendanceReference = (AttendanceReferenceBeanInterface)createBean(AttendanceReferenceBeanInterface.class);
		applicationReference = (ApplicationReferenceBeanInterface)createBean(ApplicationReferenceBeanInterface.class);
		timeSettingReference = (TimeSettingReferenceBeanInterface)createBean(TimeSettingReferenceBeanInterface.class);
		holidayRequestReference = (HolidayRequestReferenceBeanInterface)createBean(HolidayRequestReferenceBeanInterface.class);
		workflowReference = (WorkflowReferenceBeanInterface)createBean(WorkflowReferenceBeanInterface.class);
		overtimeDao = (OvertimeRequestDaoInterface)createDao(OvertimeRequestDaoInterface.class);
		differenceReference = (DifferenceRequestReferenceBeanInterface)createBean(DifferenceRequestReferenceBeanInterface.class);
		holidayDao = (HolidayRequestDaoInterface)createDao(HolidayRequestDaoInterface.class);
		substituteDao = (SubstituteDaoInterface)createDao(SubstituteDaoInterface.class);
		subHolidayDao = (SubHolidayRequestDaoInterface)createDao(SubHolidayRequestDaoInterface.class);
		// 社員勤怠集計管理参照クラス
		totalTimeEmployeeReference = (TotalTimeEmployeeTransactionReferenceBeanInterface)createBean(TotalTimeEmployeeTransactionReferenceBeanInterface.class);
		entranceReference = (EntranceReferenceBeanInterface)createBean(EntranceReferenceBeanInterface.class);
		retirementReference = (RetirementReferenceBeanInterface)createBean(RetirementReferenceBeanInterface.class);
	}
	
	@Override
	public AttendanceDtoInterface getInitDto() {
		return new TmdAttendanceDto();
	}
	
	@Override
	public void regist(AttendanceDtoInterface dto) throws MospException {
		if (dao.findForKey(dto.getPersonalId(), dto.getWorkDate(), dto.getTimesWork()) == null) {
			// 新規登録
			insert(dto);
		} else {
			// 履歴追加
			add(dto);
		}
	}
	
	@Override
	public void insert(AttendanceDtoInterface dto) throws MospException {
		// DTOの妥当性確認
		validate(dto);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// ロック対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		// 新規登録情報の検証
		checkInsert(dto);
		if (mospParams.hasErrorMessage()) {
			// ロック解除
			unLockTable();
			return;
		}
		// レコード識別ID最大値をインクリメントしてDTOに設定
		dto.setTmdAttendanceId(findForMaxId(dao) + 1);
		// 登録処理
		dao.insert(dto);
		// ロック解除
		unLockTable();
	}
	
	@Override
	public void add(AttendanceDtoInterface dto) throws MospException {
		// DTOの妥当性確認
		validate(dto);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// ロック対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		// 履歴追加情報の検証
		checkAdd(dto);
		if (mospParams.hasErrorMessage()) {
			// ロック解除
			unLockTable();
			return;
		}
		// 論理削除
		logicalDelete(dao, dto.getTmdAttendanceId());
		// レコード識別ID最大値をインクリメントしてDTOに設定
		dto.setTmdAttendanceId(findForMaxId(dao) + 1);
		// 登録処理
		dao.insert(dto);
		// ロック解除
		unLockTable();
	}
	
	/**
	 * 新規登録時の確認処理を行う。<br>
	 * @param dto 対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	protected void checkInsert(AttendanceDtoInterface dto) throws MospException {
		// 対象レコードの有効日が重複していないかを確認
		checkDuplicateInsert(dao.findForHistory(dto.getPersonalId(), dto.getWorkDate(), dto.getTimesWork()));
	}
	
	/**
	 * 履歴追加時の確認処理を行う。<br>
	 * @param dto 対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	protected void checkAdd(AttendanceDtoInterface dto) throws MospException {
		// 対象レコード識別IDのデータが削除されていないかを確認
		checkExclusive(dao, dto.getTmdAttendanceId());
	}
	
	@Override
	public void validate(AttendanceDtoInterface dto) throws MospException {
		// 時刻の必須チェック
		checkTimeExist(dto);
		// 時刻妥当性チェック
		checkTimeValidity(dto);
		// 基本情報のチェック
		attendanceReference.chkBasicInfo(dto.getPersonalId(), dto.getWorkDate());
	}
	
	@Override
	public void checkDraft(AttendanceDtoInterface dto) throws MospException {
		checkTemporaryClosingFinal(dto);
	}
	
	@Override
	public void checkAppli(AttendanceDtoInterface dto) throws MospException {
		// 下書き同様の処理を行う。
		checkDraft(dto);
		// 申請期間チェック。
		checkPeriod(dto);
		// 遅刻の限度チェック
		checkLateTime(dto);
		// 早退の限度チェック		
		checkLeaveEarlyTime(dto);
		// 申請チェック
		checkRequest(dto);
		// 勤怠の申請チェック
		checkAttendance(dto);
	}
	
	@Override
	public void checkApproval(AttendanceDtoInterface dto) throws MospException {
		// 申請時と同様の処理を行う
		checkAppli(dto);
	}
	
	@Override
	public void checkCancel(AttendanceDtoInterface dto) {
		// 現在処理無し。処理が必要になった場合追加される予定。
		
	}
	
	@Override
	public void checkPeriod(AttendanceDtoInterface dto) {
		if (dto.getWorkDate().after(DateUtility.addMonth(DateUtility.getSystemDate(), 1))) {
			addAttendancePeriodErrorMessage();
		}
	}
	
	@Override
	public void checkLateTime(AttendanceDtoInterface dto) throws MospException {
		// 設定適用の取得
		ApplicationDtoInterface applicationDto = applicationReference.findForPerson(dto.getPersonalId(), dto
			.getWorkDate());
		applicationReference.chkExistApplication(applicationDto, dto.getWorkDate());
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// 勤怠設定の取得
		TimeSettingDtoInterface timeSettingDto = timeSettingReference.getTimeSettingInfo(applicationDto
			.getWorkSettingCode(), dto.getWorkDate());
		timeSettingReference.chkExistTimeSetting(timeSettingDto, dto.getWorkDate());
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// 限度の超過チェック
		if (checkLateLimit(dto, timeSettingDto)) {
			// 超過していた場合、休暇申請の取得
			List<HolidayRequestDtoInterface> holidayRequestListDto = holidayRequestReference.getHolidayRequestList(dto
				.getPersonalId(), dto.getWorkDate());
			// 休暇申請チェック
			if (holidayRequestListDto != null && holidayRequestListDto.size() != 0) {
				// 午前休の申請チェック
				for (int i = 0; i < holidayRequestListDto.size(); i++) {
					if (holidayRequestListDto.get(i).getHolidayRange() == 2) {
						WorkflowDtoInterface workflowDto = workflowReference
							.getLatestWorkflowInfo(holidayRequestListDto.get(i).getWorkflow());
						// 下書の場合は参照しない
						if (workflowDto != null
								&& !workflowDto.getWorkflowStatus().equals(PlatformConst.CODE_STATUS_DRAFT)) {
							// 	1件でも午前休があればエラーメッセージ無し
							return;
						}
					}
				}
			}
			// メッセージ設定
			String[] rep = { mospParams.getName("StartWork"), mospParams.getName("Tardiness"),
				mospParams.getName("AmRest") };
			mospParams.addErrorMessage(TimeMessageConst.MSG_TARDINESS_LEAVE_EARLY_LIMIT_OVER, rep);
		}
	}
	
	@Override
	public void checkLeaveEarlyTime(AttendanceDtoInterface dto) throws MospException {
		// 設定適用の取得
		ApplicationDtoInterface applicationDto = applicationReference.findForPerson(dto.getPersonalId(), dto
			.getWorkDate());
		applicationReference.chkExistApplication(applicationDto, dto.getWorkDate());
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// 勤怠設定の取得
		TimeSettingDtoInterface timeSettingDto = timeSettingReference.getTimeSettingInfo(applicationDto
			.getWorkSettingCode(), dto.getWorkDate());
		timeSettingReference.chkExistTimeSetting(timeSettingDto, dto.getWorkDate());
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// 限度の超過チェック
		if (checkLeaveEarlyLimit(dto, timeSettingDto)) {
			// 超過していた場合、休暇申請の取得
			List<HolidayRequestDtoInterface> holidayRequestListDto = holidayRequestReference.getHolidayRequestList(dto
				.getPersonalId(), dto.getWorkDate());
			// 休暇申請チェック
			if (holidayRequestListDto != null && holidayRequestListDto.size() != 0) {
				// 午後休の申請チェック
				for (int i = 0; i < holidayRequestListDto.size(); i++) {
					if (holidayRequestListDto.get(i).getHolidayRange() == 3) {
						WorkflowDtoInterface workflowDto = workflowReference
							.getLatestWorkflowInfo(holidayRequestListDto.get(i).getWorkflow());
						// 下書の場合は参照しない
						if (workflowDto != null
								&& !workflowDto.getWorkflowStatus().equals(PlatformConst.CODE_STATUS_DRAFT)) {
							// 	1件でも午後休があればエラーメッセージ無し
							return;
						}
					}
				}
			}
			String[] rep = { mospParams.getName("EndWork"), mospParams.getName("LeaveEarly"),
				mospParams.getName("PmRest") };
			mospParams.addErrorMessage(TimeMessageConst.MSG_TARDINESS_LEAVE_EARLY_LIMIT_OVER, rep);
		}
	}
	
	/**
	 * @param dto 勤怠データ
	 * @param timeSettingDto 勤怠設定
	 * @return 限度時間を超えた場合true それ以外false
	 */
	public boolean checkLateLimit(AttendanceDtoInterface dto, TimeSettingDtoInterface timeSettingDto) {
		int lateEarlyHalf = (DateUtility.getHour(timeSettingDto.getLateEarlyHalf()) * TimeConst.CODE_DEFINITION_HOUR)
				+ DateUtility.getMinute(timeSettingDto.getLateEarlyHalf());
		if (lateEarlyHalf < dto.getLateTime()) {
			return true;
		}
		return false;
	}
	
	/**
	 * @param dto 勤怠データ
	 * @param timeSettingDto 勤怠設定
	 * @return 限度時間を超えた場合true それ以外false
	 */
	public boolean checkLeaveEarlyLimit(AttendanceDtoInterface dto, TimeSettingDtoInterface timeSettingDto) {
		int lateEarlyHalf = (DateUtility.getHour(timeSettingDto.getLateEarlyHalf()) * TimeConst.CODE_DEFINITION_HOUR)
				+ DateUtility.getMinute(timeSettingDto.getLateEarlyHalf());
		if (lateEarlyHalf < dto.getLeaveEarlyTime()) {
			return true;
		}
		return false;
	}
	
	public void checkRequest(AttendanceDtoInterface dto) throws MospException {
		String personalId = dto.getPersonalId();
		Date requestDate = dto.getWorkDate();
		String requestName = "";
		// 残業申請情報 
		List<OvertimeRequestDtoInterface> overtimeList = overtimeDao.findForList(personalId, requestDate);
		for (OvertimeRequestDtoInterface requestDto : overtimeList) {
			WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(requestDto.getWorkflow());
			if (workflowDto == null) {
				continue;
			}
			if (PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
				// 承認状況が取下の場合
				continue;
			}
			if (PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())) {
				// 承認状況が下書の場合
				continue;
			}
			if (!PlatformConst.CODE_STATUS_COMPLETE.equals(workflowDto.getWorkflowStatus())) {
				// 承認状況が承認済でない場合は、エラーメッセージの表示
				// 残業申請のメッセージ追加
				requestName = mospParams.getName("OvertimeWork") + mospParams.getName("Application");
				mospParams.addErrorMessage(TimeMessageConst.MSG_NOT_REQUEST_STATE_COMPLETE, DateUtility
					.getStringDate(requestDate), requestName);
			}
		}
		// 時差出勤申請情報 
		DifferenceRequestDtoInterface differenceDto = differenceReference.findForKeyOnWorkflow(personalId, requestDate);
		if (differenceDto != null) {
			WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(differenceDto.getWorkflow());
			if (workflowDto != null) {
				if (!PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())
						&& !PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
						&& !PlatformConst.CODE_STATUS_COMPLETE.equals(workflowDto.getWorkflowStatus())) {
					// 承認状況が取下でなく、下書でなく、承認済でない場合は、エラーメッセージの表示
					// 時差出勤申請のメッセージ追加
					requestName = mospParams.getName("TimeDifference") + mospParams.getName("GoingWork")
							+ mospParams.getName("Application");
					mospParams.addErrorMessage(TimeMessageConst.MSG_NOT_REQUEST_STATE_COMPLETE, DateUtility
						.getStringDate(requestDate), requestName);
				}
			}
		}
		// 休暇申請情報（半休）
		List<HolidayRequestDtoInterface> holidayList = holidayDao.findForList(personalId, requestDate);
		for (HolidayRequestDtoInterface requestDto : holidayList) {
			int holidayRange = requestDto.getHolidayRange();
			if (holidayRange == TimeConst.CODE_HOLIDAY_RANGE_AM || holidayRange == TimeConst.CODE_HOLIDAY_RANGE_PM) {
				// 休暇が午前休又は午後休の場合
				WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(requestDto.getWorkflow());
				if (workflowDto == null) {
					continue;
				}
				if (PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
					// 承認状況が取下の場合
					continue;
				}
				if (PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())) {
					// 承認状況が下書の場合
					continue;
				}
				if (!PlatformConst.CODE_STATUS_COMPLETE.equals(workflowDto.getWorkflowStatus())) {
					// 承認状況が承認済でない場合は、エラーメッセージの表示
					// 休暇申請のメッセージ追加
					requestName = mospParams.getName("Vacation") + mospParams.getName("Application");
					mospParams.addErrorMessage(TimeMessageConst.MSG_NOT_REQUEST_STATE_COMPLETE, DateUtility
						.getStringDate(requestDate), requestName);
				}
			}
		}
		// 休日出勤申請に伴う振替申請情報（半休）
		List<SubstituteDtoInterface> substituteList = substituteDao.findForList(personalId, requestDate);
		for (SubstituteDtoInterface substituteDto : substituteList) {
			int range = substituteDto.getSubstituteRange();
			if (range == TimeConst.CODE_HOLIDAY_RANGE_AM || range == TimeConst.CODE_HOLIDAY_RANGE_PM) {
				// 振替日が午前休又は午後休の場合
				WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(substituteDto.getWorkflow());
				if (workflowDto == null) {
					continue;
				}
				if (PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
					// 承認状況が取下の場合
					continue;
				}
				if (PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())) {
					// 承認状況が下書の場合
					continue;
				}
				if (!PlatformConst.CODE_STATUS_COMPLETE.equals(workflowDto.getWorkflowStatus())) {
					// 承認状況が承認済でない場合は、エラーメッセージの表示
					// 休日出勤申請のメッセージ追加
					requestName = mospParams.getName("Holiday") + mospParams.getName("GoingWork")
							+ mospParams.getName("Application");
					mospParams.addErrorMessage(TimeMessageConst.MSG_NOT_REQUEST_STATE_COMPLETE, DateUtility
						.getStringDate(requestDate), requestName);
				}
			}
		}
		// 代休申請情報（半休）
		List<SubHolidayRequestDtoInterface> subHolidayList = subHolidayDao.findForList(personalId, requestDate);
		for (SubHolidayRequestDtoInterface requestDto : subHolidayList) {
			int holidayRange = requestDto.getHolidayRange();
			if (holidayRange == TimeConst.CODE_HOLIDAY_RANGE_AM || holidayRange == TimeConst.CODE_HOLIDAY_RANGE_PM) {
				// 代休が午前休又は午後休の場合
				WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(requestDto.getWorkflow());
				if (workflowDto == null) {
					continue;
				}
				if (PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
					// 承認状況が取下の場合
					continue;
				}
				if (PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())) {
					// 承認状況が下書の場合
					continue;
				}
				if (!PlatformConst.CODE_STATUS_COMPLETE.equals(workflowDto.getWorkflowStatus())) {
					// 承認状況が承認済でない場合は、エラーメッセージの表示
					// 代休申請のメッセージ追加
					requestName = mospParams.getName("CompensatoryHoliday") + mospParams.getName("Application");
					mospParams.addErrorMessage(TimeMessageConst.MSG_NOT_REQUEST_STATE_COMPLETE, DateUtility
						.getStringDate(requestDate), requestName);
				}
			}
		}
	}
	
	/**
	 * @param dto 対象DTO
	 */
	public void checkTimeExist(AttendanceDtoInterface dto) {
		if (dto.getStartTime() == null && dto.getEndTime() == null) {
			// 始業時刻、終了時刻両方共無い場合
			// エラーメッセージ設定
			mospParams.addErrorMessage(TimeMessageConst.MSG_DRAFT_TIME_NOT_INPUT, null);
		}
	}
	
	/**
	 * @param dto 対象DTO
	 */
	public void checkTimeValidity(AttendanceDtoInterface dto) {
		if (dto.getStartTime() == null || dto.getEndTime() == null) {
			return;
		}
		if (setHourMinute(dto.getEndTime(), dto.getStartTime()) < setHourMinute(dto.getStartTime())) {
			// エラーメッセージ設定
			addInvalidOrderMessage(mospParams.getName("StartWork"), mospParams.getName("EndWork"));
		}
	}
	
	@Override
	public void checkAttendance(AttendanceDtoInterface dto) throws MospException {
		AttendanceDtoInterface attendanceDto = dao.findForKey(dto.getPersonalId(), dto.getWorkDate(), 1);
		if (attendanceDto == null) {
			return;
		}
		WorkflowDtoInterface workflowDto = workflowReference.getLatestWorkflowInfo(attendanceDto.getWorkflow());
		if (workflowDto == null) {
			return;
		}
		if (PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
			// 取下の場合
			return;
		}
		if (PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())) {
			// 下書の場合
			return;
		}
		if (PlatformConst.CODE_STATUS_REVERT.equals(workflowDto.getWorkflowStatus())) {
			// 差戻の場合
			if (workflowDto.getWorkflowStage() == PlatformConst.WORKFLOW_STAGE_ZERO) {
				// ワークフロー段階が0の場合
				return;
			}
		}
		if (dto.getWorkflow() == workflowDto.getWorkflow()) {
			// ワークフロー番号が同じ場合は同じ申請
			return;
		}
		addAttendanceTargetWorkDateAttendanceRequestErrorMessage(dto.getWorkDate());
	}
	
	@Override
	public void checkTemporaryClosingFinal(AttendanceDtoInterface dto) throws MospException {
		Integer state = totalTimeEmployeeReference.getCutoffState(dto.getPersonalId(), DateUtility.getYear(dto
			.getWorkDate()), DateUtility.getMonth(dto.getWorkDate()));
		if (state == null) {
			return;
		}
		if (state != 0) {
			// 未締でない場合
			addMonthlyTreatmentErrorMessage(dto.getWorkDate(), mospParams.getName("GoingWork")
					+ mospParams.getName("Day"));
		}
	}
	
	@Override
	public void checkApprover(AttendanceDtoInterface dto, WorkflowDtoInterface workDto) throws MospException {
		String approverId = workDto.getApproverId();
		// 自己承認以外
		if ("SELF_APPRO".equals(approverId)) {
			return;
		}
		
		String[] arrayApproverId = approverId.split(",");
		for (String element : arrayApproverId) {
			// 対象データが1件しかない場合のチェック
			if ("".equals(element)) {
				if (0 >= arrayApproverId.length) {
					mospParams.addErrorMessage(PlatformMessageConst.MSG_NOT_EXIST_AND_CHANGE_SOMETHING, mospParams
						.getName("Approver"), mospParams.getName("Work") + mospParams.getName("Day"));
				}
			} else {
				// 入社チェック
				if (!entranceReference.isEntered(element, dto.getWorkDate())) {
					mospParams.addErrorMessage(PlatformMessageConst.MSG_NOT_EXIST_AND_CHANGE_SOMETHING, mospParams
						.getName("Approver"), mospParams.getName("Work") + mospParams.getName("Day"));
				}
				// 退社チェック
				if (retirementReference.isRetired(element, dto.getWorkDate())) {
					mospParams.addErrorMessage(PlatformMessageConst.MSG_NOT_EXIST_AND_CHANGE_SOMETHING, mospParams
						.getName("Approver"), mospParams.getName("Work") + mospParams.getName("Day"));
				}
			}
		}
	}
	
}
