/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.bean.impl;

import java.sql.Connection;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import jp.mosp.framework.base.BaseDtoInterface;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.base.MospParams;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.base.PlatformBean;
import jp.mosp.platform.bean.human.HumanReferenceBeanInterface;
import jp.mosp.platform.bean.system.SectionReferenceBeanInterface;
import jp.mosp.platform.bean.workflow.WorkflowCommentReferenceBeanInterface;
import jp.mosp.platform.bean.workflow.WorkflowIntegrateBeanInterface;
import jp.mosp.platform.constant.PlatformConst;
import jp.mosp.platform.dto.human.HumanDtoInterface;
import jp.mosp.platform.dto.system.SectionDtoInterface;
import jp.mosp.platform.dto.workflow.RouteApplicationDtoInterface;
import jp.mosp.platform.dto.workflow.SubApproverDtoInterface;
import jp.mosp.platform.dto.workflow.WorkflowCommentDtoInterface;
import jp.mosp.platform.dto.workflow.WorkflowDtoInterface;
import jp.mosp.time.bean.ApprovalInfoReferenceBeanInterface;
import jp.mosp.time.bean.AttendanceReferenceBeanInterface;
import jp.mosp.time.bean.DifferenceRequestReferenceBeanInterface;
import jp.mosp.time.bean.HolidayRequestReferenceBeanInterface;
import jp.mosp.time.bean.OvertimeRequestReferenceBeanInterface;
import jp.mosp.time.bean.SubHolidayRequestReferenceBeanInterface;
import jp.mosp.time.bean.WorkOnHolidayRequestReferenceBeanInterface;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.dao.settings.HolidayDaoInterface;
import jp.mosp.time.dto.settings.AttendanceDtoInterface;
import jp.mosp.time.dto.settings.DifferenceRequestDtoInterface;
import jp.mosp.time.dto.settings.HolidayDtoInterface;
import jp.mosp.time.dto.settings.HolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.ManagementRequestListDtoInterface;
import jp.mosp.time.dto.settings.OvertimeRequestDtoInterface;
import jp.mosp.time.dto.settings.RequestListDtoInterface;
import jp.mosp.time.dto.settings.SubHolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.WorkOnHolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.impl.ManagementRequestListDto;

/**
 * 承認情報参照クラス。
 */
public class ApprovalInfoReferenceBean extends PlatformBean implements ApprovalInfoReferenceBeanInterface {
	
	/**
	 * 区切文字(申請情報詳細)。
	 */
	public static final String								SEPARATOR_REQUEST_INFO	= " ";
	
	/**
	 * 休暇種別管理DAO
	 */
	protected HolidayDaoInterface							holidayDao;
	/**
	 * 人事マスタ参照
	 */
	protected HumanReferenceBeanInterface					humanReference;
	/**
	 * 所属マスタ参照
	 */
	protected SectionReferenceBeanInterface					sectionReference;
	/**
	 * 勤怠データ参照
	 */
	protected AttendanceReferenceBeanInterface				attendanceReference;
	/**
	 * 残業申請参照
	 */
	protected OvertimeRequestReferenceBeanInterface			overtimeRequest;
	/**
	 * 休暇申請参照
	 */
	protected HolidayRequestReferenceBeanInterface			holidayRequest;
	/**
	 * 休日出勤申請参照
	 */
	protected WorkOnHolidayRequestReferenceBeanInterface	workOnHolidayRequest;
	/**
	 * 代休申請参照
	 */
	protected SubHolidayRequestReferenceBeanInterface		subHolidayRequest;
	/**
	 * 時差出勤申請参照
	 */
	protected DifferenceRequestReferenceBeanInterface		differenceRequest;
	/**
	 * ワークフローコメント参照
	 */
	protected WorkflowCommentReferenceBeanInterface			workflowCommentReference;
	
	/**
	 * ワークフロー統括
	 */
	protected WorkflowIntegrateBeanInterface				workflowIntegrate;
	

	/**
	 * {@link PlatformBean#PlatformBean()}を実行する。<br>
	 */
	public ApprovalInfoReferenceBean() {
		super();
	}
	
	/**
	 * {@link PlatformBean#PlatformBean(MospParams, Connection)}を実行する。<br>
	 * @param mospParams MosPパラメータクラス
	 * @param connection DBコネクション
	 */
	public ApprovalInfoReferenceBean(MospParams mospParams, Connection connection) {
		super(mospParams, connection);
	}
	
	@Override
	public void initBean() throws MospException {
		humanReference = (HumanReferenceBeanInterface)createBean(HumanReferenceBeanInterface.class);
		sectionReference = (SectionReferenceBeanInterface)createBean(SectionReferenceBeanInterface.class);
		attendanceReference = (AttendanceReferenceBeanInterface)createBean(AttendanceReferenceBeanInterface.class);
		overtimeRequest = (OvertimeRequestReferenceBeanInterface)createBean(OvertimeRequestReferenceBeanInterface.class);
		holidayRequest = (HolidayRequestReferenceBeanInterface)createBean(HolidayRequestReferenceBeanInterface.class);
		workOnHolidayRequest = (WorkOnHolidayRequestReferenceBeanInterface)createBean(WorkOnHolidayRequestReferenceBeanInterface.class);
		subHolidayRequest = (SubHolidayRequestReferenceBeanInterface)createBean(SubHolidayRequestReferenceBeanInterface.class);
		differenceRequest = (DifferenceRequestReferenceBeanInterface)createBean(DifferenceRequestReferenceBeanInterface.class);
		workflowCommentReference = (WorkflowCommentReferenceBeanInterface)createBean(WorkflowCommentReferenceBeanInterface.class);
		workflowIntegrate = (WorkflowIntegrateBeanInterface)createBean(WorkflowIntegrateBeanInterface.class);
		holidayDao = (HolidayDaoInterface)createDao(HolidayDaoInterface.class);
	}
	
	@Override
	public Map<String, List<ManagementRequestListDtoInterface>> getNonApprovedMap(String personalId, String functionCode)
			throws MospException {
		// 未承認一覧
		Map<String, List<ManagementRequestListDtoInterface>> map = new HashMap<String, List<ManagementRequestListDtoInterface>>();
		boolean addFlag = false;
		if (functionCode.length() == 0) {
			addFlag = true;
		}
		if (addFlag || TimeConst.CODE_FUNCTION_WORK_MANGE.equals(functionCode)) {
			map.put(TimeConst.CODE_FUNCTION_WORK_MANGE, getNonApprovedAttendanceList(personalId));
		}
		if (addFlag || TimeConst.CODE_FUNCTION_OVER_WORK.equals(functionCode)) {
			map.put(TimeConst.CODE_FUNCTION_OVER_WORK, getNonApprovedOvertimeList(personalId));
		}
		if (addFlag || TimeConst.CODE_FUNCTION_VACATION.equals(functionCode)) {
			map.put(TimeConst.CODE_FUNCTION_VACATION, getNonApprovedHolidayList(personalId));
		}
		if (addFlag || TimeConst.CODE_FUNCTION_WORK_HOLIDAY.equals(functionCode)) {
			map.put(TimeConst.CODE_FUNCTION_WORK_HOLIDAY, getNonApprovedWorkOnHolidayList(personalId));
		}
		if (addFlag || TimeConst.CODE_FUNCTION_COMPENSATORY_HOLIDAY.equals(functionCode)) {
			map.put(TimeConst.CODE_FUNCTION_COMPENSATORY_HOLIDAY, getNonApprovedSubHolidayList(personalId));
		}
		if (addFlag || TimeConst.CODE_FUNCTION_DIFFERENCE.equals(functionCode)) {
			map.put(TimeConst.CODE_FUNCTION_DIFFERENCE, getNonApprovedDifferenceList(personalId));
		}
		Date targetDate = DateUtility.getSystemDate();
		// 所属社員未承認一覧
		Map<String, List<ManagementRequestListDtoInterface>> belongMap = getNonApprovedListForBelongSection(personalId,
				targetDate, functionCode, PlatformConst.WORKFLOW_TYPE_TIME);
		Set<Entry<String, List<ManagementRequestListDtoInterface>>> belongSet = belongMap.entrySet();
		for (Entry<String, List<ManagementRequestListDtoInterface>> entry : belongSet) {
			List<ManagementRequestListDtoInterface> list = map.get(entry.getKey());
			list.addAll(entry.getValue());
			map.put(entry.getKey(), list);
		}
		// 下位所属社員未承認一覧
		Map<String, List<ManagementRequestListDtoInterface>> lowerMap = getNonApprovedListForLowerSection(personalId,
				targetDate, functionCode, PlatformConst.WORKFLOW_TYPE_TIME);
		Set<Entry<String, List<ManagementRequestListDtoInterface>>> lowerSet = lowerMap.entrySet();
		for (Entry<String, List<ManagementRequestListDtoInterface>> entry : lowerSet) {
			List<ManagementRequestListDtoInterface> list = map.get(entry.getKey());
			list.addAll(entry.getValue());
			map.put(entry.getKey(), list);
		}
		return map;
	}
	
	@Override
	public List<ManagementRequestListDtoInterface> getNonApprovedAttendanceList(String personalId) throws MospException {
		// 未承認一覧
		List<ManagementRequestListDtoInterface> list = new ArrayList<ManagementRequestListDtoInterface>();
		// 承認対象のワークフロー一覧
		List<WorkflowDtoInterface> workflowList = workflowIntegrate.getListForNonApproved(personalId, DateUtility
			.getSystemDate(), TimeConst.CODE_FUNCTION_WORK_MANGE);
		// 代理確認
		List<SubApproverDtoInterface> subApproverList = workflowIntegrate.findForSubApproverId(personalId,
				PlatformConst.WORKFLOW_TYPE_TIME, DateUtility.getSystemDate(), DateUtility.getSystemDate());
		for (SubApproverDtoInterface subApproverDto : subApproverList) {
			workflowList.addAll(workflowIntegrate.getListForNonApproved(subApproverDto.getPersonalId(), DateUtility
				.getSystemDate(), TimeConst.CODE_FUNCTION_WORK_MANGE));
		}
		// 申請の取得
		for (WorkflowDtoInterface dto : workflowList) {
			ManagementRequestListDtoInterface listDto = convert(attendanceReference.findForWorkflow(dto.getWorkflow()),
					null);
			if (listDto != null) {
				list.add(listDto);
			}
		}
		return list;
	}
	
	@Override
	public List<ManagementRequestListDtoInterface> getNonApprovedOvertimeList(String personalId) throws MospException {
		// 未承認一覧
		List<ManagementRequestListDtoInterface> list = new ArrayList<ManagementRequestListDtoInterface>();
		// 承認対象のワークフロー一覧
		List<WorkflowDtoInterface> workflowList = workflowIntegrate.getListForNonApproved(personalId, DateUtility
			.getSystemDate(), TimeConst.CODE_FUNCTION_OVER_WORK);
		// 代理確認
		List<SubApproverDtoInterface> subApproverList = workflowIntegrate.findForSubApproverId(personalId,
				PlatformConst.WORKFLOW_TYPE_TIME, DateUtility.getSystemDate(), DateUtility.getSystemDate());
		for (SubApproverDtoInterface subApproverDto : subApproverList) {
			workflowList.addAll(workflowIntegrate.getListForNonApproved(subApproverDto.getPersonalId(), DateUtility
				.getSystemDate(), TimeConst.CODE_FUNCTION_OVER_WORK));
		}
		// 申請の取得
		for (WorkflowDtoInterface dto : workflowList) {
			ManagementRequestListDtoInterface listDto = convert(overtimeRequest.findForWorkflow(dto.getWorkflow()),
					null);
			if (listDto != null) {
				list.add(listDto);
			}
		}
		return list;
	}
	
	@Override
	public List<ManagementRequestListDtoInterface> getNonApprovedHolidayList(String personalId) throws MospException {
		// 未承認一覧
		List<ManagementRequestListDtoInterface> list = new ArrayList<ManagementRequestListDtoInterface>();
		// 承認対象のワークフロー一覧
		List<WorkflowDtoInterface> workflowList = workflowIntegrate.getListForNonApproved(personalId, DateUtility
			.getSystemDate(), TimeConst.CODE_FUNCTION_VACATION);
		// 代理確認
		List<SubApproverDtoInterface> subApproverList = workflowIntegrate.findForSubApproverId(personalId,
				PlatformConst.WORKFLOW_TYPE_TIME, DateUtility.getSystemDate(), DateUtility.getSystemDate());
		for (SubApproverDtoInterface subApproverDto : subApproverList) {
			workflowList.addAll(workflowIntegrate.getListForNonApproved(subApproverDto.getPersonalId(), DateUtility
				.getSystemDate(), TimeConst.CODE_FUNCTION_VACATION));
		}
		// 申請の取得
		for (WorkflowDtoInterface dto : workflowList) {
			ManagementRequestListDtoInterface listDto = convert(holidayRequest.findForWorkflow(dto.getWorkflow()), null);
			if (listDto != null) {
				list.add(listDto);
			}
		}
		return list;
	}
	
	@Override
	public List<ManagementRequestListDtoInterface> getNonApprovedWorkOnHolidayList(String personalId)
			throws MospException {
		// 未承認一覧
		List<ManagementRequestListDtoInterface> list = new ArrayList<ManagementRequestListDtoInterface>();
		// 承認対象のワークフロー一覧
		List<WorkflowDtoInterface> workflowList = workflowIntegrate.getListForNonApproved(personalId, DateUtility
			.getSystemDate(), TimeConst.CODE_FUNCTION_WORK_HOLIDAY);
		// 代理確認
		List<SubApproverDtoInterface> subApproverList = workflowIntegrate.findForSubApproverId(personalId,
				PlatformConst.WORKFLOW_TYPE_TIME, DateUtility.getSystemDate(), DateUtility.getSystemDate());
		for (SubApproverDtoInterface subApproverDto : subApproverList) {
			workflowList.addAll(workflowIntegrate.getListForNonApproved(subApproverDto.getPersonalId(), DateUtility
				.getSystemDate(), TimeConst.CODE_FUNCTION_WORK_HOLIDAY));
		}
		// 申請の取得
		for (WorkflowDtoInterface dto : workflowList) {
			ManagementRequestListDtoInterface listDto = convert(
					workOnHolidayRequest.findForWorkflow(dto.getWorkflow()), null);
			if (listDto != null) {
				list.add(listDto);
			}
		}
		return list;
	}
	
	@Override
	public List<ManagementRequestListDtoInterface> getNonApprovedSubHolidayList(String personalId) throws MospException {
		// 未承認一覧
		List<ManagementRequestListDtoInterface> list = new ArrayList<ManagementRequestListDtoInterface>();
		// 承認対象のワークフロー一覧
		List<WorkflowDtoInterface> workflowList = workflowIntegrate.getListForNonApproved(personalId, DateUtility
			.getSystemDate(), TimeConst.CODE_FUNCTION_COMPENSATORY_HOLIDAY);
		// 代理確認
		List<SubApproverDtoInterface> subApproverList = workflowIntegrate.findForSubApproverId(personalId,
				PlatformConst.WORKFLOW_TYPE_TIME, DateUtility.getSystemDate(), DateUtility.getSystemDate());
		for (SubApproverDtoInterface subApproverDto : subApproverList) {
			workflowList.addAll(workflowIntegrate.getListForNonApproved(subApproverDto.getPersonalId(), DateUtility
				.getSystemDate(), TimeConst.CODE_FUNCTION_COMPENSATORY_HOLIDAY));
		}
		// 申請の取得
		for (WorkflowDtoInterface dto : workflowList) {
			ManagementRequestListDtoInterface listDto = convert(subHolidayRequest.findForWorkflow(dto.getWorkflow()),
					null);
			if (listDto != null) {
				list.add(listDto);
			}
		}
		return list;
	}
	
	@Override
	public List<ManagementRequestListDtoInterface> getNonApprovedDifferenceList(String personalId) throws MospException {
		// 未承認一覧
		List<ManagementRequestListDtoInterface> list = new ArrayList<ManagementRequestListDtoInterface>();
		// 承認対象のワークフロー一覧
		List<WorkflowDtoInterface> workflowList = workflowIntegrate.getListForNonApproved(personalId, DateUtility
			.getSystemDate(), TimeConst.CODE_FUNCTION_DIFFERENCE);
		// 代理確認
		List<SubApproverDtoInterface> subApproverList = workflowIntegrate.findForSubApproverId(personalId,
				PlatformConst.WORKFLOW_TYPE_TIME, DateUtility.getSystemDate(), DateUtility.getSystemDate());
		for (SubApproverDtoInterface subApproverDto : subApproverList) {
			workflowList.addAll(workflowIntegrate.getListForNonApproved(subApproverDto.getPersonalId(), DateUtility
				.getSystemDate(), TimeConst.CODE_FUNCTION_DIFFERENCE));
		}
		// 申請の取得
		for (WorkflowDtoInterface dto : workflowList) {
			ManagementRequestListDtoInterface listDto = convert(differenceRequest.findForWorkflow(dto.getWorkflow()),
					null);
			if (listDto != null) {
				list.add(listDto);
			}
		}
		return list;
	}
	
	/**
	 * 申請情報詳細の設定
	 * @param dto 勤怠情報DTO
	 * @return 申請情報詳細
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	public String getAttendanceInfo(AttendanceDtoInterface dto) throws MospException {
		StringBuffer infoStringBuffer = new StringBuffer();
		// 勤怠設定で表示する内容は無し
		infoStringBuffer.append("");
		return infoStringBuffer.toString();
	}
	
	/**
	 * 申請情報詳細の設定
	 * @param dto 残業申請DTO
	 * @return 申請情報詳細
	 */
	public String getOvertimeRequestInfo(OvertimeRequestDtoInterface dto) {
		// 表示例 予定:2時間30分　理由:顧客サポート
		StringBuffer sb = new StringBuffer();
		// 予定
		sb.append(mospParams.getName("Schedule") + mospParams.getName("Colon") + (dto.getRequestTime() / 60)
				+ mospParams.getName("Time") + (dto.getRequestTime() % 60) + mospParams.getName("Minutes"));
		// 空欄
		sb.append(" ");
		// 理由
		sb.append(mospParams.getName("Reason") + mospParams.getName("Colon") + dto.getRequestReason());
		return sb.toString();
	}
	
	/**
	 * 申請情報詳細の設定
	 * @param dto 休暇申請DTO
	 * @return 申請情報詳細
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	public String getHolidayRequestInfo(HolidayRequestDtoInterface dto) throws MospException {
		// 表示例 結婚休暇
		StringBuffer sb = new StringBuffer();
		// 休暇種別管理DTO取得
		HolidayDtoInterface holidayDto = holidayDao.findForKey(dto.getHolidayType2(), dto.getRequestStartDate(), dto
			.getHolidayType1());
		if (holidayDto != null) {
			// 休暇名称
			sb.append(holidayDto.getHolidayName());
			sb.append(SEPARATOR_REQUEST_INFO);
		}
		// 休暇範囲設定
		sb.append(mospParams.getProperties().getCodeItemName(TimeConst.CODE_HOLIDAY_TYPE3_RANGE1,
				String.valueOf(dto.getHolidayRange())));
		return sb.toString();
	}
	
	/**
	 * 申請情報詳細の設定
	 * @param dto 代休申請DTO
	 * @return 申請情報詳細
	 * @throws MospException 例外処理発生時
	 */
	public String getSubHolidayRequestInfo(SubHolidayRequestDtoInterface dto) throws MospException {
		// 表示例 休出:2010/12/1 勤務7時間00分
		StringBuffer sb = new StringBuffer();
		int type = dto.getWorkDateSubHolidayType();
		if (type == 1 || type == 2) {
			// 休出
			sb.append(mospParams.getName("WorkingHoliday"));
		} else if (type == 3) {
			// 深夜
			sb.append(mospParams.getName("Midnight"));
		}
		sb.append(mospParams.getName("Colon"));
		sb.append(DateUtility.getStringDate(dto.getWorkDate()));
		// 空欄
		sb.append(SEPARATOR_REQUEST_INFO);
		sb.append(mospParams.getName("Work"));
		sb.append(mospParams.getName("Colon"));
		AttendanceDtoInterface attendanceDto = attendanceReference
			.findForKey(dto.getPersonalId(), dto.getWorkDate(), 1);
		if (attendanceDto == null) {
			sb.append(0);
			sb.append(mospParams.getName("Time"));
			sb.append("00");
			sb.append(mospParams.getName("Minutes"));
		} else {
			sb.append(attendanceDto.getWorkTime() / 60);
			sb.append(mospParams.getName("Time"));
			sb.append(getMinute(attendanceDto.getWorkTime()));
			sb.append(mospParams.getName("Minutes"));
		}
		// 休暇範囲設定
		sb.append(SEPARATOR_REQUEST_INFO);
		sb.append(mospParams.getProperties().getCodeItemName(TimeConst.CODE_HOLIDAY_TYPE3_RANGE1,
				String.valueOf(dto.getHolidayRange())));
		return sb.toString();
	}
	
	@Override
	public Map<String, List<ManagementRequestListDtoInterface>> getNonApprovedListForBelongSection(String personalId,
			Date targetDate, String functionCode, int workflowType) throws MospException {
		// 承認者人事マスタ
		HumanDtoInterface approverDto = humanReference.getHumanInfo(personalId, targetDate);
		// 未承認一覧リスト
		Map<String, List<ManagementRequestListDtoInterface>> map = new HashMap<String, List<ManagementRequestListDtoInterface>>();
		// 所属部署のリーダーで有る場合
		if (humanReference.isLeader(personalId, targetDate)) {
			// 所属中の社員一覧
			List<HumanDtoInterface> menberList = humanReference.getSectionMembers(approverDto.getSectionCode(),
					targetDate);
			for (HumanDtoInterface menberDto : menberList) {
				if (menberDto.getPersonalId().equals(approverDto.getPersonalId())) {
					continue;
				}
				// 対象社員個人ID
				String memberPersonalId = menberDto.getPersonalId();
				// ルート適用を取得。
				RouteApplicationDtoInterface routeApplicationDto = workflowIntegrate.findForPerson(memberPersonalId,
						targetDate, workflowType);
				// ルート適用が無い場合。
				if (routeApplicationDto == null) {
					boolean addFlag = false;
					if (functionCode.length() == 0) {
						addFlag = true;
					}
					List<ManagementRequestListDtoInterface> list = new ArrayList<ManagementRequestListDtoInterface>();
					if (addFlag || TimeConst.CODE_FUNCTION_WORK_MANGE.equals(functionCode)) {
						list = new ArrayList<ManagementRequestListDtoInterface>();
						List<AttendanceDtoInterface> attendanceList = attendanceReference.getListForNotApproved(
								memberPersonalId, 1);
						for (AttendanceDtoInterface attendanceDto : attendanceList) {
							ManagementRequestListDtoInterface listDto = convert(attendanceDto, menberDto);
							if (listDto != null) {
								list.add(listDto);
							}
						}
						map.put(TimeConst.CODE_FUNCTION_WORK_MANGE, list);
					}
					if (addFlag || TimeConst.CODE_FUNCTION_OVER_WORK.equals(functionCode)) {
						list = new ArrayList<ManagementRequestListDtoInterface>();
						List<OvertimeRequestDtoInterface> overtimeList = overtimeRequest.getListForNotApproved(
								memberPersonalId, 1);
						for (OvertimeRequestDtoInterface overtimeDto : overtimeList) {
							ManagementRequestListDtoInterface listDto = convert(overtimeDto, menberDto);
							if (listDto != null) {
								list.add(listDto);
							}
						}
						map.put(TimeConst.CODE_FUNCTION_OVER_WORK, list);
					}
					if (addFlag || TimeConst.CODE_FUNCTION_VACATION.equals(functionCode)) {
						list = new ArrayList<ManagementRequestListDtoInterface>();
						List<HolidayRequestDtoInterface> holidayList = holidayRequest.getListForNotApproved(
								memberPersonalId, 1);
						for (HolidayRequestDtoInterface holidayDto : holidayList) {
							ManagementRequestListDtoInterface listDto = convert(holidayDto, menberDto);
							if (listDto != null) {
								list.add(listDto);
							}
						}
						map.put(TimeConst.CODE_FUNCTION_VACATION, list);
					}
					if (addFlag || TimeConst.CODE_FUNCTION_WORK_HOLIDAY.equals(functionCode)) {
						list = new ArrayList<ManagementRequestListDtoInterface>();
						List<WorkOnHolidayRequestDtoInterface> workOnHolidayList = workOnHolidayRequest
							.getListForNotApproved(memberPersonalId, 1);
						for (WorkOnHolidayRequestDtoInterface workOnHolidayDto : workOnHolidayList) {
							ManagementRequestListDtoInterface listDto = convert(workOnHolidayDto, menberDto);
							if (listDto != null) {
								list.add(listDto);
							}
						}
						map.put(TimeConst.CODE_FUNCTION_WORK_HOLIDAY, list);
					}
					if (addFlag || TimeConst.CODE_FUNCTION_COMPENSATORY_HOLIDAY.equals(functionCode)) {
						list = new ArrayList<ManagementRequestListDtoInterface>();
						List<SubHolidayRequestDtoInterface> subHolidayList = subHolidayRequest.getListForNotApproved(
								memberPersonalId, 1);
						for (SubHolidayRequestDtoInterface subHolidayDto : subHolidayList) {
							ManagementRequestListDtoInterface listDto = convert(subHolidayDto, menberDto);
							if (listDto != null) {
								list.add(listDto);
							}
						}
						map.put(TimeConst.CODE_FUNCTION_COMPENSATORY_HOLIDAY, list);
					}
					if (addFlag || TimeConst.CODE_FUNCTION_DIFFERENCE.equals(functionCode)) {
						list = new ArrayList<ManagementRequestListDtoInterface>();
						List<DifferenceRequestDtoInterface> differenceList = differenceRequest.getListForNotApproved(
								memberPersonalId, 1);
						for (DifferenceRequestDtoInterface differenceDto : differenceList) {
							ManagementRequestListDtoInterface listDto = convert(differenceDto, menberDto);
							if (listDto != null) {
								list.add(listDto);
							}
						}
						map.put(TimeConst.CODE_FUNCTION_DIFFERENCE, list);
					}
				}
			}
		}
		return map;
	}
	
	@Override
	public Map<String, List<ManagementRequestListDtoInterface>> getNonApprovedListForLowerSection(String personalId,
			Date targetDate, String functionCode, int workflowType) throws MospException {
		// 承認者人事マスタ
		HumanDtoInterface approverDto = humanReference.getHumanInfo(personalId, targetDate);
		// 未承認一覧リスト
		Map<String, List<ManagementRequestListDtoInterface>> map = new HashMap<String, List<ManagementRequestListDtoInterface>>();
		// 所属部署のリーダーで有る場合
		if (humanReference.isLeader(personalId, targetDate)) {
			// 下位所属マスタ一覧
			List<SectionDtoInterface> lowerSectionList = sectionReference.getLowerSectionList(approverDto
				.getSectionCode(), targetDate);
			for (SectionDtoInterface lowerSectionDto : lowerSectionList) {
				// 上位階層所属マスタ一覧
				List<SectionDtoInterface> higherSectionList = sectionReference.getHigherSectionList(lowerSectionDto
					.getSectionCode(), targetDate);
				// 承認階層
				int approvalCount = higherSectionList.size() + 1;
				int descCount = 0;
				for (SectionDtoInterface higherSectionDto : higherSectionList) {
					if (higherSectionDto.getSectionCode().equals(approverDto.getSectionCode())) {
						break;
					}
					descCount++;
				}
				// 承認段階
				int approvalStage = approvalCount - descCount;
				// 所属社員一覧
				List<HumanDtoInterface> lowerMenberList = humanReference.getSectionMembers(lowerSectionDto
					.getSectionCode(), targetDate);
				for (HumanDtoInterface menberDto : lowerMenberList) {
					// 対象社員個人ID
					String memberPersonalId = menberDto.getPersonalId();
					// ルート適用を取得。
					RouteApplicationDtoInterface routeApplicationDto = workflowIntegrate.findForPerson(
							memberPersonalId, targetDate, workflowType);
					// ルート適用が無い場合。
					if (routeApplicationDto == null) {
						boolean addFlag = false;
						if (functionCode.length() == 0) {
							addFlag = true;
						}
						List<ManagementRequestListDtoInterface> list = new ArrayList<ManagementRequestListDtoInterface>();
						if (addFlag || TimeConst.CODE_FUNCTION_WORK_MANGE.equals(functionCode)) {
							list = new ArrayList<ManagementRequestListDtoInterface>();
							List<AttendanceDtoInterface> attendanceList = attendanceReference.getListForNotApproved(
									memberPersonalId, approvalStage);
							for (AttendanceDtoInterface attendanceDto : attendanceList) {
								ManagementRequestListDtoInterface listDto = convert(attendanceDto, menberDto);
								if (listDto != null) {
									list.add(listDto);
								}
							}
							map.put(TimeConst.CODE_FUNCTION_WORK_MANGE, list);
						}
						if (addFlag || TimeConst.CODE_FUNCTION_OVER_WORK.equals(functionCode)) {
							list = new ArrayList<ManagementRequestListDtoInterface>();
							List<OvertimeRequestDtoInterface> overtimeList = overtimeRequest.getListForNotApproved(
									memberPersonalId, approvalStage);
							for (OvertimeRequestDtoInterface overtimeDto : overtimeList) {
								ManagementRequestListDtoInterface listDto = convert(overtimeDto, menberDto);
								if (listDto != null) {
									list.add(listDto);
								}
							}
							map.put(TimeConst.CODE_FUNCTION_OVER_WORK, list);
						}
						if (addFlag || TimeConst.CODE_FUNCTION_VACATION.equals(functionCode)) {
							list = new ArrayList<ManagementRequestListDtoInterface>();
							List<HolidayRequestDtoInterface> holidayList = holidayRequest.getListForNotApproved(
									memberPersonalId, approvalStage);
							for (HolidayRequestDtoInterface holidayDto : holidayList) {
								ManagementRequestListDtoInterface listDto = convert(holidayDto, menberDto);
								if (listDto != null) {
									list.add(listDto);
								}
							}
							map.put(TimeConst.CODE_FUNCTION_VACATION, list);
						}
						if (addFlag || TimeConst.CODE_FUNCTION_WORK_HOLIDAY.equals(functionCode)) {
							list = new ArrayList<ManagementRequestListDtoInterface>();
							List<WorkOnHolidayRequestDtoInterface> workOnHolidayList = workOnHolidayRequest
								.getListForNotApproved(memberPersonalId, approvalStage);
							for (WorkOnHolidayRequestDtoInterface workOnHolidayDto : workOnHolidayList) {
								ManagementRequestListDtoInterface listDto = convert(workOnHolidayDto, menberDto);
								if (listDto != null) {
									list.add(listDto);
								}
							}
							map.put(TimeConst.CODE_FUNCTION_WORK_HOLIDAY, list);
						}
						if (addFlag || TimeConst.CODE_FUNCTION_COMPENSATORY_HOLIDAY.equals(functionCode)) {
							list = new ArrayList<ManagementRequestListDtoInterface>();
							List<SubHolidayRequestDtoInterface> subHolidayList = subHolidayRequest
								.getListForNotApproved(memberPersonalId, approvalStage);
							for (SubHolidayRequestDtoInterface subHolidayDto : subHolidayList) {
								ManagementRequestListDtoInterface listDto = convert(subHolidayDto, menberDto);
								if (listDto != null) {
									list.add(listDto);
								}
							}
							map.put(TimeConst.CODE_FUNCTION_COMPENSATORY_HOLIDAY, list);
						}
						if (addFlag || TimeConst.CODE_FUNCTION_DIFFERENCE.equals(functionCode)) {
							list = new ArrayList<ManagementRequestListDtoInterface>();
							List<DifferenceRequestDtoInterface> differenceList = differenceRequest
								.getListForNotApproved(memberPersonalId, approvalStage);
							for (DifferenceRequestDtoInterface differenceDto : differenceList) {
								ManagementRequestListDtoInterface listDto = convert(differenceDto, menberDto);
								if (listDto != null) {
									list.add(listDto);
								}
							}
							map.put(TimeConst.CODE_FUNCTION_DIFFERENCE, list);
						}
					}
				}
			}
		}
		return map;
	}
	
	/**
	 * 各種申請DTOを申請情報確認一覧DTOへ変換する。
	 * @param baseDto 各種申請DTO
	 * @param humanDto 人事マスタDTO
	 * @return 変換後申請情報確認一覧DTO
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected ManagementRequestListDtoInterface convert(BaseDtoInterface baseDto, HumanDtoInterface humanDto)
			throws MospException {
		long tmdManagementRequestListId = 0;
		String personalId = "";
		WorkflowDtoInterface workflowDto = null;
		String requestType = "";
		Date requestDate = null;
		String requestInfo = "";
		checkExclusive(baseDto);
		if (mospParams.hasErrorMessage()) {
			return null;
		}
		// 勤怠情報
		if (baseDto instanceof AttendanceDtoInterface) {
			AttendanceDtoInterface attendanceDto = (AttendanceDtoInterface)baseDto;
			tmdManagementRequestListId = attendanceDto.getTmdAttendanceId();
			personalId = attendanceDto.getPersonalId();
			workflowDto = workflowIntegrate.getLatestWorkflowInfo(attendanceDto.getWorkflow());
			requestType = TimeConst.CODE_FUNCTION_WORK_MANGE;
			requestDate = attendanceDto.getWorkDate();
			requestInfo = getAttendanceInfo(attendanceDto);
		}
		// 残業申請
		if (baseDto instanceof OvertimeRequestDtoInterface) {
			OvertimeRequestDtoInterface overtimeDto = (OvertimeRequestDtoInterface)baseDto;
			tmdManagementRequestListId = overtimeDto.getTmdOvertimeRequestId();
			personalId = overtimeDto.getPersonalId();
			workflowDto = workflowIntegrate.getLatestWorkflowInfo(overtimeDto.getWorkflow());
			requestType = TimeConst.CODE_FUNCTION_OVER_WORK;
			requestDate = overtimeDto.getRequestDate();
			requestInfo = getOvertimeRequestInfo(overtimeDto);
		}
		// 休暇申請
		if (baseDto instanceof HolidayRequestDtoInterface) {
			HolidayRequestDtoInterface holidayDto = (HolidayRequestDtoInterface)baseDto;
			tmdManagementRequestListId = holidayDto.getTmdHolidayRequestId();
			personalId = holidayDto.getPersonalId();
			workflowDto = workflowIntegrate.getLatestWorkflowInfo(holidayDto.getWorkflow());
			requestType = TimeConst.CODE_FUNCTION_VACATION;
			requestDate = holidayDto.getRequestStartDate();
			requestInfo = getHolidayRequestInfo(holidayDto);
		}
		// 休日出勤
		if (baseDto instanceof WorkOnHolidayRequestDtoInterface) {
			WorkOnHolidayRequestDtoInterface workOnHolidayDto = (WorkOnHolidayRequestDtoInterface)baseDto;
			tmdManagementRequestListId = workOnHolidayDto.getTmdWorkOnHolidayRequestId();
			personalId = workOnHolidayDto.getPersonalId();
			workflowDto = workflowIntegrate.getLatestWorkflowInfo(workOnHolidayDto.getWorkflow());
			requestType = TimeConst.CODE_FUNCTION_WORK_HOLIDAY;
			requestDate = workOnHolidayDto.getRequestDate();
			requestInfo = workOnHolidayRequest.getWorkOnHolidayInfo(workOnHolidayDto);
		}
		// 代休申請
		if (baseDto instanceof SubHolidayRequestDtoInterface) {
			SubHolidayRequestDtoInterface subHolidayDto = (SubHolidayRequestDtoInterface)baseDto;
			tmdManagementRequestListId = subHolidayDto.getTmdSubHolidayRequestId();
			personalId = subHolidayDto.getPersonalId();
			workflowDto = workflowIntegrate.getLatestWorkflowInfo(subHolidayDto.getWorkflow());
			requestType = TimeConst.CODE_FUNCTION_COMPENSATORY_HOLIDAY;
			requestDate = subHolidayDto.getRequestDate();
			requestInfo = getSubHolidayRequestInfo(subHolidayDto);
		}
		// 時差出勤
		if (baseDto instanceof DifferenceRequestDtoInterface) {
			DifferenceRequestDtoInterface differenceDto = (DifferenceRequestDtoInterface)baseDto;
			tmdManagementRequestListId = differenceDto.getTmdDifferenceRequestId();
			personalId = differenceDto.getPersonalId();
			workflowDto = workflowIntegrate.getLatestWorkflowInfo(differenceDto.getWorkflow());
			requestType = TimeConst.CODE_FUNCTION_DIFFERENCE;
			requestDate = differenceDto.getRequestDate();
			requestInfo = differenceRequest.getDifferenceRequestInfo(differenceDto);
		}
		if (humanDto == null) {
			humanDto = humanReference.getHumanInfo(personalId, requestDate);
		}
		checkExclusive(workflowDto);
		checkExclusive(humanDto);
		if (mospParams.hasErrorMessage()) {
			return null;
		}
		ManagementRequestListDtoInterface dto = new ManagementRequestListDto();
		// 申請者情報
		dto.setEmployeeCode(humanDto.getEmployeeCode());
		dto.setLastName(humanDto.getLastName());
		dto.setFirstName(humanDto.getFirstName());
		dto.setSectionCode(humanDto.getSectionCode());
		// 申請内容
		dto.setTmdManagementRequestListId(tmdManagementRequestListId);
		dto.setRequestType(requestType);
		dto.setWorkflow(workflowDto.getWorkflow());
		dto.setRequestDate(requestDate);
		dto.setRequestInfo(requestInfo);
		dto.setStage(workflowDto.getWorkflowStage());
		dto.setState(workflowDto.getWorkflowStatus());
		return dto;
	}
	
	@Override
	public long getRequestWorkflow(BaseDtoInterface dto) {
		// 勤怠データ
		if (dto instanceof AttendanceDtoInterface) {
			return ((AttendanceDtoInterface)dto).getWorkflow();
		}
		// 残業申請
		if (dto instanceof OvertimeRequestDtoInterface) {
			return ((OvertimeRequestDtoInterface)dto).getWorkflow();
		}
		// 休暇申請
		if (dto instanceof HolidayRequestDtoInterface) {
			return ((HolidayRequestDtoInterface)dto).getWorkflow();
		}
		// 休日出勤
		if (dto instanceof WorkOnHolidayRequestDtoInterface) {
			return ((WorkOnHolidayRequestDtoInterface)dto).getWorkflow();
		}
		// 代休申請
		if (dto instanceof SubHolidayRequestDtoInterface) {
			return ((SubHolidayRequestDtoInterface)dto).getWorkflow();
		}
		// 時差出勤
		if (dto instanceof DifferenceRequestDtoInterface) {
			return ((DifferenceRequestDtoInterface)dto).getWorkflow();
		}
		return 0;
	}
	
	@Override
	public String getRequestPersonalId(BaseDtoInterface dto) {
		if (dto != null) {
			// 勤怠データ
			if (dto instanceof AttendanceDtoInterface) {
				return ((AttendanceDtoInterface)dto).getPersonalId();
			}
			// 残業申請
			if (dto instanceof OvertimeRequestDtoInterface) {
				return ((OvertimeRequestDtoInterface)dto).getPersonalId();
			}
			// 休暇申請
			if (dto instanceof HolidayRequestDtoInterface) {
				return ((HolidayRequestDtoInterface)dto).getPersonalId();
			}
			// 休日出勤
			if (dto instanceof WorkOnHolidayRequestDtoInterface) {
				return ((WorkOnHolidayRequestDtoInterface)dto).getPersonalId();
			}
			// 代休申請
			if (dto instanceof SubHolidayRequestDtoInterface) {
				return ((SubHolidayRequestDtoInterface)dto).getPersonalId();
			}
			// 時差出勤
			if (dto instanceof DifferenceRequestDtoInterface) {
				return ((DifferenceRequestDtoInterface)dto).getPersonalId();
			}
		}
		return "";
	}
	
	@Override
	public BaseDtoInterface getRequestDtoForWorkflow(long workflow, boolean isApproval) throws MospException {
		WorkflowDtoInterface dto = workflowIntegrate.getLatestWorkflowInfo(workflow);
		if (dto != null) {
			// 下書きを除く
			if (isApproval && PlatformConst.CODE_STATUS_DRAFT.equals(dto.getWorkflowStatus())) {
				return null;
			}
			// 勤怠データ
			if (TimeConst.CODE_FUNCTION_WORK_MANGE.equals(dto.getFunctionCode())) {
				return attendanceReference.findForWorkflow(workflow);
			}
			// 残業申請
			if (TimeConst.CODE_FUNCTION_OVER_WORK.equals(dto.getFunctionCode())) {
				return overtimeRequest.findForWorkflow(workflow);
			}
			// 休暇申請
			if (TimeConst.CODE_FUNCTION_VACATION.equals(dto.getFunctionCode())) {
				return holidayRequest.findForWorkflow(workflow);
			}
			// 休日出勤
			if (TimeConst.CODE_FUNCTION_WORK_HOLIDAY.equals(dto.getFunctionCode())) {
				return workOnHolidayRequest.findForWorkflow(workflow);
			}
			// 代休申請
			if (TimeConst.CODE_FUNCTION_COMPENSATORY_HOLIDAY.equals(dto.getFunctionCode())) {
				return subHolidayRequest.findForWorkflow(workflow);
			}
			// 時差出勤
			if (TimeConst.CODE_FUNCTION_DIFFERENCE.equals(dto.getFunctionCode())) {
				return differenceRequest.findForWorkflow(workflow);
			}
		}
		return null;
	}
	
	@Override
	public long getRequestRecordId(BaseDtoInterface dto) {
		if (dto != null) {
			// 勤怠データ
			if (dto instanceof AttendanceDtoInterface) {
				return ((AttendanceDtoInterface)dto).getTmdAttendanceId();
			}
			// 残業申請
			if (dto instanceof OvertimeRequestDtoInterface) {
				return ((OvertimeRequestDtoInterface)dto).getTmdOvertimeRequestId();
			}
			// 休暇申請
			if (dto instanceof HolidayRequestDtoInterface) {
				return ((HolidayRequestDtoInterface)dto).getTmdHolidayRequestId();
			}
			// 休日出勤
			if (dto instanceof WorkOnHolidayRequestDtoInterface) {
				return ((WorkOnHolidayRequestDtoInterface)dto).getTmdWorkOnHolidayRequestId();
			}
			// 代休申請
			if (dto instanceof SubHolidayRequestDtoInterface) {
				return ((SubHolidayRequestDtoInterface)dto).getTmdSubHolidayRequestId();
			}
			// 時差出勤
			if (dto instanceof DifferenceRequestDtoInterface) {
				return ((DifferenceRequestDtoInterface)dto).getTmdDifferenceRequestId();
			}
		}
		return 0;
	}
	
	@Override
	public Date getRequestDate(BaseDtoInterface dto) {
		if (dto != null) {
			// 勤怠データ
			if (dto instanceof AttendanceDtoInterface) {
				return ((AttendanceDtoInterface)dto).getWorkDate();
			}
			// 残業申請
			if (dto instanceof OvertimeRequestDtoInterface) {
				return ((OvertimeRequestDtoInterface)dto).getRequestDate();
			}
			// 休暇申請
			if (dto instanceof HolidayRequestDtoInterface) {
				return ((HolidayRequestDtoInterface)dto).getRequestStartDate();
			}
			// 休日出勤
			if (dto instanceof WorkOnHolidayRequestDtoInterface) {
				return ((WorkOnHolidayRequestDtoInterface)dto).getRequestDate();
			}
			// 代休申請
			if (dto instanceof SubHolidayRequestDtoInterface) {
				return ((SubHolidayRequestDtoInterface)dto).getRequestDate();
			}
			// 時差出勤
			if (dto instanceof DifferenceRequestDtoInterface) {
				return ((DifferenceRequestDtoInterface)dto).getRequestDate();
			}
		}
		return null;
	}
	
	@Override
	public WorkflowDtoInterface getWorkflowForFunctionRecordId(String functionCode, long id) throws MospException {
		long workflow = 0;
		if (functionCode != null && functionCode.length() != 0) {
			// 勤怠データ
			if (TimeConst.CODE_FUNCTION_WORK_MANGE.equals(functionCode)) {
				AttendanceDtoInterface dto = attendanceReference.findForId(id);
				if (mospParams.hasErrorMessage()) {
					return null;
				}
				workflow = dto.getWorkflow();
			}
			// 残業申請
			if (TimeConst.CODE_FUNCTION_OVER_WORK.equals(functionCode)) {
				OvertimeRequestDtoInterface dto = overtimeRequest.findForId(id);
				if (mospParams.hasErrorMessage()) {
					return null;
				}
				workflow = dto.getWorkflow();
			}
			// 休暇申請
			if (TimeConst.CODE_FUNCTION_VACATION.equals(functionCode)) {
				HolidayRequestDtoInterface dto = holidayRequest.findForId(id);
				if (mospParams.hasErrorMessage()) {
					return null;
				}
				workflow = dto.getWorkflow();
			}
			// 休日出勤
			if (TimeConst.CODE_FUNCTION_WORK_HOLIDAY.equals(functionCode)) {
				WorkOnHolidayRequestDtoInterface dto = workOnHolidayRequest.findForId(id);
				if (mospParams.hasErrorMessage()) {
					return null;
				}
				workflow = dto.getWorkflow();
			}
			// 代休申請
			if (TimeConst.CODE_FUNCTION_COMPENSATORY_HOLIDAY.equals(functionCode)) {
				SubHolidayRequestDtoInterface dto = subHolidayRequest.findForId(id);
				if (mospParams.hasErrorMessage()) {
					return null;
				}
				workflow = dto.getWorkflow();
			}
			// 時差出勤
			if (TimeConst.CODE_FUNCTION_DIFFERENCE.equals(functionCode)) {
				DifferenceRequestDtoInterface dto = differenceRequest.findForId(id);
				if (mospParams.hasErrorMessage()) {
					return null;
				}
				workflow = dto.getWorkflow();
			}
		}
		return workflowIntegrate.getLatestWorkflowInfo(workflow);
	}
	
	@Override
	public void setWorkflowInfo(RequestListDtoInterface dto, WorkflowDtoInterface workflowDto) throws MospException {
		if (dto != null && workflowDto != null) {
			WorkflowCommentDtoInterface commentDto = workflowCommentReference.getLatestWorkflowCommentInfo(workflowDto
				.getWorkflow());
			if (commentDto != null) {
				// 承認段階
				dto.setStage(workflowDto.getWorkflowStage());
				// 承認状況
				dto.setState(workflowDto.getWorkflowStatus());
				String approverName = humanReference.getHumanName(commentDto.getPersonalId(), commentDto
					.getWorkflowDate());
				// THINK 次の承認者の個人IDを取得。
				
				// 下書き、申請、完了の場合
				if (PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
						|| PlatformConst.CODE_STATUS_APPLY.equals(workflowDto.getWorkflowStatus())
						|| PlatformConst.CODE_STATUS_COMPLETE.equals(workflowDto.getWorkflowStatus())) {
					approverName = "";
				}
				// 承認者名
				dto.setApproverName(approverName);
			}
		}
	}
	
	@Override
	public String[][][] getArrayForApproverSetting(String personalId, Date targetDate, int workflowType)
			throws MospException {
		// 承認ルート人事マスタ一覧
		List<List<String[]>> approvalRouteList = workflowIntegrate.getRouteApproverList(personalId, targetDate,
				workflowType);
		// エラー確認
		if (mospParams.hasErrorMessage()) {
			return new String[0][][];
		}
		// 承認階層分の申請用承認者情報配列準備
		String[][][] aryApproverSetting = new String[approvalRouteList.size()][][];
		// 承認階層インデックス準備
		int approvalLevel = 0;
		// 承認階層毎に申請用承認者情報を作成
		for (List<String[]> approverList : approvalRouteList) {
			// i次承認者配列作成用リスト準備
			List<String[]> list = new ArrayList<String[]>();
			// 承認者リストから申請用承認者情報を作成
			for (String[] approver : approverList) {
				// 申請用承認者情報配列作成用リストに追加
				list.add(approver);
			}
			aryApproverSetting[approvalLevel++] = list.toArray(new String[list.size()][]);
		}
		return aryApproverSetting;
	}
	
	/**
	 * 分取得。
	 * @param time 時間
	 * @return 分
	 */
	protected String getMinute(int time) {
		String min = Integer.toString(time % 60);
		if (min.length() == 1) {
			return 0 + min;
		}
		return min;
	}
	
}
