/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.bean;

import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.MospException;
import jp.mosp.time.dto.settings.WorkTypeDtoInterface;

/**
 * 勤務形態マスタ参照インターフェース。
 */
public interface WorkTypeReferenceBeanInterface {
	
	/**
	 * 勤務形態マスタ取得。
	 * <p>
	 * 勤務形態コードと対象日から勤務形態マスタを取得。
	 * </p>
	 * @param workTypeCode 勤務形態コード
	 * @param targetDate 対象年月日
	 * @return 勤務形態マスタ
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	WorkTypeDtoInterface getWorkTypeInfo(String workTypeCode, Date targetDate) throws MospException;
	
	/**
	 * 履歴一覧を取得する。<br>
	 * 勤務形態コードから勤務形態マスタリストを取得する。<br>
	 * @param workTypeCode 勤務形態コード
	 * @return 勤務形態マスタリスト
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	List<WorkTypeDtoInterface> getWorkTypeHistory(String workTypeCode) throws MospException;
	
	/**
	 * 勤務形態略称を取得する。<br><br>
	 * 対象となる勤務形態情報が存在しない場合は、勤務形態コードを返す。<br>
	 * @param workTypeCode 勤務形態コード
	 * @param targetDate 対象年月日
	 * @return 勤務形態略称
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	String getWorkTypeAbbr(String workTypeCode, Date targetDate) throws MospException;
	
	/**
	 * 勤務形態略称を取得する。<br><br>
	 * 対象となる勤務形態情報が存在しない場合は、勤務形態コードを返す。<br>
	 * 表示内容は、勤務形態略称【出勤時刻～退勤時刻】。<br>
	 * @param workTypeCode 勤務形態コード
	 * @param targetDate 対象年月日
	 * @return 勤務形態略称
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	String getWorkTypeAbbrStartTimeEndTime(String workTypeCode, Date targetDate) throws MospException;
	
	/**
	 * プルダウン用配列取得。
	 * <p>
	 * 対象年月日からプルダウン用配列を取得。
	 * </p>
	 * @param targetDate 対象年月日
	 * @return プルダウン用配列
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	String[][] getSelectArray(Date targetDate) throws MospException;
	
	/**
	 * 略称プルダウン用配列取得。
	 * <p>
	 * 対象年月日から略称プルダウン用配列を取得。
	 * </p>
	 * @param targetDate 対象年月日
	 * @return プルダウン用配列
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	String[][] getSelectAbbrArray(Date targetDate) throws MospException;
	
	/**
	 * プルダウン用配列を取得する。<br>
	 * 対象年月日からプルダウン用配列を取得する。<br>
	 * 表示内容は、コード + 勤務形態略称。<br>
	 * @param targetDate 対象年月日
	 * @return プルダウン用配列
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	String[][] getCodedSelectArray(Date targetDate) throws MospException;
	
	/**
	 * プルダウン用配列を取得する。<br>
	 * 対象年月日からプルダウン用配列を取得する。<br>
	 * 表示内容は、勤務形態略称【出勤時刻～退勤時刻】。<br>
	 * @param targetDate 対象年月日
	 * @return プルダウン用配列
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	String[][] getStartTimeEndTimeSelectArray(Date targetDate) throws MospException;
	
	/**
	 * 勤務形態マスタからレコードを取得する。<br>
	 * 勤務形態コード、有効日で合致するレコードが無い場合、nullを返す。<br>
	 * @param workTypeCode 雇用契約コード
	 * @param activateDate 有効日
	 * @return 勤務形態マスタDTO
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	WorkTypeDtoInterface findForKey(String workTypeCode, Date activateDate) throws MospException;
	
	/**
	 * 勤務形態マスタからレコードを取得する。<br>
	 * 勤務形態コード、有効日で合致するレコードが無い場合、nullを返す。<br>
	 * @param workTypeCode 雇用契約コード
	 * @param activateDate 有効日
	 * @return 勤務形態マスタDTO
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	WorkTypeDtoInterface findForInfo(String workTypeCode, Date activateDate) throws MospException;
}
