/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * 
 */
package jp.mosp.time.bean;

import java.util.Date;
import java.util.List;
import java.util.Map;

import jp.mosp.framework.base.BaseDtoInterface;
import jp.mosp.framework.base.MospException;
import jp.mosp.platform.dto.workflow.WorkflowDtoInterface;
import jp.mosp.time.dto.settings.ManagementRequestListDtoInterface;
import jp.mosp.time.dto.settings.RequestListDtoInterface;

/**
 * 承認情報参照インターフェース。
 */
public interface ApprovalInfoReferenceBeanInterface {
	
	/**
	 * 未承認申請リスト取得。
	 * @param personalId 承認者個人ID
	 * @param functionCode 機能コード
	 * @return 未承認申請リスト
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	Map<String, List<ManagementRequestListDtoInterface>> getNonApprovedMap(String personalId, String functionCode)
			throws MospException;
	
	/**
	 * 未承認申請リスト取得(勤怠)。
	 * @param personalId 承認者個人ID
	 * @return 未承認申請リスト(勤怠)
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	List<ManagementRequestListDtoInterface> getNonApprovedAttendanceList(String personalId) throws MospException;
	
	/**
	 * 未承認申請リスト取得(残業)。
	 * @param personalId 承認者個人ID
	 * @return 未承認申請リスト(残業)
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	List<ManagementRequestListDtoInterface> getNonApprovedOvertimeList(String personalId) throws MospException;
	
	/**
	 * 未承認申請リスト取得(休暇)。
	 * @param personalId 承認者個人ID
	 * @return 未承認申請リスト(休暇)
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	List<ManagementRequestListDtoInterface> getNonApprovedHolidayList(String personalId) throws MospException;
	
	/**
	 * 未承認申請リスト取得(休日出勤)。
	 * @param personalId 承認者個人ID
	 * @return 未承認申請リスト(休日出勤)
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	List<ManagementRequestListDtoInterface> getNonApprovedWorkOnHolidayList(String personalId) throws MospException;
	
	/**
	 * 未承認申請リスト取得(代休)。
	 * @param personalId 承認者個人ID
	 * @return 未承認申請リスト(代休)
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	List<ManagementRequestListDtoInterface> getNonApprovedSubHolidayList(String personalId) throws MospException;
	
	/**
	 * 未承認申請リスト取得(時差出勤)。
	 * @param personalId 承認者個人ID
	 * @return 未承認申請リスト(時差出勤)
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	List<ManagementRequestListDtoInterface> getNonApprovedDifferenceList(String personalId) throws MospException;
	
	/**
	 * 未承認申請リストを取得する。
	 * @param personalId 承認者個人ID
	 * @param targetDate 対象年月日
	 * @param functionCode 機能コード
	 * @param workflowType フロー区分
	 * @return 未承認申請リスト
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	Map<String, List<ManagementRequestListDtoInterface>> getNonApprovedListForBelongSection(String personalId,
			Date targetDate, String functionCode, int workflowType) throws MospException;
	
	/**
	 * 未承認申請リストを取得する。
	 * @param personalId 承認者個人ID
	 * @param targetDate 対象年月日
	 * @param functionCode 機能コード
	 * @param workflowType フロー区分
	 * @return 未承認申請リスト
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	Map<String, List<ManagementRequestListDtoInterface>> getNonApprovedListForLowerSection(String personalId,
			Date targetDate, String functionCode, int workflowType) throws MospException;
	
	/**
	 * 申請者個人ID取得。
	 * @param dto 対象DTO
	 * @return 申請者個人ID
	 */
	String getRequestPersonalId(BaseDtoInterface dto);
	
	/**
	 * ワークフロー番号から申請情報を取得する。
	 * @param workflow ワークフロー番号
	 * @param isApproval 承認利用フラグ、申請で利用する場合はfalse
	 * @return 各種申請DTO
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	BaseDtoInterface getRequestDtoForWorkflow(long workflow, boolean isApproval) throws MospException;
	
	/**
	 * 申請日取得。
	 * @param dto 対象DTO
	 * @return 申請日
	 */
	Date getRequestDate(BaseDtoInterface dto);
	
	/**
	 * @param dto 申請一覧用DTO
	 * @param workflowDto ワークフローDTO
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	void setWorkflowInfo(RequestListDtoInterface dto, WorkflowDtoInterface workflowDto) throws MospException;
	
	/**
	 * @param functionCode 機能コード
	 * @param id 各種申請レコード識別ID
	 * @return ワークフローDTO
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	WorkflowDtoInterface getWorkflowForFunctionRecordId(String functionCode, long id) throws MospException;
	
	/**
	 * レコード識別ID取得。
	 * @param dto 対象DTO
	 * @return レコード識別ID
	 */
	long getRequestRecordId(BaseDtoInterface dto);
	
	/**
	 * 申請情報からワークフロー番号を取得。
	 * @param dto 対象DTO
	 * @return ワークフロー番号
	 */
	long getRequestWorkflow(BaseDtoInterface dto);
	
	/**
	 * 申請用承認者情報配列を取得する。<br>
	 * 配列の内容は以下の通り。<br>
	 * <ul><li>
	 * 1次元目：承認階層
	 * </li><li>
	 * 2次元目：承認階層における申請用承認者情報インデックス
	 * </li><li>
	 * 3次元目：各申請用承認者情報
	 * </li></ul>
	 * @param personalId 申請者個人ID
	 * @param targetDate 申請年月日
	 * @param workflowType フロー区分
	 * @return 申請用承認者情報配列
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	String[][][] getArrayForApproverSetting(String personalId, Date targetDate, int workflowType) throws MospException;
	
}
