/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.platform.human.action;

import java.util.List;

import jp.mosp.framework.base.BaseVo;
import jp.mosp.framework.base.MospException;
import jp.mosp.framework.utils.MospUtility;
import jp.mosp.platform.base.PlatformAction;
import jp.mosp.platform.bean.system.EmploymentContractReferenceBeanInterface;
import jp.mosp.platform.bean.system.PositionReferenceBeanInterface;
import jp.mosp.platform.bean.system.SectionReferenceBeanInterface;
import jp.mosp.platform.bean.system.WorkPlaceReferenceBeanInterface;
import jp.mosp.platform.dto.human.HumanDtoInterface;
import jp.mosp.platform.human.base.PlatformHumanAction;
import jp.mosp.platform.human.base.PlatformHumanVo;
import jp.mosp.platform.human.constant.PlatformHumanConst;
import jp.mosp.platform.human.vo.BasicListVo;

/**
 * 社員一覧画面で選択した社員の個人基本情報の履歴の一覧を表示する。<br>
 * <br>
 * 以下のコマンドを扱う。<br>
 * <ul><li>
 * {@link #CMD_SELECT}
 * </li><li>
 * {@link #CMD_RE_SEARCH}
 * </ul></li>
 */
public class BasicListAction extends PlatformHumanAction {
	
	/**
	 * 選択表示コマンド。<br>
	 * <br>
	 * 社員一覧画面で選択された社員の個人IDを基に個人基本情報の履歴を表示する。<br>
	 * 社員コード入力欄に入力された社員コードを基に検索を行い、
	 * 条件に当てはまる社員の基本情報履歴一覧を表示する。<br>
	 * 社員コード順にページを送るボタンがクリックされた場合には
	 * 遷移元の社員一覧リスト検索結果を参照して前後それぞれページ移動を行う。<br>
	 * 入力した社員コードが存在しない、または入力されていない状態で
	 * 「検索ボタン」がクリックされた場合はエラーメッセージにて通知。<br>
	 * 現在表示されている社員の社員コードの前後にデータが存在しない時に
	 * コード順送りボタンをクリックした場合も同様にエラーメッセージにて通知。<br>
	 */
	public static final String	CMD_SELECT		= "PF1146";
	
	/**
	 * 再検索コマンド。<br>
	 * <br>
	 * パンくずリスト等を用いてこれよりも奥の階層の画面から改めて遷移した場合、
	 * 各種情報の登録状況が更新された状態で再表示を行う。<br>
	 */
	public static final String	CMD_RE_SEARCH	= "PF1147";
	

	/**
	 * {@link PlatformAction#PlatformAction()}を実行する。<br>
	 */
	public BasicListAction() {
		super();
		// パンくずリスト用コマンド設定
		topicPathCommand = CMD_RE_SEARCH;
	}
	
	@Override
	public void action() throws MospException {
		// コマンド毎の処理
		if (mospParams.getCommand().equals(CMD_SELECT)) {
			// 選択表示
			prepareVo(true, false);
			select();
		} else if (mospParams.getCommand().equals(CMD_RE_SEARCH)) {
			// 再表示
			prepareVo(true, false);
			reSearch();
		} else {
			throwInvalidCommandException();
		}
		
	}
	
	@Override
	protected BaseVo getSpecificVo() {
		return new BasicListVo();
	}
	
	/**
	 * 検索処理を行う。
	 * @throws MospException VO、或いは社員情報の取得に失敗した場合
	 */
	protected void select() throws MospException {
		// 人事管理共通情報利用設定
		setPlatformHumanSettings(CMD_SELECT, PlatformHumanConst.MODE_HUMAN_NO_ACTIVATE_DATE);
		// 共通情報設定
		setTransferredParams();
		// 項目初期化
		setDefaultValues();
		// 人事履歴情報リストを取得しVOに設定
		getBasicList();
	}
	
	/**
	 * VOに保持された情報を基に、再検索を行う。
	 * @throws MospException 例外処理が発生した場合
	 */
	protected void reSearch() throws MospException {
		// VO取得
		BasicListVo vo = (BasicListVo)mospParams.getVo();
		// 人事管理共通情報設定
		setHumanCommonInfo(vo.getEmployeeCode(), vo.getActivateDate());
		// 項目初期化
		setDefaultValues();
		// 人事履歴情報リストを取得しVOに設定
		getBasicList();
	}
	
	/**
	 * 初期値を設定する。<br>
	 */
	protected void setDefaultValues() {
		// VO取得
		BasicListVo vo = (BasicListVo)mospParams.getVo();
		// 初期値設定
		vo.setAryActiveteDate(new String[0]);
		vo.setAryEmployeeKana(new String[0]);
		vo.setAryEmployeeName(new String[0]);
		vo.setAryWorkPlace(new String[0]);
		vo.setAryEmployment(new String[0]);
		vo.setAryPosition(new String[0]);
		vo.setArySection(new String[0]);
	}
	
	/**
	 * 人事履歴情報リストを取得し、VOに設定する。<br>
	 * 対象社員コードは{@link PlatformHumanVo#getEmployeeCode()}を、
	 * 対象年月日は{@link PlatformHumanVo#getActivateDate()}を用いる。<br>
	 * @throws MospException 社員情報の取得に失敗した場合
	 */
	protected void getBasicList() throws MospException {
		// VO取得
		BasicListVo vo = (BasicListVo)mospParams.getVo();
		// 社員マスタ履歴情報取得
		List<HumanDtoInterface> list = reference().human().getHistory(vo.getEmployeeCode(),
				getDate(vo.getActivateDate()));
		// 設定配列準備
		String[] aryActiveteDate = new String[list.size()];
		String[] aryEmployeeKana = new String[list.size()];
		String[] aryEmployeeName = new String[list.size()];
		String[] aryWorkPlace = new String[list.size()];
		String[] aryEmployment = new String[list.size()];
		String[] aryPosition = new String[list.size()];
		String[] arySection = new String[list.size()];
		// 参照クラス準備
		PositionReferenceBeanInterface position = reference().position();
		SectionReferenceBeanInterface section = reference().section();
		EmploymentContractReferenceBeanInterface contract = reference().employmentContract();
		WorkPlaceReferenceBeanInterface workPlace = reference().workPlace();
		// フィールド設定
		for (int i = 0; i < list.size(); i++) {
			// 人事情報取得(フィールド設定順序は有効日の昇順)
			HumanDtoInterface dto = list.get(list.size() - 1 - i);
			// 有効日設定
			aryActiveteDate[i] = getStringDate(dto.getActivateDate());
			// 氏名設定
			aryEmployeeKana[i] = MospUtility.getHumansName(dto.getFirstKana(), dto.getLastKana());
			aryEmployeeName[i] = MospUtility.getHumansName(dto.getFirstName(), dto.getLastName());
			// 勤務地
			aryWorkPlace[i] = workPlace.getWorkPlaceName(dto.getWorkPlaceCode(), dto.getActivateDate());
			// 雇用契約
			aryEmployment[i] = contract.getContractName(dto.getEmploymentContractCode(), dto.getActivateDate());
			// 職位
			aryPosition[i] = position.getPositionName(dto.getPositionCode(), dto.getActivateDate());
			// 所属
			arySection[i] = section.getSectionName(dto.getSectionCode(), dto.getActivateDate());
		}
		// VOに設定
		vo.setAryActiveteDate(aryActiveteDate);
		vo.setAryEmployeeKana(aryEmployeeKana);
		vo.setAryEmployeeName(aryEmployeeName);
		vo.setAryWorkPlace(aryWorkPlace);
		vo.setAryEmployment(aryEmployment);
		vo.setAryPosition(aryPosition);
		vo.setArySection(arySection);
	}
	
}
