/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.platform.bean.workflow;

import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.MospException;
import jp.mosp.platform.dto.workflow.RouteApplicationDtoInterface;

/**
 * ルート適用マスタ参照インターフェース。
 */
public interface RouteApplicationReferenceBeanInterface {
	
	/**
	 * ルート適用マスタ取得。
	 * <p>
	 * ルート適用コードと対象日からルート適用マスタを取得。
	 * </p>
	 * @param routeApplicationCode ルート適用コード
	 * @param targetDate 対象年月日
	 * @return ルート適用マスタ
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	RouteApplicationDtoInterface getRouteApplicationInfo(String routeApplicationCode, Date targetDate)
			throws MospException;
	
	/**
	 * 履歴一覧取得。
	 * <p>
	 * ルート適用コードからルート適用マスタを取得。
	 * </p>
	 * @param routeApplicationCode ルート適用コード
	 * @return ルート適用マスタ
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	List<RouteApplicationDtoInterface> getRouteApplicationHistory(String routeApplicationCode) throws MospException;
	
	/**
	 * 個人ID、対象日に有効なルート適用マスタ一覧取得。
	 * @param personalId 個人ID
	 * @param targetDate 対象年月日
	 * @param workflowType フロー区分
	 * @return ルート適用マスタリスト
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	List<RouteApplicationDtoInterface> getListForPersonalId(String personalId, Date targetDate, int workflowType)
			throws MospException;
	
	/**
	 * ルート適用名称を取得する。<br><br>
	 * 対象となるルート適用マスタ情報が存在しない場合は、ルート適用コードを返す。<br>
	 * @param routeApplicationCode ルート適用コード
	 * @param targetDate 対象年月日
	 * @return ルート適用名称
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	String getRouteApplicationName(String routeApplicationCode, Date targetDate) throws MospException;
	
	/**
	 * プルダウン用配列を取得する。<br>
	 * 対象年月日からプルダウン用配列を取得する。<br>
	 * 表示内容は、コード + ルート適用名称。<br>
	 * @param targetDate 対象年月日
	 * @param needBlank 空白行要否(true：空白行要、false：空白行不要)
	 * @return プルダウン用配列
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	String[][] getCodedSelectArray(Date targetDate, boolean needBlank) throws MospException;
	
	/**
	 * ルート適用マスタからレコードを取得する。<br>
	 * ルート適用コード、有効日で合致するレコードが無い場合、nullを返す。<br>
	 * @param routeApplicationCode ルート適用コード
	 * @param activateDate 有効日
	 * @return ルート適用マスタDTO
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	RouteApplicationDtoInterface findForKey(String routeApplicationCode, Date activateDate) throws MospException;
	
	/**
	 * 個人ID及び対象日から、適用されている設定を取得する。<br>
	 * @param personalId 個人ID
	 * @param targetDate 対象日
	 * @param workflowType フロー区分
	 * @return ルート適用マスタDTO
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	RouteApplicationDtoInterface findForPerson(String personalId, Date targetDate, int workflowType)
			throws MospException;
	
}
