package monalipse.widgets;

import java.util.ArrayList;
import java.util.Collection;
import java.util.EventListener;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.jface.text.ITextOperationTarget;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.dnd.TextTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.ControlListener;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.MouseMoveListener;
import org.eclipse.swt.events.MouseTrackListener;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.events.TypedEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.ScrollBar;
import org.eclipse.swt.widgets.Shell;

public class ColoredText extends Canvas implements ITextOperationTarget, ISelectionProvider
{
	private List lines = new ArrayList();
	private VisualLines visualLines = new VisualLines();
	private GC gc;
	private Color selectionDark;
	private Color selectionLight;
	private Cursor handCursor;
	private Cursor ibeamCursor;
	private int width = Integer.MAX_VALUE;
	private int lineSkip = 10;
	private int[] insets = new int[]{10, 10, 10, 10};
	private TextPosition selectionStart = TextPosition.INVALID;
	private TextPosition selectionEnd= TextPosition.INVALID;
	private List textTrackListeners = new ArrayList();
	private ToolTipProvider toolTipProvider;
	private Shell toolTipShell;
	private boolean toolTipLocked;
	private ColoredText toolTipSource = this;
	private List selectionChangedListeners = new ArrayList();

	public ColoredText(Composite parent, int style)
	{
		super(parent, style | SWT.NO_BACKGROUND | SWT.SMOOTH);
		selectionDark = new Color(null, 0x00, 0x00, 0x00);
		selectionLight = new Color(null, 0x8f, 0x99, 0xa6);
		handCursor = new Cursor(parent.getDisplay(), SWT.CURSOR_HAND);
		ibeamCursor = new Cursor(parent.getDisplay(), SWT.CURSOR_IBEAM);
		gc = new GC(this);

		addPaintListener(new PaintListener()
			{
				public void paintControl(PaintEvent e)
				{
					ColoredText.this.paintControl(e);
				}
			});

		addControlListener(new ControlListener()
			{
				public void controlMoved(ControlEvent e)
				{
				}
	
				public void controlResized(ControlEvent e)
				{
					ColoredText.this.controlResized(e);
				}
			});
		
		addKeyListener(new PageMoveKeyListener());
		
		MouseDragListener dsl = new MouseDragListener();
		addMouseListener(dsl);
		addMouseMoveListener(dsl);
		addMouseTrackListener(dsl);
		
		addTextTrackListener(new TextTrackListener()
			{
				public void textEnter(TextEvent e)
				{
					if(e.fragment instanceof LinkTarget)
						setCursor(handCursor);
					else
						setCursor(ibeamCursor);
				}
	
				public void textExit(TextEvent e)
				{
					setCursor(null);

					if(!toolTipLocked)
					{
						setCapture(false);
						disposeHover();
					}
				}
	
				public void textHover(TextEvent e)
				{
					if(toolTipProvider != null && e.fragment instanceof ToolTipTarget)
						showHover(e.x, e.y, (ToolTipTarget)e.fragment, null);
				}
				
				public void textSelectionEnter(TextEvent e)
				{
				}
				
				public void textSelectionExit(TextEvent e)
				{
				}
				
				public void textSelectionHover(TextEvent e)
				{
					TextSelection sel = e.text.getTextSelection();
					if(toolTipProvider != null && !sel.isEmpty())
						showHover(e.x, e.y, null, e.text.getText(sel.from, sel.to));
				}
			});

		getVerticalBar().setMinimum(0);
		getVerticalBar().setMaximum(0);
		
		getVerticalBar().addSelectionListener(new SelectionListener()
			{
				public void widgetSelected(SelectionEvent e)
				{
					redraw();
				}
	
				public void widgetDefaultSelected(SelectionEvent e)
				{
				}
			});
	}
	
	private void showHover(int x, int y, ToolTipTarget fragment, String selection)
	{
		disposeHover();
			
		toolTipShell = new Shell(getShell(), SWT.NONE);

		Point pt = toDisplay(new Point(x, y));
		Rectangle scr = getShell().getDisplay().getClientArea();
		int cursorSize = 16;
		scr.x += cursorSize;
		scr.y += cursorSize;
		scr.width -= cursorSize * 2;
		scr.height -= cursorSize * 2;

		int maxWidth = Math.max(pt.x - scr.x, scr.x + scr.width - pt.x) - 2;

		Composite content = new Composite(toolTipShell, SWT.NONE);
		content.setLayout(new FillLayout(SWT.VERTICAL | SWT.HORIZONTAL));
		Point size;
		if(fragment != null)
			size = toolTipProvider.fillToolTip(content, toolTipSource, maxWidth, fragment);
		else
			size = toolTipProvider.fillToolTip(content, toolTipSource, maxWidth, selection);

		size.x += 2;
		size.y += 2;

		if(scr.x + scr.width < pt.x + size.x)
			pt.x -= size.x + cursorSize;
		else
			pt.x += cursorSize;

		if(scr.y + scr.height < pt.y + size.y)
		{
			if(scr.height / 2 < pt.y - scr.y)
			{
				size.y = Math.min(size.y, pt.y - scr.y);
				pt.y -= size.y + cursorSize;
			}
			else
			{
				size.y = scr.y + scr.height - pt.y;
				pt.y += cursorSize;
			}
		}
		else
		{
			pt.y += cursorSize;
		}

		content.setSize(size.x - 2, size.y - 2);
		content.setLocation(0, 0);
		toolTipShell.setLocation(pt);
		toolTipShell.setSize(size.x, size.y);
		
		toolTipShell.setVisible(true);
		setCapture(true);
		
		toolTipProvider.toolTipActivated(ColoredText.this);
	}
	
	private void disposeHover()
	{
		toolTipLocked = false;
		if(toolTipShell != null)
		{
			try
			{
				toolTipProvider.toolTipDeactivated(this);
				toolTipShell.setVisible(false);
				toolTipShell.dispose();
			}
			catch (RuntimeException e)
			{
			}
			toolTipShell = null;
		}
	}
	
	public void dispose()
	{
		disposeHover();
		if(gc != null)
			gc.dispose();
		gc = null;
		if(selectionDark != null)
			selectionDark.dispose();
		selectionDark = null;
		if(selectionLight != null)
			selectionLight.dispose();
		selectionLight = null;
		super.dispose();
	}
	
	public void lockToolTip()
	{
		if(!toolTipLocked && toolTipShell != null)
			toolTipLocked = true;
	}
	
	public Point computeSize(int wHint, int hHint, boolean changed)
	{
		int width = 0;
		for(int i = 0; i < getLineCount(); i++)
			width = Math.max(width, getLineAt(i).getPreferredLineWidth(gc));
		if(width == 0)
			return super.computeSize(wHint, hHint);
		else
			return new Point(width + insets[1] + insets[3] + getVerticalBar().getSize().x, lineSkip * getLineCount() + insets[0] + insets[2] + 1); 
	}

	private void controlResized(ControlEvent e)
	{
		wrapLines(getClientArea().width - insets[1] - insets[3]);
		
		ScrollBar vertical = getVerticalBar();
		
		int height = getClientArea().height;
		vertical.setIncrement(lineSkip);
		vertical.setPageIncrement(height * 9 / 10);

		updateScrollThumb();
	}

	private void paintControl(PaintEvent e)
	{
		GC gc = e.gc;
		
		int pos = getVerticalBar().getSelection();
		int originX = insets[1];
		int originY = insets[0] - pos;
		
		int startLine = Math.max(0, (e.y - originY) / lineSkip);
		int endLine = Math.min(visualLines.getLineCount(), (e.y + e.height - originY) / lineSkip + 1);
		
		int lineWidth = width + insets[1] + insets[3];
		
		gc.setBackground(getBackground());

		if(0 < originY)
			gc.fillRectangle(0, 0, lineWidth, originY);
		
		for(int i = startLine; i < endLine; i++)
		{
			int logicalLine = visualLines.getLogicalLineAt(i);
			Line line = getLineAt(logicalLine);
			if(selectionStart.row <= logicalLine && logicalLine <= selectionEnd.row)
			{
				int selStart = selectionStart.row < logicalLine ? 0 : selectionStart.column;
				int selEnd = logicalLine < selectionEnd.row ? line.getTextLength() : selectionEnd.column;
				line.paintSelected(gc, originX, originY + lineSkip * i, visualLines.getOffsetAt(i), visualLines.getLengthAt(i), lineWidth, lineSkip, selStart, selEnd - selStart, selectionDark, selectionLight);
			}
			else
			{
				line.paint(gc, originX, originY + lineSkip * i, visualLines.getOffsetAt(i), visualLines.getLengthAt(i), lineWidth, lineSkip);
			}
		}

		int bottom = originY + lineSkip * visualLines.getLineCount();		
		int clientHeight = getClientArea().height;
		if(bottom < clientHeight)
		{
			gc.setBackground(getBackground());
			gc.fillRectangle(0, bottom, lineWidth, clientHeight - bottom);
		}
	}

	public void setLineSkip(int lineSkip)
	{
		this.lineSkip = lineSkip;
	}
	
	public int getLineSkip()
	{
		return lineSkip;
	}
	
	public void setInsets(int top, int left, int bottom, int right)
	{
		insets[0] = top;
		insets[1] = left;
		insets[2] = bottom;
		insets[3] = right;
	}

	public void addLine(Line line)
	{
		Point bottom = getPointFromVisualLine(visualLines.getLineCount(), 0);
		line.wrapLine(visualLines, width, lines.size(), gc);
		lines.add(line);
		recalcScrollMaximum();
		int h = getClientArea().height;
		if(bottom.y < h)
			redraw(0, bottom.y, width, h - bottom.y, true);
	}

	public void addLines(Collection colection)
	{
		Point bottom = getPointFromVisualLine(visualLines.getLineCount(), 0);
		for(Iterator i = colection.iterator(); i.hasNext(); )
		{
			Line line = (Line)i.next();
			if(width == Integer.MAX_VALUE)
				visualLines.addLine(lines.size(), 0, line.getTextLength());
			else
				line.wrapLine(visualLines, width, lines.size(), gc);
			lines.add(line);
		}
		recalcScrollMaximum();
		int h = getClientArea().height;
		if(bottom.y < h)
			redraw(0, bottom.y, width, h - bottom.y, true);
	}
	
	public int getLineCount()
	{
		return lines.size();
	}

	public Line getLineAt(int index)
	{
		return (Line)lines.get(index);
	}
	
	public void clear()
	{
		lines.clear();
		visualLines.clear();
		getVerticalBar().setMaximum(insets[0] + insets[2]);
		setSelection(TextPosition.INVALID, TextPosition.INVALID, false);
		redraw();
	}
	
	public boolean canDoOperation(int operation)
	{
		if(operation == ITextOperationTarget.COPY)
			return !getSelection().isEmpty();
		else if(operation == ITextOperationTarget.SELECT_ALL)
			return true;
		return false;
	}
	
	public void doOperation(int operation)
	{
		if(operation == ITextOperationTarget.COPY)
			copyToClipboard();
		else if(operation == ITextOperationTarget.SELECT_ALL)
			selectAll();
	}
	
	public void copyToClipboard()
	{
		if(selectionStart.isValid() && selectionEnd.isValid())
			new Clipboard(getDisplay()).setContents(new Object[]{getText(selectionStart, selectionEnd)}, new Transfer[]{TextTransfer.getInstance()});
	}
	
	public String getText(TextPosition from, TextPosition to)
	{
		if(from.row == to.row)
		{
			return getLineAt(from.row).getText().substring(from.column, to.column);
		}
		else
		{
			StringBuffer buf = new StringBuffer();
			String f = getLineAt(from.row).getText();
			buf.append(f.substring(from.column, f.length())).append('\n');
			for(int i = from.row + 1; i < to.row - 1; i++)
				buf.append(getLineAt(i).getText()).append('\n');
			buf.append(getLineAt(to.row).getText().substring(0, to.column));
			return buf.toString();
		}
	}
	
	public void selectAll()
	{
		setSelection(new TextPosition(0, 0), new TextPosition(getLineCount() - 1, getLineAt(getLineCount() - 1).getTextLength()), true);
		redraw();
	}
	
	public void setSelection(ISelection selection)
	{
		if(selection instanceof TextSelection)
		{
			TextSelection sel = (TextSelection)selection;
			setSelection(sel.from, sel.to, true);
			redraw();
		}
	}
	
	public ISelection getSelection()
	{
		return getTextSelection();
	}
	
	public TextSelection getTextSelection()
	{
		return new TextSelection(selectionStart, selectionEnd);
	}

	public void setSelection(TextPosition selectionStart, TextPosition selectionEnd, boolean redraw)
	{
		if(0 < selectionStart.compareTo(selectionEnd))
		{
			TextPosition t = selectionStart;
			selectionStart = selectionEnd;
			selectionEnd = t;
		}
		
		if(this.selectionStart.equals(selectionStart) && this.selectionEnd.equals(selectionEnd))
			redraw = false;

		if(redraw)
			redrawSelection();

		this.selectionStart = selectionStart;
		this.selectionEnd = selectionEnd;
		
		if(redraw)
			redrawSelection();
			
		fireSelectionChangedEvent(new SelectionChangedEvent(this, getSelection()));
	}

	public void setSelection(LineFragment fragment, boolean redraw)
	{
		for(int i = 0; i < getLineCount(); i++)
		{
			Line line = getLineAt(i);
			int col = 0;
			for(int j = 0; j < line.getLineFragmentCount(); j++)
			{
				LineFragment f = line.getLineFragmentAt(j);
				if(f == fragment)
				{
					setSelection(new TextPosition(i, col), new TextPosition(i, col + f.getTextLength()), redraw);
					return;
				}
				else
				{
					col += f.getTextLength();
				}
			}
		}
	}
	
	private void redrawSelection()
	{
		if(selectionStart.isValid())
		{
			Point top = getPointFromIndex(selectionStart);
			Point bottom = getPointFromIndex(selectionEnd);
			redraw(0, top.y - lineSkip, width, bottom.y - top.y + lineSkip * 2, true);
		}
	}
	
	public void scrollTo(TextPosition index, int side)
	{
		int hit = findVisualLineOf(index);

		if(side == SWT.BOTTOM)
			getVerticalBar().setSelection(lineSkip * (hit + 1) + insets[0] - getVerticalBar().getThumb());
		else
			getVerticalBar().setSelection(lineSkip * hit + insets[0]);
		
		redraw();
	}
	
	public TextPosition getIndexFromPoint(Point point, boolean strict)
	{
		if(lineSkip == 0 || visualLines.getLineCount() == 0)
			return TextPosition.INVALID;
		int y = point.y + getVerticalBar().getSelection() - insets[0];
		int vline = y / lineSkip;
		if(strict && (y < 0 || vline < 0 || visualLines.getLineCount() <= vline))
			return TextPosition.INVALID;
		vline = Math.min(visualLines.getLineCount() - 1, Math.max(0, vline));
		int line = visualLines.getLogicalLineAt(vline);
		int col = getLineAt(line).getColumnIndexOf(point.x - insets[1], visualLines.getOffsetAt(vline), visualLines.getLengthAt(vline), strict, gc);
		if(col == -1)
			return TextPosition.INVALID;
		else
			return new TextPosition(line, col);
	}
	
	public Point getPointFromIndex(TextPosition index)
	{
		return getPointFromVisualLine(findVisualLineOf(index), index.column);
	}
	
	public LineFragment getLineFragmentFromIndex(TextPosition index)
	{
		if(index.row < 0 || getLineCount() <= index.row)
			return null;
		return getLineAt(index.row).getLineFragmentFromIndex(index.column);
	}
	
	private Point getPointFromVisualLine(int visualLineIndex, int logicalColumn)
	{
		if(visualLineIndex < 0 || visualLines.getLineCount() <= visualLineIndex)
		{
			return new Point(insets[1], insets[0]);
		}
		else
		{
			return new Point(insets[1] + getLineAt(visualLines.getLogicalLineAt(visualLineIndex)).getPointFromIndex(logicalColumn, visualLines.getOffsetAt(visualLineIndex), visualLines.getLengthAt(visualLineIndex), gc),
							insets[0] + lineSkip * visualLineIndex - getVerticalBar().getSelection());
		}
	}

	private int findVisualLineOf(TextPosition index)
	{
		int logicalLine = index.row;
		int offset = index.column;
		int start = 0;
		int end = visualLines.getLineCount();
		int hit = -1;
		while(1 < end - start)
		{
			int mid = start + (end - start) / 2;
			int log = visualLines.getLogicalLineAt(mid);
			if(log == logicalLine)
			{
				int off = visualLines.getOffsetAt(mid);
				int len = visualLines.getLengthAt(mid);
				if(off <= offset && offset < off + len)
				{
					hit = mid;
					break;
				}
				else if(off + len < off)
				{
					start = mid;
				}
				else
				{
					end = mid;
				}
			}
			else if(log < logicalLine)
			{
				start = mid;
			}
			else
			{
				end = mid;
			}
		}
		
		if(hit == -1)
			hit = end;
		return hit;
	}
	
	private void recalcScrollMaximum()
	{
		getVerticalBar().setMaximum(lineSkip * visualLines.getLineCount() + insets[0] + insets[2]);
		updateScrollThumb();
	}
	
	private void updateScrollThumb()
	{
		ScrollBar vertical = getVerticalBar();

		if(vertical.getMaximum() < getSize().y)
		{
			vertical.setEnabled(false);
			vertical.setThumb(getVerticalBar().getMaximum());
		}
		else
		{
			vertical.setEnabled(true);
			vertical.setThumb(getSize().y);
		}
	}
	
	public int setWrapWidth(int width)
	{
		wrapLines(width - getVerticalBar().getSize().x - insets[1] - insets[3]);
		return lineSkip * visualLines.getLineCount() + insets[0] + insets[2];
	}
	
	private void wrapLines(int width)
	{
		if(this.width != width)
		{
			TextPosition scroll = getIndexFromPoint(new Point(0, 0), false);

			visualLines.clear();
			wrapLines(width, lines, 0);
			this.width = width;
			recalcScrollMaximum();

			if(scroll.row == 0)
			{
				getVerticalBar().setSelection(0);
				redraw();
			}
			else
			{
				scrollTo(scroll, SWT.TOP);
			}
		}
	}
	
	private void wrapLines(int width, List lines, int lineNumberOffset)
	{
		int lc = lines.size();
		for(int i = 0; i < lc; i++)
			((Line)lines.get(i)).wrapLine(visualLines, width, lineNumberOffset + i, gc);
	}
	
	public void addSelectionChangedListener(ISelectionChangedListener listener)
	{
		selectionChangedListeners.add(listener);
	}
	
	public void removeSelectionChangedListener(ISelectionChangedListener listener)
	{
		selectionChangedListeners.remove(listener);
	}
	
	private void fireSelectionChangedEvent(SelectionChangedEvent e)
	{
		for(int i = 0; i < selectionChangedListeners.size(); i++)
		{
			ISelectionChangedListener l = (ISelectionChangedListener)selectionChangedListeners.get(i);
			l.selectionChanged(e);
		}
	}

	public void addTextTrackListener(TextTrackListener listener)
	{
		textTrackListeners.add(listener);
	}
	
	public void removeTextTrackListener(TextTrackListener listener)
	{
		textTrackListeners.remove(listener);
	}
	
	public void setToolTipSource(ColoredText toolTipSource)
	{
		this.toolTipSource = toolTipSource;
	}

	private void fireTextEvent(TextEvent e, int event)
	{
		for(int i = 0; i < textTrackListeners.size(); i++)
		{
			TextTrackListener l = (TextTrackListener)textTrackListeners.get(i);
			switch(event)
			{
			case TextEvent.EVENT_TEXT_ENTER:
				l.textEnter(e);
				break;

			case TextEvent.EVENT_TEXT_EXIT:
				l.textExit(e);
				break;

			case TextEvent.EVENT_TEXT_HOVER:
				l.textHover(e);
				break;

			case TextEvent.EVENT_TEXT_SELECTION_ENTER:
				l.textSelectionEnter(e);
				break;

			case TextEvent.EVENT_TEXT_SELECTION_EXIT:
				l.textSelectionExit(e);
				break;

			case TextEvent.EVENT_TEXT_SELECTION_HOVER:
				l.textSelectionHover(e);
				break;
			}
		}
	}
	
	public void setToolTipProvider(ToolTipProvider toolTipProvider)
	{
		this.toolTipProvider = toolTipProvider;
	}
	
	public ToolTipProvider getToolTipProvider()
	{
		return toolTipProvider;
	}

	private static final class VisualLines
	{
		private int[] lines = new int[30000];
		private int count;
		
		public final void addLine(int logicalLine, int offset, int length)
		{
			if(count * 3 == lines.length)
			{
				int[] newLines = new int[lines.length * 2];
				System.arraycopy(lines, 0, newLines, 0, lines.length);
				lines = newLines;
			}
			lines[count * 3] = logicalLine;
			lines[count * 3 + 1] = offset;
			lines[count * 3 + 2] = length;
			count++;
		}
		
		public final void clear()
		{
			count = 0;
		}
		
		public final int getLogicalLineAt(int visualLine)
		{
			return lines[visualLine * 3];
		}
		
		public final int getOffsetAt(int visualLine)
		{
			return lines[visualLine * 3 + 1];
		}
		
		public final int getLengthAt(int visualLine)
		{
			return lines[visualLine * 3 + 2];
		}
		
		public final int getLineCount()
		{
			return count;
		}
	}
	
	public static interface LinkTarget
	{
		public void linkClicked(ColoredText text);
	}
	
	public static interface ToolTipTarget
	{
	}
	
	public static interface ToolTipProvider
	{
		public Point fillToolTip(Composite parent, ColoredText text, int maxWidth, ToolTipTarget target);
		public Point fillToolTip(Composite parent, ColoredText text, int maxWidth, String selection);
		public void toolTipActivated(ColoredText text);
		public void toolTipDeactivated(ColoredText text);
	}
	
	public static interface TextTrackListener extends EventListener
	{
		public void textEnter(TextEvent e);
		public void textExit(TextEvent e);
		public void textHover(TextEvent e);
		public void textSelectionEnter(TextEvent e);
		public void textSelectionExit(TextEvent e);
		public void textSelectionHover(TextEvent e);
	}
	
	public static class TextEvent extends TypedEvent
	{
		static final int EVENT_TEXT_ENTER = 0;
		static final int EVENT_TEXT_EXIT = 1;
		static final int EVENT_TEXT_HOVER = 2;
		static final int EVENT_TEXT_SELECTION_ENTER = 3;
		static final int EVENT_TEXT_SELECTION_EXIT = 4;
		static final int EVENT_TEXT_SELECTION_HOVER = 5;

		public ColoredText text;
		public LineFragment fragment;
		public int x;
		public int y;

		private TextEvent(ColoredText source, MouseEvent e, LineFragment frag)
		{
			super(source);
			display = e.display;
			widget = source;
			time = e.time;
			text = source;
			fragment = frag;
			x = e.x;
			y = e.y;
		}
	}
	
	public static class TextSelection implements ISelection
	{
		public TextPosition from;
		public TextPosition to;

		public TextSelection(TextPosition from, TextPosition to)
		{
			this.from = from;
			this.to = to;
		}
		
		public boolean isEmpty()
		{
			return !from.isValid() || !to.isValid() || from.equals(to);
		}
		
		public boolean contains(TextPosition pos)
		{
			return pos.isValid() && from.compareTo(pos) <= 0 && pos.compareTo(to) <= 0;
		}
		
		public boolean equals(Object obj)
		{
			if(obj instanceof TextSelection)
				return ((TextSelection)obj).from.equals(from) && ((TextSelection)obj).to.equals(to);
			return false;
		}
	}
	
	public static class TextPosition
	{
		public static final TextPosition INVALID = new TextPosition(-1, -1);

		public int row;
		public int column;
		
		public TextPosition(int row, int column)
		{
			this.row = row;
			this.column = column;
		}
		
		public boolean isValid()
		{
			return this != INVALID;
		}
		
		public int hashCode()
		{
			return row + column;
		}
		
		public boolean equals(Object obj)
		{
			if(obj instanceof TextPosition)
				return ((TextPosition)obj).row == row && ((TextPosition)obj).column == column;
			return false;
		}
		
		public int compareTo(TextPosition pos)
		{
			if(row < pos.row)
				return -1;
			else if(row > pos.row)
				return 1;
			else if(column < pos.column)
				return -1;
			else if(column > pos.column)
				return 1;
			else
				return 0;
		}
	}
	
	public static class Line
	{
		private LineFragment fragment;
		private List fragments;
		private int charsCount;
		private int extentCache = -1;
		private int indent;

		public Line(int indent)
		{
			this.indent = indent;
		}
		
		public int getTextLength()
		{
			return charsCount;
		}
		
		public LineFragment getLineFragmentFromIndex(int col)
		{
			int fc = getLineFragmentCount();
			for(int i = 0; i < fc;  i++)
			{
				LineFragment f = getLineFragmentAt(i);
				if(f.getTextLength() <= col)
					col -= f.getTextLength();
				else
					return f;
			}
			return null;
		}
	
		public int getPointFromIndex(int index, int off, int len, GC gc)
		{
			if(off <= index && index < off + len)
				len = index - off;
			
			int x = indent;
			
			int fc = getLineFragmentCount();
			for(int i = 0; i < fc && 0 < len;  i++)
			{
				LineFragment f = getLineFragmentAt(i);
				if(f.getTextLength() <= off)
				{
					off -= f.getTextLength();
				}
				else
				{
					int l = Math.min(f.getTextLength() - off, len);
					if(off == 0 && l == f.getTextLength())
						x += f.getExtent(gc);
					else
						x += TextMeasurer.getTextExtent(f.getText(), off, l, gc);
					off = 0;
					len -= l;
				}
			}

			return x;
		}

		public int getColumnIndexOf(int x, int off, int len, boolean strict, GC gc)
		{
			x -= indent;
			
			if(strict && x < 0)
				return -1;
			
			int col = 0;
			
			int fc = getLineFragmentCount();
			for(int i = 0; i < fc && 0 < len;  i++)
			{
				LineFragment f = getLineFragmentAt(i);
				if(f.getTextLength() <= off)
				{
					off -= f.getTextLength();
					col += f.getTextLength();
				}
				else
				{
					int l = Math.min(f.getTextLength() - off, len);
					int index = f.getColumnIndexOf(x, off, l, gc);
					if(index == -1)
						col += f.getTextLength();
					else
						return col + off + index;
					if(off == 0 && l == f.getTextLength())
						x -= f.getExtent(gc);
					else
						x -= TextMeasurer.getTextExtent(f.getText(), off, l, gc);
					off = 0;
					len -= l;
				}
			}

			if(strict)
				return -1;
			else
				return col;
		}

		public void paint(GC gc, int x, int y, int off, int len, int lineWidth, int lineHeight)
		{
			x += indent;
			gc.fillRectangle(0, y, x, lineHeight);
			int fontHeight = gc.getFontMetrics().getHeight();
			gc.fillRectangle(0, y + fontHeight, lineWidth, lineHeight - fontHeight);

			int fc = getLineFragmentCount();
			for(int i = 0; i < fc && 0 < len;  i++)
			{
				LineFragment f = getLineFragmentAt(i);
				if(f.getTextLength() <= off)
				{
					off -= f.getTextLength();
				}
				else
				{
					int l = Math.min(f.getTextLength() - off, len);
					f.paint(gc, x, y, off, l);
					if(off == 0 && l == f.getTextLength())
						x += f.getExtent(gc);
					else
						x += TextMeasurer.getTextExtent(f.getText(), off, l, gc);
					off = 0;
					len -= l;
				}
			}
			
			gc.fillRectangle(x, y, lineWidth - x, lineHeight);
		}
		

		public void paintSelected(GC gc, int x, int y, int off, int len, int lineWidth, int lineHeight, int selectionOffset, int selectionLength, Color selectionDark, Color selectionLight)
		{
			if(selectionOffset < off)
			{
				selectionLength -= off - selectionOffset;
				selectionOffset = off;
			}
			
			if(off + len <= selectionOffset || selectionOffset + selectionLength <= off)
			{
				paint(gc, x, y, off, len, lineWidth, lineHeight);
				return;
			}
			
			x += indent;
			gc.fillRectangle(0, y, x, lineHeight);
			int fontHeight = gc.getFontMetrics().getHeight();
			gc.fillRectangle(0, y + fontHeight, lineWidth, lineHeight - fontHeight);

			int fc = getLineFragmentCount();
			for(int i = 0; i < fc && 0 < len;  i++)
			{
				LineFragment f = getLineFragmentAt(i);
				if(f.getTextLength() <= off)
				{
					off -= f.getTextLength();
					selectionOffset -= f.getTextLength();
				}
				else
				{
					int l = Math.min(f.getTextLength() - off, len);
					int sl = Math.min(f.getTextLength() - selectionOffset, selectionLength);
					f.paintSelected(gc, x, y, off, l, selectionOffset, sl, selectionDark, selectionLight);
					if(off == 0 && l == f.getTextLength())
						x += f.getExtent(gc);
					else
						x += TextMeasurer.getTextExtent(f.getText(), off, l, gc);
					if(selectionOffset < off + l)
					{
						selectionOffset = 0;
						selectionLength -= sl;
					}
					else
					{
						selectionOffset -= l;
					}
					off = 0;
					len -= l;
				}
			}
			
			gc.fillRectangle(x, y, lineWidth - x, lineHeight);
		}
		
		public void wrapLine(VisualLines visualLines, int width, int lineNumber, GC gc)
		{
			width -= indent;
			int extent = getExtent(gc);
			if(extent < width)
			{
				visualLines.addLine(lineNumber, 0, charsCount);
			}
			else
			{
				int off = 0;
				int len = 0;
				int w = 0;
				int fc = getLineFragmentCount();
				for(int i = 0; i < fc;  i++)
				{
					LineFragment f = getLineFragmentAt(i);
					int ext = f.getExtent(gc);
					if(w + ext <= width)
					{
						len += f.getTextLength();
						w += ext;
					}
					else
					{
						int br = 0;
						while(br < f.getTextLength())
						{
							int next = TextMeasurer.getWrap(f.getText(), br, width - w, gc);
							if(w == 0 && next == br)
								next = br + 1;
							len += next - br;
							if(next < f.getTextLength())
							{
								visualLines.addLine(lineNumber, off, len);
								off += len;
								len = 0;
								w = 0;
							}
							else
							{
								w = TextMeasurer.getTextExtent(f.getText(), br, next - br, gc);
							}
							br = next;
						}
					}
				}
				
				if(len != 0)
					visualLines.addLine(lineNumber, off, len);
			}
		}
		
		public int getPreferredLineWidth(GC gc)
		{
			return indent + getExtent(gc);
		}
		
		public int getExtent(GC gc)
		{
			if(extentCache == -1)
			{
				extentCache = 0;
				int fc = getLineFragmentCount();
				for(int i = 0; i < fc;  i++)
					extentCache += getLineFragmentAt(i).getExtent(gc);
			}
			return extentCache;
		}
		
		public void addLineFragment(LineFragment lineFragment)
		{
			charsCount += lineFragment.getTextLength();
			switch(getLineFragmentCount())
			{
			case 0:
				fragment = lineFragment;
				break;
			
			case 1:
				fragments = new ArrayList();
				fragments.add(fragment);
				fragments.add(lineFragment);
				fragment = null;
				break;
				
			default:
				fragments.add(lineFragment);
				break;
			}
		}
		
		public int getLineFragmentCount()
		{
			if(fragment != null)
				return 1;
			else if(fragments != null)
				return fragments.size();
			else
				return 0;
		}
		
		public LineFragment getLineFragmentAt(int index)
		{
			switch(getLineFragmentCount())
			{
			case 0:
				throw new IndexOutOfBoundsException();
			
			case 1:
				if(index == 0)
					return fragment;
				else
					throw new IndexOutOfBoundsException();
				
			default:
				return (LineFragment)fragments.get(index);
			}
		}
		
		public String toString()
		{
			return getText();
		}
		
		public String getText()
		{
			StringBuffer buf = new StringBuffer();
			for(int i = 0; i < getLineFragmentCount(); i++)
				buf.append(getLineFragmentAt(i).toString());
			return buf.toString();
		}
	}
	
	public static class LineFragment
	{
		private String text;
		private int offset;
		private int length;
		private Color color;
		private Font font;
		private boolean underline;
		private int extentCache = -1;

		public LineFragment(String text, Color color, Font font, boolean underline)
		{
			this(text, 0, text.length(), color, font, underline);
		}

		public LineFragment(String text, int offset, int length, Color color, Font font, boolean underline)
		{
			this.text = text;
			this.offset = offset;
			this.length = length;
			this.color = color;
			this.font = font;
			this.underline = underline;
		}

		public int getColumnIndexOf(int x, int off, int len, GC gc)
		{
			if(getExtent(gc) < x)
				return -1;
			return TextMeasurer.getColumnIndexOf(x, text, offset + off, len, gc);
		}

		public void paint(GC gc, int x, int y, int off, int len)
		{
			if(gc.getFont() != font)
				gc.setFont(font);
			if(gc.getForeground() != color)
				gc.setForeground(color);
			gc.drawString(text.substring(offset + off, offset + off + len), x, y, false);
			if(underline)
			{
				int ascent = gc.getFontMetrics().getAscent();
				gc.drawLine(x, y + ascent, x + getExtent(gc), y + ascent);
			}
		}
		
		public void paintSelected(GC gc, int x, int y, int off, int len, int selectionOffset, int selectionLength, Color selectionDark, Color selectionLight)
		{
			if(off + len <= selectionOffset || selectionOffset + selectionLength <= off)
			{
				paint(gc, x, y, off, len);
				return;
			}

			if(gc.getFont() != font)
				gc.setFont(font);
			if(gc.getForeground() != color)
				gc.setForeground(color);

			int ascent = underline ? gc.getFontMetrics().getAscent() : 0;
			int ulx = x;

			if(off < selectionOffset)
			{
				String f = text.substring(offset + off, offset + selectionOffset);
				gc.drawString(f, x, y, false);
				int ext = TextMeasurer.getTextExtent(f, 0, f.length(), gc);;
				x += ext;
				if(underline)
				{
					gc.drawLine(ulx, y + ascent, ulx + ext, y + ascent);
					ulx += ext;
				}
			}

			int postLen = (off + len) - (selectionOffset + selectionLength);

			Color bg = gc.getBackground();
			Color fg = gc.getForeground();
			if(fg.getRed() + fg.getGreen() + fg.getGreen() < 0x100)
			{
				gc.setBackground(selectionDark);
				gc.setForeground(selectionLight);
			}
			else
			{
				gc.setBackground(selectionLight);
				gc.setForeground(selectionDark);
			}
			String f = text.substring(offset + selectionOffset, offset + selectionOffset + selectionLength);
			gc.drawString(f, x, y, false);
			if(0 < postLen)
			{
				int ext = TextMeasurer.getTextExtent(f, 0, f.length(), gc);;
				x += ext;
				if(underline)
				{
					gc.drawLine(ulx, y + ascent, ulx + ext, y + ascent);
					ulx += ext;
				}
			}
			else
			{
				if(underline)
					gc.drawLine(ulx, y + ascent, x + getExtent(gc), y + ascent);
			}
			gc.setBackground(bg);
			gc.setForeground(fg);

//System.err.println(off + " " + len + " " + selectionOffset + " " + selectionLength);
//System.err.println(postLen);
			if(0 < postLen)
			{
				int o = selectionOffset + selectionLength;
				f = text.substring(offset + o, offset + o + postLen);
				gc.drawString(f, x, y, false);
				if(underline)
					gc.drawLine(ulx, y + ascent, x + getExtent(gc), y + ascent);
			}
		}
		
		public int getExtent(GC gc)
		{
			if(gc.getFont() != font)
				gc.setFont(font);
			if(extentCache == -1)
				extentCache = TextMeasurer.getTextExtent(text, offset, length, gc);
			return extentCache;
		}
		
		public String toString()
		{
			return getText();
		}
		
		public String getText()
		{
			return text.substring(offset, offset + length);
		}
		
		public Color getColor()
		{
			return color;
		}
		
		public void setColor(Color color)
		{
			this.color = color;
		}
		
		public Font getFont()
		{
			return font;
		}

		public int getTextLength()
		{
			return length;
		}

		public boolean getUnderline()
		{
			return underline;
		}

	}
	
	private static class TextMeasurer
	{
		private static final Map CHAR_ADVANCE_CACHE = new HashMap();

		public static int getColumnIndexOf(int x, String text, int off, int len, GC gc)
		{
			int extent = 0;
			int[] cache = getCharAdvanceCache(gc.getFont());
			for(int i = 0; i < len; i++)
			{
				int adv = getAdvanceWidth(cache, text.charAt(off + i), gc);
				if(x < extent + adv / 2)
					return i;
				extent += adv;
			}
			return len;
		}

		public static int getTextExtent(String text, int off, int len, GC gc)
		{
			int extent = 0;
			int[] cache = getCharAdvanceCache(gc.getFont());
			for(int i = 0; i < len; i++)
				extent += getAdvanceWidth(cache, text.charAt(off + i), gc);
			return extent;
		}

		private static int getAdvanceWidth(int[] cache, char ch, GC gc)
		{
			int w = cache[ch];
			if(w == 0)
			{
				w = gc.getAdvanceWidth(ch);
				cache[ch] = w;
			}
			return w;
		}
		
		private static int[] getCharAdvanceCache(Font font)
		{
			if(CHAR_ADVANCE_CACHE.containsKey(font))
			{
				return (int[])CHAR_ADVANCE_CACHE.get(font);
			}
			else
			{
				int[] cache = new int[65536];
				CHAR_ADVANCE_CACHE.put(font, cache);
				return cache;
			}
		}

		private static int getWrap(String text, int off, int width, GC gc)
		{
			int br = off;
			int len = text.length();
			int w = 0;
			int[] cache = getCharAdvanceCache(gc.getFont());
			for(int i = off; i < len; i++)
			{
				int chw = getAdvanceWidth(cache, text.charAt(i), gc);
				if(width < w + chw)
					break;
				w += chw;
				br++;
			}
			return br;
		}
	}
	
	private class PageMoveKeyListener implements KeyListener
	{
		private int keyRepeat;
		private boolean repeating;
		private long baseTime;
	
		public void keyPressed(KeyEvent e)
		{
			if(keyRepeat == 0)
			{
				switch(e.keyCode)
				{
				case SWT.ARROW_DOWN:
				case SWT.ARROW_UP:
				case SWT.PAGE_DOWN:
				case SWT.PAGE_UP:
					keyRepeat = e.keyCode;
					break;
				}
				
				if(e.character == ' ' || e.character == SWT.BS)
					keyRepeat = e.character;
				
				if(keyRepeat != 0)
				{
					repeating = false;
					baseTime = System.currentTimeMillis();
					final long pressedTime = baseTime;
					final int keyRepeat = this.keyRepeat;
	
					getShell().getDisplay().timerExec(10, new Runnable()
						{
							private static final int FIRST_MOVE = 70;
	
							public void run()
							{
								long curTime = System.currentTimeMillis();
								if(curTime - pressedTime < FIRST_MOVE && !repeating)
									getShell().getDisplay().timerExec(10, this);
	
								if(!repeating)
								{
									long firstPeriod = Math.min(pressedTime + FIRST_MOVE, curTime);
									keyAction(keyRepeat, getVerticalBar().getThumb() / 8 * (int)(firstPeriod - baseTime) / FIRST_MOVE);
									baseTime = firstPeriod;
								}
							}
						});
				}
				else if(e.keyCode == SWT.HOME)
				{
					getVerticalBar().setSelection(0);
					redraw();
				}
				else if(e.keyCode == SWT.END)
				{
					getVerticalBar().setSelection(getVerticalBar().getMaximum());
					redraw();
				}
			}
			else if(!repeating)
			{
				repeating = true;
				baseTime = System.currentTimeMillis();
				getShell().getDisplay().timerExec(10, new Runnable()
					{
						private static final int FIRST_MOVE = 100;
	
						public void run()
						{
							if(repeating)
							{
								getShell().getDisplay().timerExec(10, this);
								long curTime = System.currentTimeMillis();
								keyAction(keyRepeat, (int)((curTime - baseTime) * 1.5));
								baseTime = curTime;
							}
						}
					});
			}
		}
		
		private void keyAction(int keyCode, int increment)
		{
			switch(keyCode)
			{
			case SWT.ARROW_DOWN:
				break;
	
			case SWT.ARROW_UP:
				increment = -increment;
				break;
	
			case SWT.PAGE_DOWN:
			case ' ':
				increment *= 7;
				break;
	
			case SWT.PAGE_UP:
			case SWT.BS:
				increment *= -7;
				break;
			}
			
			getVerticalBar().setSelection(Math.min(Math.max(getVerticalBar().getSelection() + increment, 0), getVerticalBar().getMaximum()));
			redraw();
		}
	
		public void keyReleased(KeyEvent e)
		{
			keyRepeat = 0;
			repeating = false;
		}
	}
	
	private class MouseDragListener implements MouseListener, MouseMoveListener, MouseTrackListener
	{
		private TextPosition selectionStart = TextPosition.INVALID;
		private TextPosition selectionEnd = TextPosition.INVALID;
		private int increment;
		private Point lastMouseLocation;
		private TextPosition lastTextPosition = TextPosition.INVALID;
		private LineFragment lastLineFragment;
		private LineFragment lastHoverLineFragment;
		private TextSelection lastSelection;
		private TextSelection lastHoverSelection;
		private LinkTarget lastLinkTarget;
		
		public void mouseDown(MouseEvent e)
		{
			if(e.button == 1)
			{
				toolTipLocked = false;
				disposeHover();

				setCapture(true);
	
				selectionStart = getIndexFromPoint(new Point(e.x, e.y), false);
				selectionEnd = TextPosition.INVALID;
				setSelection(TextPosition.INVALID, selectionEnd, true);
				getShell().getDisplay().timerExec(10, new Runnable()
					{
						public void run()
						{
							if(increment != 0)
							{
								getVerticalBar().setSelection(Math.min(Math.max(getVerticalBar().getSelection() + increment, 0), getVerticalBar().getMaximum()));
								selectionEnd = getIndexFromPoint(lastMouseLocation, false);
								setSelection(selectionStart, selectionEnd, false);
								redraw();
							}
							if(selectionStart.isValid())
								getShell().getDisplay().timerExec(10, this);
						}
					});
				
				if(selectionStart.isValid())
				{
					TextPosition position = getIndexFromPoint(new Point(e.x, e.y), true);
					if(position.isValid())
					{
						LineFragment f = getLineFragmentFromIndex(position);
						if(f instanceof LinkTarget)
						{
							setSelection(f, true);
							lastLinkTarget = (LinkTarget)f;
						}
					}
				}
			}
		}
		
		public void mouseUp(MouseEvent e)
		{
			if(e.button == 1)
			{
				TextPosition position = getIndexFromPoint(new Point(e.x, e.y), true);
				if(position.isValid())
				{
					LineFragment f = getLineFragmentFromIndex(position);
					if(f != null && f == lastLinkTarget)
					{
						disposeHover();
						setSelection(f, true);
						lastLinkTarget.linkClicked(ColoredText.this);
						lastLinkTarget = null;
					}
				}
			}

			setCapture(false);
			selectionStart = TextPosition.INVALID;
			selectionEnd = TextPosition.INVALID;
			increment = 0;
		}
		
		public void mouseDoubleClick(MouseEvent e)
		{
			TextPosition position = getIndexFromPoint(new Point(e.x, e.y), false);
			if(position.isValid())
			{
				Line line = getLineAt(position.row);
				String s = line.toString();
				if(s.length() == 0)
					return;
				if(position.column == s.length())
					position.column--;

				TextPosition start = new TextPosition(position.row, position.column);
				TextPosition end = new TextPosition(position.row, position.column);

				Character.UnicodeBlock block = Character.UnicodeBlock.of(s.charAt(position.column));
				
				while(0 < start.column && Character.UnicodeBlock.of(s.charAt(start.column - 1)) == block)
					start.column--;
				
				while(end.column < s.length() && Character.UnicodeBlock.of(s.charAt(end.column)) == block)
					end.column++;
				
				setSelection(start, end, true);
			}
		}
		
		public void mouseMove(MouseEvent e)
		{
			if(e.y < 0 || getSize().y < e.y)
			{
				setCapture(false);
				if(!toolTipLocked)
					disposeHover();
				return;
			}

			Point mouseLocation = new Point(e.x, e.y);
			TextPosition position = getIndexFromPoint(mouseLocation, true);
			
			if(position.isValid())
			{
				LineFragment f = getLineFragmentFromIndex(position);
				if(lastLineFragment != f)
				{
					if(lastLineFragment != null)
						fireTextEvent(new TextEvent(ColoredText.this, e, lastLineFragment), TextEvent.EVENT_TEXT_EXIT);
					if(f != null)
						fireTextEvent(new TextEvent(ColoredText.this, e, f), TextEvent.EVENT_TEXT_ENTER);

					lastLineFragment = f;
					lastHoverLineFragment = null;
				}
				
				TextSelection s = getTextSelection();
				if(!s.contains(position))
					s = null;
				if((s == null && lastSelection != null) || (s != null && lastSelection == null))
				{
					if(lastSelection != null)
						fireTextEvent(new TextEvent(ColoredText.this, e, f), TextEvent.EVENT_TEXT_SELECTION_EXIT);
					if(s != null)
						fireTextEvent(new TextEvent(ColoredText.this, e, f), TextEvent.EVENT_TEXT_SELECTION_ENTER);

					lastSelection = s;
					lastHoverSelection = null;
				}
			}
			else
			{
				if(lastLineFragment != null)
					fireTextEvent(new TextEvent(ColoredText.this, e, lastLineFragment), TextEvent.EVENT_TEXT_EXIT);
				lastLineFragment = null;
				lastHoverLineFragment = null;
				if(lastSelection != null)
					fireTextEvent(new TextEvent(ColoredText.this, e, null), TextEvent.EVENT_TEXT_SELECTION_EXIT);
				lastSelection = null;
				lastHoverSelection = null;
			}

			if(selectionStart.isValid())
			{
				lastMouseLocation = mouseLocation;
				selectionEnd = getIndexFromPoint(mouseLocation, false);
				ScrollBar vertical = getVerticalBar();
				increment = 0;
				int h =  getSize().y;
				int grad = h / 16;
				int base = h / 8;
				if(e.y < 0)
					increment = -base;
				else if(e.y < grad && grad != 0)
					increment = -base * (grad - e.y) / grad;
				else if(h < e.y)
					increment = base;
				else if(h - grad < e.y && grad != 0)
					increment = base * (grad - (h - e.y)) / grad;

				setSelection(selectionStart, selectionEnd, increment == 0);
				if(increment != 0)
				{
					vertical.setSelection(vertical.getSelection() + increment);
					redraw();
				}
			}
		}
		
		public void mouseEnter(MouseEvent e)
		{
		}
		
		public void mouseExit(MouseEvent e)
		{
		}
		
		public void mouseHover(MouseEvent e)
		{
			if(lastHoverSelection == null)
			{
				Point mouseLocation = new Point(e.x, e.y);
				TextPosition position = getIndexFromPoint(mouseLocation, true);
				
				if(position.isValid())
				{
					TextSelection s = getTextSelection();
					if(!s.contains(position))
						s = null;
					if(s != null)
					{
						lastHoverSelection = s;
						fireTextEvent(new TextEvent(ColoredText.this, e, null), TextEvent.EVENT_TEXT_SELECTION_HOVER);
						return;
					}
				}
			}

			if(lastHoverLineFragment == null)
			{
				Point mouseLocation = new Point(e.x, e.y);
				TextPosition position = getIndexFromPoint(mouseLocation, true);
				
				if(position.isValid())
				{
					LineFragment f = getLineFragmentFromIndex(position);
					if(f != null)
					{
						lastHoverLineFragment = f;
						fireTextEvent(new TextEvent(ColoredText.this, e, f), TextEvent.EVENT_TEXT_HOVER);
					}
				}
			}
		}

	}
}
