package monalipse.editors;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;

import monalipse.part.CancelableRunner;
import monalipse.server.BBSServerManager;
import monalipse.server.IBBSServer;
import monalipse.server.INewResponseLineFragment;
import monalipse.server.IResponseEnumeration;
import monalipse.server.IResponseHeaderLine;
import monalipse.server.IThreadContentProvider;
import monalipse.server.IThreadToolTipProvider;
import monalipse.server.RendererResource;
import monalipse.widgets.ColoredText;
import monalipse.widgets.ColoredText.TextEvent;
import monalipse.widgets.ColoredText.TextSelection;
import monalipse.widgets.ColoredText.ToolTipTarget;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.text.ITextOperationTarget;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.ScrollBar;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.EditorPart;

public class ThreadViewerEditor extends EditorPart implements ColoredText.ToolTipProvider
{
	private ThreadEditorInput input;
	private IThreadContentProvider content;
	private Display display;
	private ColoredText text;
	private RendererResource rendererResource;
	private int sequence;
	private int responseCount;
	private boolean writable;
	private boolean toolTipActive;

	public ThreadViewerEditor()
	{
	}

	public void updateThread()
	{
		final CancelableRunner cancelable = ((ThreadViewerEditorActionBarContributor)getEditorSite().getActionBarContributor()).getCancelable();
		cancelable.run(this, new IRunnableWithProgress()
			{
				public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException
				{
					updateThread(monitor, cancelable, true);
				}
			});
	}
	
	private void updateThread(IProgressMonitor monitor, CancelableRunner cancelable, final boolean download)
	{
		monitor.beginTask("getting thread", 130);

		try
		{
			IThreadContentProvider thread = getContentProvider();
	
			display.syncExec(new Runnable()
				{
					public void run()
					{
						for(int i = 0; i < text.getLineCount(); i++)
							unmarkNewResponse(text.getLineAt(i));
						text.redraw();
					}
				});

			monitor.worked(10);
			IResponseEnumeration e;
			if(download)
				e = thread.updateResponses(cancelable, sequence, responseCount, rendererResource);
			else
				e = thread.getResponses(sequence, responseCount, rendererResource);
			writable = e.isWritable();
			monitor.worked(10);
	
			if(e == null)
				return;

			try
			{
				if(!e.isPartialContent())
				{
					responseCount = 0;
					display.syncExec(new Runnable()
						{
							public void run()
							{
								text.clear();
							}
						});
				}

				monitor.worked(10);
				
				final ColoredText.TextPosition[] latestResponseLocation = new ColoredText.TextPosition[1];
				if(download)
				{
					display.syncExec(new Runnable()
						{
							public void run()
							{
								if(responseCount != 0 && saveScrollPosition() == responseCount)
								{
									for(int i = 0; i < text.getLineCount(); i++)
									{
										ColoredText.Line line = text.getLineAt(i);
										if(line instanceof IResponseHeaderLine &&
											((IResponseHeaderLine)line).getReponseNumber() == responseCount)
										{
											latestResponseLocation[0] = new ColoredText.TextPosition(i, 0);
										}
									}
								}
							}
						});
				}

				final List lines = new ArrayList();
				long bulkWrite = System.currentTimeMillis();
				int writePeriod = responseCount + 10;
				
				Runnable applyLines = new Runnable()
					{
						public void run()
						{
							if(!download)
							{
								for(int i = 0; i < lines.size(); i++)
									unmarkNewResponse((ColoredText.Line)lines.get(i));
							}

							text.addLines(lines);
							lines.clear();
							
							if(latestResponseLocation[0] != null)
							{
								Point pt = text.getPointFromIndex(latestResponseLocation[0]);
								ScrollBar vert = text.getVerticalBar();
								if(vert.getSelection() + pt.y < vert.getMaximum() - vert.getThumb())
								{
									text.scrollTo(latestResponseLocation[0], SWT.TOP);
									latestResponseLocation[0] = null;
								}
							}
						}
					};

				try
				{
					int respWork = 0;
					while(e.hasNextResponse())
					{
						int newWork = e.getProgressHint();
						if(respWork < newWork)
						{
							monitor.worked(newWork - respWork);
							respWork = newWork;
						}

						if(e.getNextResponse(lines))
						{
							if(responseCount == 0)
							{
								input.setTitle(e.getTitle());
								display.syncExec(new Runnable()
									{
										public void run()
										{
											setTitle(input.getTitle());
										}
									});
							}

							responseCount++;

							if(responseCount < 5 || responseCount == writePeriod || 300 < System.currentTimeMillis() - bulkWrite)
							{
								bulkWrite = System.currentTimeMillis();
								display.syncExec(applyLines);
							}
							
							if(responseCount == 1)
							{
								StringBuffer buf = new StringBuffer();
								for(int i = 0; i < lines.size(); i++)
								{
									ColoredText.Line line = (ColoredText.Line)lines.get(i);
									buf.append(line.toString());
								}
								String tooltipText = input.getTitle() + "\n" + input.getURLHint() + "\n" + buf.toString();
								input.setToolTipText(tooltipText);
								display.syncExec(new Runnable()
									{
										public void run()
										{
											setTitleToolTip(input.getToolTipText());
											setTitle(input.getTitle());
										}
									});
							}
						}
					}
				}
				catch (InterruptedException ex)
				{
				}
				finally
				{
					if(0 < lines.size())
						display.syncExec(applyLines);
				}

				sequence = e.getSequenceNumber();
				
				if(latestResponseLocation[0] != null)
				{
					display.syncExec(new Runnable()
						{
							public void run()
							{
								text.scrollTo(latestResponseLocation[0], SWT.TOP);
							}
						});
				}
			}
			finally
			{
				e.close();
			}
		}
		finally
		{
			monitor.done();
		}
	}
	
	private void unmarkNewResponse(ColoredText.Line line)
	{
		for(int j = 0; j < line.getLineFragmentCount(); j++)
		{
			ColoredText.LineFragment f = line.getLineFragmentAt(j);
			if(f instanceof INewResponseLineFragment)
				((INewResponseLineFragment)f).unmark(rendererResource);
		}
	}
	
	public void disposeCache()
	{
		try
		{
			getContentProvider().getLogFile().delete(true, false, new NullProgressMonitor());
		}
		catch (CoreException e)
		{
		}
	}
	
	public void lockToolTip()
	{
		text.lockToolTip();
	}
	
	public ITextOperationTarget getTextOperationTarget()
	{
		return text;
	}

	private int saveScrollPosition()
	{
		ColoredText.TextPosition bottomLine = text.getIndexFromPoint(new Point(0, text.getSize().y - 1), false);
		if(bottomLine.isValid())
		{
			for(int i = bottomLine.row; 0 <= i; i--)
			{
				ColoredText.Line line = text.getLineAt(i);
				if(line instanceof IResponseHeaderLine)
				{
					int read = ((IResponseHeaderLine)line).getReponseNumber();
					try
					{
						input.getLogFolder().setPersistentProperty(new QualifiedName(ThreadViewerEditor.class.getName(), input.getID() + ".read"), String.valueOf(read));
						input.getLogFolder().setPersistentProperty(new QualifiedName(ThreadViewerEditor.class.getName(), input.getID() + ".readLine"), String.valueOf(bottomLine.row));
					}
					catch (CoreException e)
					{
					}
					return read;
				}
			}
		}
		return 0;
	}
	
	private void loadScrollPosition()
	{
		try
		{
			ColoredText.TextPosition read = new ColoredText.TextPosition(Integer.parseInt(input.getLogFolder().getPersistentProperty(new QualifiedName(ThreadViewerEditor.class.getName(), input.getID() + ".readLine"))), 0);
			text.scrollTo(read, SWT.BOTTOM);
		}
		catch (CoreException e)
		{
		}
		catch (NumberFormatException e)
		{
		}
	}

	public IThreadContentProvider getContentProvider()
	{
		if(content == null)
		{
			IBBSServer server = BBSServerManager.getInstanceOf(input.getLogFolder().getProject(), getSite().getWorkbenchWindow());
			content = server.getThreadContentProviderOf(input.getBaseURL(), input.getLogFolder(), input.getID(), input.getName());
		}
		return content;
	}

	public boolean isWritable()
	{
		return writable;
	}

	public void doSave(IProgressMonitor monitor)
	{
	}

	public void doSaveAs()
	{
	}

	public void gotoMarker(IMarker marker)
	{
	}

	public void init(IEditorSite site, IEditorInput input) throws PartInitException
	{
		setSite(site);
		setInput(input);
		
		if(input instanceof ThreadEditorInput)
			this.input = (ThreadEditorInput)input;
	}

	public boolean isDirty()
	{
		return false;
	}

	public boolean isSaveAsAllowed()
	{
		return false;
	}

	public void createPartControl(Composite parent)
	{
		display = parent.getShell().getDisplay();
		
		rendererResource = new RendererResource(display);

		text = new ColoredText(parent, SWT.V_SCROLL);
		text.setLineSkip(rendererResource.lineSkip);
		text.setBackground(rendererResource.backgroundGray);
		text.setForeground(rendererResource.black);
		text.setToolTipProvider(this);
		text.addTextTrackListener(new ColoredText.TextTrackListener()
			{
				public void textEnter(TextEvent e)
				{
					if(e.fragment instanceof IThreadToolTipProvider)
						((IThreadToolTipProvider)e.fragment).prefetchToolTip();
				}
	
				public void textExit(TextEvent e)
				{
				}
	
				public void textHover(TextEvent e)
				{
				}
				
				public void textSelectionEnter(TextEvent e)
				{
				}
				
				public void textSelectionExit(TextEvent e)
				{
				}
				
				public void textSelectionHover(TextEvent e)
				{
				}

			});
		text.addSelectionChangedListener(new ISelectionChangedListener()
			{
				public void selectionChanged(SelectionChangedEvent event)
				{
					((ThreadViewerEditorActionBarContributor)getEditorSite().getActionBarContributor()).updateSelectionDependentActions();
				}
			});
		
		hookContextMenu();

		getSite().getPage().addPartListener(new PartActivationListener());

		CancelableRunner cancelable = ((ThreadViewerEditorActionBarContributor)getEditorSite().getActionBarContributor()).getCancelable();
		cancelable.setDisplay(display);

		updateThread(new NullProgressMonitor(), cancelable, false);
		new Thread(new Runnable()
			{
				public void run()
				{
					display.asyncExec(new Runnable()
						{
							public void run()
							{
								loadScrollPosition();
							}
						});
				}
			}).start();
	}

	private void hookContextMenu()
	{
		MenuManager menuMgr = new MenuManager("#PopupMenu");
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener()
			{
				public void menuAboutToShow(IMenuManager manager)
				{
					ThreadViewerEditorActionBarContributor cont = (ThreadViewerEditorActionBarContributor)getEditorSite().getActionBarContributor();
					cont.contributeToContextMenu(manager);
				}
			});
		Menu menu = menuMgr.createContextMenu(text);
		text.setMenu(menu);
	}
	
	public Point fillToolTip(Composite parent, ColoredText text, int maxWidth, ToolTipTarget target)
	{
		if(target instanceof IThreadToolTipProvider)
			return ((IThreadToolTipProvider)target).fillToolTip(parent, text, maxWidth);
		return new Point(0, 0);
	}
	
	public Point fillToolTip(Composite parent, ColoredText text, int maxWidth, String selection)
	{
		return getContentProvider().fillToolTip(parent, text, maxWidth, selection);
	}
	
	public void toolTipActivated(ColoredText text)
	{
		toolTipActive = true;
		((ThreadViewerEditorActionBarContributor)getEditorSite().getActionBarContributor()).updateToolTipDependentActions();
	}
	
	public void toolTipDeactivated(ColoredText text)
	{
		toolTipActive = false;
		((ThreadViewerEditorActionBarContributor)getEditorSite().getActionBarContributor()).updateToolTipDependentActions();
	}
	
	public boolean isToolTipActive()
	{
		return toolTipActive;
	}

	public void setFocus()
	{
		text.setFocus();
	}
	
	public void dispose()
	{
		if(text != null)
			text.dispose();
		text = null;
		if(rendererResource != null)
			rendererResource.dispose();
		rendererResource = null;
		super.dispose();
	}
	
	private class PartActivationListener implements IPartListener
	{
		private boolean active = true;

		public void partActivated(IWorkbenchPart part)
		{
			active = true;
		}

		public void partBroughtToTop(IWorkbenchPart part)
		{
		}

		public void partClosed(IWorkbenchPart part)
		{
		}

		public void partDeactivated(IWorkbenchPart part)
		{
			if(part == ThreadViewerEditor.this && active)
			{
				saveScrollPosition();
				active = false;
			}
		}

		public void partOpened(IWorkbenchPart part)
		{
		}
	}

}
