package com.meterware.httpunit;
/********************************************************************************************************************
 * $Id: UncheckedParameterHolder.java,v 1.5 2002/02/04 22:33:55 russgold Exp $
 *
 * Copyright (c) 2002, Russell Gold
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
 * documentation files (the "Software"), to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and
 * to permit persons to whom the Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO
 * THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 *******************************************************************************************************************/
import java.io.File;
import java.io.InputStream;
import java.io.IOException;

import java.util.Hashtable;
import java.util.Enumeration;
import java.net.URLEncoder;


/**
 *
 * @author <a href="mailto:russgold@acm.org">Russell Gold</a>
 **/
class UncheckedParameterHolder extends ParameterHolder {

    private static final String[] NO_VALUES = new String[ 0 ];
    private final String _characterSet;

    private Hashtable _parameters = new Hashtable();
    private boolean   _submitAsMime;


    UncheckedParameterHolder() {
        _characterSet = HttpUnitOptions.getDefaultCharacterSet();
    }


    UncheckedParameterHolder( WebRequestSource source ) {
        _characterSet = source.getCharacterSet();
        _submitAsMime = source.isSubmitAsMime();
        String[] names = source.getParameterNames();
        for (int i = 0; i < names.length; i++) {
            if (!source.isFileParameter( names[i] )) {
                _parameters.put( names[i], source.getParameterValues( names[i] ) );
            }
        }
    }


    /**
     * Specifies the position at which an image button (if any) was clicked.
     **/
    void selectImageButtonPosition( SubmitButton imageButton, int x, int y ) {
        setParameter( imageButton.getName() + ".x", Integer.toString( x ) );
        setParameter( imageButton.getName() + ".y", Integer.toString( y ) );
    }


    /**
     * Does nothing, since unchecked requests treat all parameters the same.
     **/
    void recordPredefinedParameters( ParameterProcessor processor ) throws IOException {
    }


    /**
     * Iterates through the parameters in this holder, recording them in the supplied parameter processor.
     **/
    void recordParameters( ParameterProcessor processor ) throws IOException {
        Enumeration e = _parameters.keys();

        while (e.hasMoreElements()) {
            String name = (String) e.nextElement();
            Object value = _parameters.get( name );
            if (value instanceof String) {
                processor.addParameter( name, (String) value, _characterSet );
            } else if (value instanceof String[]) {
                String[] values = (String[]) value;
                for (int i = 0; i < values.length; i++) processor.addParameter( name, values[i], _characterSet );
            } else if (value instanceof UploadFileSpec[]) {
                UploadFileSpec[] files = (UploadFileSpec[]) value;
                for (int i = 0; i < files.length; i++) processor.addFile( name, files[i] );
            }
        }
    }


    String[] getParameterNames() {
        return (String[]) _parameters.keySet().toArray( new String[ _parameters.size() ] );
    }


    String getParameterValue( String name ) {
        String[] values = getParameterValues( name );
        return values.length == 0 ? null : values[0];
    }


    String[] getParameterValues( String name ) {
        Object result = _parameters.get( name );
        if (result instanceof String) return new String[] { (String) result };
        if (result instanceof String[]) return (String[]) result;
        if (result instanceof UploadFileSpec) return new String[] { result.toString() };
        return NO_VALUES;
    }


    void removeParameter( String name ) {
        _parameters.remove( name );
    }


    void setParameter( String name, String value ) {
        _parameters.put( name, value );
    }


    void setParameter( String name, String[] values ) {
        _parameters.put( name, values );
    }


    void setParameter( String name, UploadFileSpec[] files ) {
        _parameters.put( name, files );
    }


    boolean isFileParameter( String name ) {
        return true;
    }


    boolean isSubmitAsMime() {
        return _submitAsMime;
    }


    void setSubmitAsMime( boolean mimeEncoded ) {
        _submitAsMime = mimeEncoded;
    }
}
