/*
 * Copyright 2023 yaman.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.synthtarou.midimixer.mx70console;

import java.awt.Color;
import java.awt.Component;
import java.util.ArrayList;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JList;
import javax.swing.ListCellRenderer;
import javax.swing.ListModel;
import javax.swing.SwingUtilities;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;
import jp.synthtarou.midimixer.libs.MXTimer;
import jp.synthtarou.midimixer.libs.midi.MXTraceNumber;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class ConsoleModel implements ListModel<String>{
    ArrayList<ConsoleElement> _list = new ArrayList();
    ArrayList<ListDataListener> _listener = new ArrayList();
    static final int _capacity = 5000;
    static final int _timer = 500;
    int _startPos = 0;
    int _writePos = 0;
    JList _refList;
    MXTraceNumber _selectedTraceNumber = null;

    ListCellRenderer<Object> _renderer = new ListCellRenderer<Object>() {
        DefaultListCellRenderer _def = new DefaultListCellRenderer();

        public Component getListCellRendererComponent(JList list, Object var, int index, boolean isSelected, boolean cellHasFocus) {
            Component c = _def.getListCellRendererComponent(list, var, index, isSelected, cellHasFocus);

            ConsoleElement value = getConsoleElement(index);
            if (value == null) {
                return c;
            }

            if (_selectedTraceNumber ==  value.getTraceNumber()) {
                c.setForeground(Color.red);
                if (_selectedTraceNumber != null) {
                    c.setBackground(Color.cyan);
                }else {
                    c.setBackground(Color.gray);
                }
            }
            return c;
        }
    };
    
    public void bind(JList list) {
        list.setModel(this);
        list.setCellRenderer(_renderer);
        _refList = list;
    }
    
    @Override
    public int getSize() {
        return _capacity;
        //return _list.size();
    }

    @Override
    public String getElementAt(int index) {
        ConsoleElement e = getConsoleElement(index);
        if (e == null) {
            return "";
        }
        return e.formatMessageLong();
    }
    
    public int viewIndex(int viewpos) {
        int index = viewpos + _list.size() - _capacity + _startPos;
        if (index < 0) {
            return -1;
        }
        while (index >= _capacity) {
            index  -= _capacity;
        }
        if (index  >= _list.size()) {
            return -1;
        }
        return index;
    }

    public int indexView(int index) {
        int  viewpos = index - _list.size() + _capacity - _startPos;
        while (viewpos < 0) {
            viewpos += _capacity;
        }
        while (viewpos >= _capacity) {
            viewpos  -= _capacity;
        }
        return viewpos;
    }
    
    public ConsoleElement getConsoleElement(int viewpos) {
        int index = viewIndex(viewpos);
        if (index >= 0) {
            return _list.get(index);
        }
        return null;
    }

    public synchronized void add(ConsoleElement e) {
        /* Ignore */
        if (_writePos >= _capacity) {
            _writePos = 0;
        }
        if (_list.size() >= _capacity) {
            int prev = _writePos - 1;
            if (prev < 0) prev = _capacity - 1;
            ConsoleElement prevE = _list.get(prev);
            if (prevE != null) {
                MXTraceNumber prevNumber = prevE.getTraceNumber();
                int comp = prevNumber.compareTo(e.getTraceNumber());
                if (comp > 0) {
                    System.err.println("************************************");
                    System.err.println("This");
                    e.getTraceNumber()._traceBirthDay.printStackTrace();
                    System.err.println("Before");
                    prevE.getTraceNumber()._traceBirthDay.printStackTrace();
                }
            }
            
            _list.set(_writePos, e);
            _writePos ++;
            _startPos = _writePos;
        }else {
            _list.add(e);
            _writePos ++;
        }
        fireRepaint();
    }
    
    boolean reserved = false;
    long lastTick = 0;

    public void fireRepaint() {
        long tickNow = System.currentTimeMillis();
        if (tickNow - lastTick < _timer) {
            if (reserved) {
                return;
            }
            reserved = true;
            MXTimer.letsCountdown(_timer - (tickNow - lastTick), new Runnable() {
                @Override
                public void run() {
                    fireImpl();
                }
            });
        }else {
            reserved = true;
            fireImpl();
        }
    }
    
    protected void fireImpl() {
        if (SwingUtilities.isEventDispatchThread() == false) {
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    fireImpl();
                }
            });
            return;
        }
        if (reserved) {
            lastTick = System.currentTimeMillis();
            reserved = false;
            synchronized(ConsoleModel.this) {
                final ListDataEvent e = new ListDataEvent(this, ListDataEvent.CONTENTS_CHANGED, 0, _list.size());
                try {
                    for (ListDataListener listener : _listener) {
                        listener.contentsChanged(e);
                    }
                    _refList.ensureIndexIsVisible(_capacity -1);
                    _refList.repaint();
                }catch(Throwable ex) {

                }
            };
        }
    }

    @Override
    public void addListDataListener(ListDataListener l) {
        _listener.add(l);
    }

    @Override
    public void removeListDataListener(ListDataListener l) {
        _listener.remove(l);
    }

    
    boolean reserved2 = false;
    long lastTick2 = 0;

    public void setSelectedTraceNumber(MXTraceNumber selection) {
        _selectedTraceNumber = selection;

        long tickNow = System.currentTimeMillis();
        if (tickNow - lastTick2 < 100) {
            if (reserved2) {
                return;
            }
            reserved2 = true;
            MXTimer.letsCountdown(100 - (tickNow - lastTick2), new Runnable() {
                @Override
                public void run() {
                    fireSelectTraceNumber(selection);
                }
            });
        }else {
            reserved2 = true;
            fireSelectTraceNumber(selection);
        }
    }

    public void fireSelectTraceNumber(MXTraceNumber selection) {
        if (SwingUtilities.isEventDispatchThread() == false) {
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    fireImpl();
                }
            });
            return;
        }
        if (selection == null) {
            _refList.repaint();
            return;
        }
        if (reserved2) {
            lastTick2 = System.currentTimeMillis();
            reserved2 = false;
            synchronized(ConsoleModel.this) {
                int low = 0;
                int high = getSize();
                while(low < high) {
                    int middle = (high + low) / 2;
                    ConsoleElement elem = getConsoleElement(middle);
                    if (elem == null) {
                        low = middle + 1;
                        continue;
                    }
                    MXTraceNumber middlesNumber = elem.getTraceNumber();
                    if (middlesNumber == null) {
                        low = middle + 1;
                        continue;
                    }
                    
                    int comp = middlesNumber.compareTo(selection);
                    if (comp == 0) {
                        low = high = middle;
                        break;
                    }
                    if (comp < 0) {
                        low = middle + 1;
                    }else {
                        high = middle - 1;
                    }
                }
                if (low != high) {
                    //該当しなかった
                    _refList.ensureIndexIsVisible(low);
                    _refList.repaint();
                    return;
                }
                while(low >= 0) {
                    ConsoleElement e1 = getConsoleElement(low);
                    ConsoleElement e2 = getConsoleElement(low - 1);
                    if (e1 == null || e2 == null) {
                        break;
                    }
                    MXTraceNumber t1 = e1.getTraceNumber();
                    MXTraceNumber t2 = e2.getTraceNumber();
                    if (t1 == null || t2 == null) {
                        break;
                    }
                    if (t1.compareTo(t2) == 0) {
                        low --;
                        continue;
                    }else {
                        break;
                    }
                }
                while(high + 1 < getSize()) {
                    ConsoleElement e1 = getConsoleElement(high);
                    ConsoleElement e2 = getConsoleElement(high + 1);
                    if (e1 == null || e2 == null) {
                        break;
                    }
                    MXTraceNumber t1 = e1.getTraceNumber();
                    MXTraceNumber t2 = e2.getTraceNumber();
                    if (t1 == null || t2 == null) {
                        break;
                    }
                    if (t1.compareTo(t2) == 0) {
                        high ++;
                        continue;
                    }else {
                        break;
                    }
                }
                _refList.ensureIndexIsVisible(low);
                _refList.repaint();
            };
        }
    }
}
