/*
 * Copyright 2023 Syntarou YOSHIDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.synthtarou.midimixer.mx20patchbay;

import javax.swing.JPanel;
import jp.synthtarou.midimixer.MXStatic;
import jp.synthtarou.midimixer.libs.MXDebugLines;
import jp.synthtarou.midimixer.libs.midi.MXMessage;
import jp.synthtarou.midimixer.libs.midi.MXMessageFactory;
import jp.synthtarou.midimixer.libs.midi.MXMidi;
import jp.synthtarou.midimixer.libs.midi.MXUtilMidi;
import jp.synthtarou.midimixer.libs.midi.MXReceiver;
import jp.synthtarou.midimixer.libs.midi.port.MXVisitant;
import jp.synthtarou.midimixer.libs.settings.MXSetting;
import jp.synthtarou.midimixer.libs.settings.MXSettingTarget;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MX20Process extends MXReceiver implements  MXSettingTarget {
    private static final MXDebugLines _debug = new MXDebugLines(MX20Process.class);
    MX20Data _data;
    private MX20View _view;
    private MXSetting _setting;

    private boolean _typeInput = false;
    
    public boolean isTypeInput() {
        return _typeInput;
    }

    public MX20Process(boolean typeInput) {
        _typeInput = typeInput;
        _data = new MX20Data();
        _view = new MX20View(this);
        if (typeInput) {
            _setting = new MXSetting("InputPortPatching");
        }else {
            _setting = new MXSetting("OutputPortPatching");
        }
        _setting.setTarget(this);
    }
    
    public void readSettings() {
        _setting.readFile();
        if (_setting.isEmpty()) {
            initStraight();
        }
        _view.setupViewFromModel();
    }
    
    public MX20Data getData() {
        return _data;
    }

    @Override
    public void processMXMessage(MXMessage message) {
        if (isUsingThisRecipe() == false) {
            sendToNext(message); 
            return;
        }

        int port = message.getPort();
        boolean proc = false;
        
        for (int j = 0; j < MXStatic.TOTAL_PORT_COUNT; ++ j) {
            if (_data.isEnabled(port, j)) {
                if (port == j) {
                    proc = true;
                    sendToNext(message);
                }else {
                    proc = true;
                    MXMessage message2 = MXMessageFactory.fromClone(message._traceNumber, message);
                    message2.setPort(j);
                    sendToNext(message2);
                }
            }
        }
    }

    public void initStraight() {
        for (int i = 0; i < MXStatic.TOTAL_PORT_COUNT; ++ i) {
            _data.set(i, i, true);
        }
        _view.setupViewFromModel();
    }

    public void initStraightAndAllLast() {
        for (int i = 0; i < MXStatic.TOTAL_PORT_COUNT; ++ i) {
            _data.set(i, i, true);
        }
        for (int i = 0; i < MXStatic.TOTAL_PORT_COUNT; ++ i) {
            _data.set(i, MXStatic.TOTAL_PORT_COUNT - 1, true);
        }
        _view.setupViewFromModel();
    }

    @Override
    public String getReceiverName() {
        if (_typeInput) {
            return "Input Patch";
        }else {
            return "Output Patch";
        }
    }

    @Override
    public JPanel getReceiverView() {
        return _view;
    }

    @Override
    public void prepareSettingFields(MXSetting setting) {
        for(int i = 0; i < MXStatic.TOTAL_PORT_COUNT; ++ i) {
            setting.register("From" + MXUtilMidi.nameOfPortShort(i));
        }
    }

    @Override
    public void afterReadSettingFile(MXSetting setting) {
        _data.clear();

        for(int port = 0; port < MXStatic.TOTAL_PORT_COUNT; ++ port) {
            String text = setting.getSetting("From" + MXUtilMidi.nameOfPortShort(port));
            if (text == null) {
                continue;
            }
            String[] array = text.split("[ ,]+");
            if (array.length == 0) {
                continue;
            }
            for (String elem : array) {
                try {
                   int to = Integer.parseInt(elem);
                   _data.set(port, to, true);
                }catch(NumberFormatException e) {
                }
            }
        }
    }

    @Override
    public void beforeWriteSettingFile(MXSetting setting) {
        setting.clearValue();
        for(int port = 0; port < MXStatic.TOTAL_PORT_COUNT; ++ port) {
            StringBuffer text = new StringBuffer();
            for (int to = 0; to < MXStatic.TOTAL_PORT_COUNT; ++ to) {
                if (_data.isEnabled(port, to)) {
                    if (text.length() != 0) {
                        text.append(",");
                    }
                    text.append(String.valueOf(to));
                }
            }
            setting.setSetting("From" + MXUtilMidi.nameOfPortShort(port), text.toString());
        }
    }
}
