/*
 * Copyright 2023 Syntarou YOSHIDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.synthtarou.midimixer.libs.midi.port;

import jp.synthtarou.midimixer.libs.midi.driver.MXMIDIDevice;
import jp.synthtarou.midimixer.libs.midi.driver.MXMIDIDriver;
import java.io.File;
import java.util.TreeMap;
import jp.synthtarou.midimixer.MXStatic;
import jp.synthtarou.midimixer.libs.MXWrapList;
import jp.synthtarou.midimixer.libs.midi.driver.MXMIDIDriverJava;
import jp.synthtarou.midimixer.libs.midi.driver.MXMIDIDriverUWP;
import jp.synthtarou.midimixer.libs.settings.MXSetting;
import jp.synthtarou.midimixer.libs.settings.MXSettingTarget;
import org.xml.sax.SAXException;

/**
 *
 * @author YOSHIDA Shintarou
 */

public class MXMIDIOutManager implements MXSettingTarget {
    private static final MXMIDIOutManager _instance = new MXMIDIOutManager();
 
    public static MXMIDIOutManager getManager() {
        return _instance;
    }

    public void reloadDeviceList() {
        //TODO Java not support?
    }

    private MXSetting _setting;

    public void initWithSetting() {
        if (_setting == null) {
            _setting = new MXSetting("MIDIOutput");
            _setting.setTarget(this);
            listAllOutput();
            _setting.readFile();
            
            MXMIDIOut reserve = MXMIDIOutManager.getManager().findMIDIOutput("Gervill");
            MXMIDIOut reserve2 = MXMIDIOutManager.getManager().findMIDIOutput("Microsoft GS Wavetable Synth");
            MXMIDIOut reserve3 = MXMIDIOutManager.getManager().findMIDIOutput("VirtualMIDISynth #1");
            
            if (reserve2 != null && reserve2.assignedPort() >= 0) {
                reserve = reserve2;
            }
            if (reserve3 != null && reserve3.assignedPort() >= 0) {
                reserve = reserve3;
            }
            if (reserve.assignedPort() < 0) {
                int found = -1;
                for (int i = 0; i < MXStatic.TOTAL_PORT_COUNT; ++ i) {
                    boolean retry = false;
                    for (MXMIDIOut out : listAllOutput().valueList()) {
                        if (out.assignedPort() == i) {
                            retry = true;
                            break;
                        }
                    }
                    if (!retry) {
                        found = i;
                        break;
                    }
                }
                if (found >= 0) {
                    reserveOutput(reserve, found);
                    reserve.open();
                }
            }
        }
    }

    protected MXMIDIOutManager() {
    }

    protected MXWrapList<MXMIDIOut> _listAllOutput;
    protected MXWrapList<MXMIDIOut> _selectedOutput = null;
    protected MXMIDIOut[] _cache;
    
    public synchronized MXWrapList<MXMIDIOut> listAllOutput() {
        if (_listAllOutput != null) {
            return _listAllOutput;
        }

        MXWrapList<MXMIDIOut> temp = new MXWrapList<MXMIDIOut>();

        MXMIDIDriver java = MXMIDIDriverJava._instance;
        for (int i = 0; i < java.OutputDevicesRoomSize(); i++) {
            MXMIDIDevice device = new MXMIDIDevice(java, false, i);

            String name = device.getName();
            if (MXMIDIDriverUWP._instance.isUsable()) {
                if (name.equals("Real Time Sequencer") || name.equals("Unknown name")) {
                    continue;
                }
                if (name.startsWith("Microsoft GS Wave")) {
                    continue;
                }
            }
            
            temp.addNameAndValue(name, new MXMIDIOut(name, device));
        }
        if (MXMIDIDriverUWP._instance.isUsable()) {
            MXMIDIDriver uwp = MXMIDIDriverUWP._instance;
            for (int i = 0; i < uwp.OutputDevicesRoomSize(); i++) {
                MXMIDIDevice device = new MXMIDIDevice(uwp, false, i);
                String name = device.getName();

                if (name.equals("Real Time Sequencer") || name.equals("Unknown name")) {
                    continue;
                }
                if (name.equals("MIDI")) {
                    continue;
                }
                if (temp.indexOfName(name) >= 0) {
                    continue;
                }
                name = name +" (UWP)";
                temp.addNameAndValue(name, new MXMIDIOut(name, device));
            }
        }                

        _listAllOutput = temp;
        return _listAllOutput;
    }

    public MXMIDIOut findMIDIOutput(String deviceName) {
        MXWrapList<MXMIDIOut> model = listAllOutput();
        return model.valueOfName(deviceName);
    }

    public MXMIDIOut findMIDIOutput(int assigned) {
        listSelectedOutput();
        if (assigned >= 0) {
            return _cache[assigned];
        }else {
            return null;
        }
    }

    synchronized void onClose(MXMIDIOut output) {
        clearMIDIOutCache();
    }
    
    protected synchronized void clearMIDIOutCache() {
        _selectedOutput = null;
        _cache = null;
    }

    public synchronized MXWrapList<MXMIDIOut> listSelectedOutput() {
        if (_selectedOutput != null) {
            return _selectedOutput;
        }
        _selectedOutput = new MXWrapList();
        _cache = new MXMIDIOut[MXStatic.TOTAL_PORT_COUNT];
        for (MXMIDIOut midi : listAllOutput().valueList()) {
            int port = midi.assignedPort();
            if (port >= 0) {
                _selectedOutput.addNameAndValue(midi.getName(), midi);
                _cache[port] = midi;
            }
        }
        return _selectedOutput;
    }
    
    public synchronized void closeAll() {
        for(MXMIDIOut output : listAllOutput().valueList()) {
            if (output.isOpen()) {
                output.close();
            }
        }
    }
    
    public synchronized boolean reserveOutput(MXMIDIOut output, int assignnew) {
        MXWrapList<MXMIDIOut> list = listAllOutput();

        if (output.assignedPort() >= 0) {
            if (output.assignedPort() == assignnew) {
                return true;
            }
        }

        if (assignnew < 0) {
            output.close();
            clearMIDIOutCache();
            output._assigned = assignnew;
            return true;
        }else {
            for (int i = 0; i < list.size(); ++ i) {
                MXMIDIOut x = list.valueOfIndex(i);
                if (x.assignedPort() == assignnew) {
                    x.close();
                    x._assigned = -1;
                }
            }
            output._assigned = assignnew;
            clearMIDIOutCache();
            return true;
        }
    }

    @Override
    public void prepareSettingFields(MXSetting setting) {
        setting.register("device[].name");
        setting.register("device[].open");
        setting.register("device[].fromDXML");
    }

    @Override
    public void afterReadSettingFile(MXSetting setting) {
        TreeMap<String, MXMIDIOut> dummy = new TreeMap();
        
        for (int x = 0; x < MXStatic.TOTAL_PORT_COUNT; ++ x) {
            String deviceName = setting.getSetting("device[" + x + "].name");
            String deviceOpen = setting.getSetting("device[" + x + "].open");
            String deviceFile = setting.getSetting("device[" + x + "].fromDXML");
            
            if (deviceName == null) {
                continue;
            }
            if (deviceOpen == null) {
                deviceOpen = "0";
            }
            
            MXWrapList<MXMIDIOut> detected = listAllOutput();
            MXMIDIOut out = detected.valueOfName(deviceName);
            if (out != null) {
                reserveOutput(out, x);
                if (deviceOpen.equals("1")) {
                    out.open();
                }
            }else {
                out  = new MXMIDIOut(deviceName, null);
                reserveOutput(out, x);
                dummy.put(deviceName, out);
            }
            if (deviceFile != null) {
                File f = new File(deviceFile);
                if (f.isFile()) {
                    try {   
                    out.setDXMLFile(f);
                    }catch(SAXException e) {
                        e.printStackTrace();
                    }
                }
            }
        }
        for (MXMIDIOut out : dummy.values()) {
            _listAllOutput.addNameAndValue(out.getName(),out);
        }
        clearMIDIOutCache();
    }

    @Override
    public void beforeWriteSettingFile(MXSetting setting) {
        MXWrapList<MXMIDIOut> all = listAllOutput();
        for (MXMIDIOut e : all.valueList()) {
            int x = e.assignedPort();
            if (x >= 0) {
                setting.setSetting("device[" + x + "].name", e.getName());
                setting.setSetting("device[" + x + "].open", e.isOpen() ? "1" : "0");
                setting.setSetting("device[" + x + "].fromDXML", e.getDXMLFile() == null ? "" : e.getDXMLFile().getPath());
            }
        }
    }
}
