/*
 * Copyright 2023 Syntarou YOSHIDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.synthtarou.midimixer.libs.midi;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import jp.synthtarou.midimixer.MXMain;
import jp.synthtarou.midimixer.libs.MXDebugLines;

/**
 *
 * @author YOSHIDA Shintarou
 */

public class MXNoteOffWatcher {
    private static final MXDebugLines _debug = new MXDebugLines(MXNoteOffWatcher.class);
    Object lock = MXMain.getMain();

    public static interface Handler {
        public void onNoteOffEvent(MXMessage target, MXTraceNumber  trace);
    }
    
    private static class Element {
        MXMessage caughtNoteOn;
        MXMessage sendingNoteOff;
        Handler listener;
    }

    private LinkedList<Element> _list = new LinkedList<Element>();

    public MXNoteOffWatcher() {
    }
    
    public String toString() {
        return "(MXNoteOff + " + _list.size();
    }
    
    public boolean addListener(MXMessage noteOn, MXMessage noteOff, Handler listener) {
        synchronized(lock) {
            if (noteOff == null) {
                noteOff = noteOn;
            }
            return addListener(noteOn, noteOff, listener, null);
        }
    }
    
    Handler _lastHandler;

    public boolean addListener(MXMessage noteOn, MXMessage noteOff, Handler listener, String dolog) {
        synchronized(lock) {
            if (noteOn.getCommand() != MXMidi.COMMAND_NOTEON) {
                _debug.println("Its not note on " + noteOn);
                return false;
            }
            _lastHandler = listener;
            Element e = new Element();
            e.caughtNoteOn = noteOn;
            e.sendingNoteOff = noteOff;
            e.listener = listener;
            /*
            if (dolog != null) {
                if (dolog.equals("@1")) {
                    System.out.println("Add Listener " + dolog + " " + noteOn + " ... " + noteOff);
                }
            }
            */
            _list.add(e);
            return true;
        }
    }
    
    public void allNoteOff(MXTraceNumber trace) {
        synchronized(lock) {
            ArrayList<Element> list = new ArrayList<>(_list);
            _list.clear();

            for (Element e : list) {
                MXMessage base = e.sendingNoteOff;
                MXMessage msg = MXMessageFactory.fromShortMessage(base.getPort(), trace, MXMidi.COMMAND_NOTEOFF + base.getChannel(), base.getGate(), 0);
                e.listener.onNoteOffEvent(msg, trace);
            }
        }
    }
    
    public boolean notifyNoteOffEvent(int port, MXTraceNumber traceNumber, int ch, int note) {
        synchronized(lock) {
            return notifyNoteOffEvent(port, traceNumber, ch, note, null);
        }
    }

    public boolean notifyNoteOffEvent(int port, MXTraceNumber traceNumber, int ch, int note, String dolog) {
        synchronized(lock) {
            boolean proc = false;
            boolean saveProc = false;

            do {
                proc = false;
                Iterator<Element> it = _list.iterator();
                while (it.hasNext()) {
                    Element e = it.next();
                    if (e.caughtNoteOn.getPort() == port
                     && e.caughtNoteOn.getChannel() == ch
                     && e.caughtNoteOn.getData1()== note) {
                        MXMessage noteOff = MXMessageFactory.fromShortMessage(port, traceNumber, MXMidi.COMMAND_NOTEOFF + ch, note, 0);
                        e.listener.onNoteOffEvent(noteOff, traceNumber);
                        it.remove();
                        proc = true;
                    }
                }
                if (proc) {
                    saveProc = true;
                }
            }while(proc);

            return saveProc;
        }
    }
}
