/*
 * Copyright 2023 Syntarou YOSHIDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.synthtarou.midimixer.libs;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 *
 * @author YOSHIDA Shintarou
 */
public class MXDOMElement {
    private static final MXDebugLines _debug = new MXDebugLines(MXDOMElement.class);

    MXDOMElement _parent;
    String _nodeName;
    String _text;
    List<String> _nodePath;
    MXWrapList<MXDOMElement> _childElements;
    MXWrapList<String> _attributes;

    public int _userDataType;
    public Object _userData;
    
    public MXDOMElement(MXDOMElement parent, Element element) {
        _parent = parent;
        _childElements = new MXWrapList();
        _attributes = new MXWrapList();
        _attributes.setIgnoreCase(true);
        _nodeName = element.getNodeName();
        ArrayList nodePath = new ArrayList();

        nodePath.add(element.getNodeName());
        Node seek = element.getParentNode();
        while (seek != null) {
            if (!seek.getNodeName().startsWith("#")) {
                nodePath.add(0, seek.getNodeName());
            }
            seek = seek.getParentNode();
        }
        _nodePath = Collections.unmodifiableList(nodePath);
        NodeList list = element.getChildNodes();
        for (int x = 0; x < list.getLength(); ++ x) {
            Node node = list.item(x);
            if (node.getNodeType() == Node.ELEMENT_NODE) {
                _childElements.addNameAndValue(node.getNodeName(), new MXDOMElement(this, (Element)node));
            }
            if (node.getNodeType() == Node.TEXT_NODE) {
                String text1 = shrinkSpace(node.getNodeValue());
                if (text1 != null) {
                    _text = text1;
                }
            }
        }
        NamedNodeMap fusion = element.getAttributes();
        for (int x = 0; x < fusion.getLength(); ++ x) {
            Node n = fusion.item(x);
            String name = n.getNodeName();
            String value = n.getNodeValue();
            value = MXDOMElement.shrinkSpace(value);
            if (value != null && value.length() == 0) {
                value = null;
            }
            _attributes.addNameAndValue(name, value);
        }
    }
    
    public String getNodeName() {
        return _nodeName;
    }

    public List<String> getNodePath() {
        return _nodePath;
    }

    public static MXDOMElement fromDocument(Document doc) {
        return new MXDOMElement(null, doc.getDocumentElement());
    }
    
    public static MXDOMElement fromFile(File file) throws SAXException {
        DocumentBuilderFactory factory;
        DocumentBuilder builder;
        Document document;
        factory = DocumentBuilderFactory.newInstance();
        try {
            builder = factory.newDocumentBuilder();
        } catch (ParserConfigurationException ex) {
            ex.printStackTrace();
            return null;
        }
        try {
            document = builder.parse(file);
        } catch (SAXException ex) {
            throw ex;
        } catch (IOException ex) {
            ex.printStackTrace();
            return null;
        }

        MXDOMElement docElement = MXDOMElement.fromDocument(document);
        return docElement;
    }
    
    public MXDOMElement getParentNode() {
        return _parent;
    }
    
    public List<MXDOMElement> getChildElements() {
        return _childElements.valueList();
    }

    public List<MXDOMElement> getChildElements(String name) {
        ArrayList<MXDOMElement> list = new ArrayList();
        for (MXWrap<MXDOMElement> child : _childElements) {
            if (name.equals(child.value.getNodeName())) {
                list.add(child.value);
            }
        }
        return list;
    }
    
    public String getText() {
        return _text;
    }

    public int countAttributes() {
        return _attributes.size();
    }
    
    public String getAttributeName(int x) {
        return _attributes.nameOfIndex(x);
    }

    public String getAttributeValue(int x) {
        return _attributes.valueOfIndex(x);
    }

    public String getAttributeValue(String name) {
        return _attributes.valueOfName(name);
    }
    
    public MXWrapList<String> getAttributesMap() {
        return _attributes;
    }

    public static String shrinkSpace(String original) {
        if (original == null) {
            return null;
        }
        StringBuffer text = new StringBuffer(original);
        
        while(text.length() > 0) {
            char c = text.charAt(0);
            if (c == ' ' || c == '\t' || c == '\r' || c == '\n') {
                text.deleteCharAt(0);
                continue;
            }
            c = text.charAt(text.length() - 1);
            if (c == ' ' || c == '\t' || c == '\r' || c == '\n') {
                text.deleteCharAt(text.length() - 1);
                continue;
            }
            break;
        }
        
        return text.toString();
    }
    
    public static void main(String[] args) throws Exception {
        PrintStream output = System.out;

        MXDebugLines.globalSwitchOn();

        MXDOMElement rootElement = MXDOMElement.fromFile(new File("C:/Domino144/Module/GMLevel1.xml"));
        MXDOMElement docElement = rootElement;

        ArrayList<MXDOMElement> process = new ArrayList();
        process.add(docElement);
        

        if (docElement.countAttributes() > 0) {
            output.print("{");
            for (int i = 0; i < docElement.countAttributes(); ++ i) {
                String name = docElement.getAttributeName(i);
                String value = docElement.getAttributeValue(i);
                if (i > 0) {
                    output.print(",");
                }
                output.print(name + "=" + value);
            }
            output.print("}");
        }

        while (process.size() > 0) {
            MXDOMElement element = process.remove(process.size() - 1);
            
            String nodeName = element.getNodeName();
            String text = element.getText();
            List<MXDOMElement> child = element.getChildElements();
            List<String> path = element.getNodePath();

            output.print("<" + path + ">");
            if (text != null && text.length() > 0) {
                output.print("text = " +  text);
            }
            if (element.countAttributes() > 0) {
                output.print("{");
                for (int i = 0; i < element.countAttributes(); ++ i) {
                    String name = element.getAttributeName(i);
                    String value = element.getAttributeValue(i);
                    if (i > 0) {
                        output.print(",");
                    }
                    output.print(name + "=" + value);
                }
                output.print("}");
            }
            _debug.println();
            
            if (child.size() > 0) {
                for (int x = child.size() - 1; x >= 0; x --) {                    
                    process.add(child.get(x));
                }
            }
        }

        process.clear();
        process.add(docElement);

        while(process.size() > 0) {
            MXDOMElement e = process.remove(0);
            _debug.println("-" + e.getNodeName());
            List<MXDOMElement> instList = e.getChildElements("InstrumentList");

            while(instList.size() > 0) {
                MXDOMElement inst = instList.remove(0);
                List<MXDOMElement> mapList = inst.getChildElements("Map");
                
                while(mapList.size() > 0) {
                    MXDOMElement map = mapList.remove(0);

                    List<MXDOMElement> listPC = map.getChildElements("PC");
                    while(listPC.size() > 0) {
                        MXDOMElement pc = listPC.remove(0);
                        _debug.println(pc.getNodeName() + "=" + pc.getText() + pc.getAttributesMap());

                        List<MXDOMElement> listBank = pc.getChildElements("Bank");
                        while(listBank.size() > 0) {
                            MXDOMElement bank = listBank.remove(0);
                            _debug.println(bank.getNodeName() + "=" + bank.getText() + bank.getAttributesMap());
                        }
                    }
                }
            }
        }
    }
    
    public void dump(PrintStream output) {
        ArrayList<MXDOMElement> process = new ArrayList();
        MXDOMElement e = this;
        process.add(e);

        if (e.countAttributes() > 0) {
            output.print("{");
            for (int i = 0; i < e.countAttributes(); ++ i) {
                String name = e.getAttributeName(i);
                String value = e.getAttributeValue(i);
                if (i > 0) {
                    output.print(",");
                }
                output.print(name + "=" + value);
            }
            output.print("}");
        }

        while (process.size() > 0) {
            MXDOMElement element = process.remove(process.size() - 1);
            
            String nodeName = element.getNodeName();
            String text = element.getText();
            List<MXDOMElement> child = element.getChildElements();
            List<String> path = element.getNodePath();

            output.print("<" + path + ">");
            if (text != null && text.length() > 0) {
                output.print("text = " +  text);
            }
            if (element.countAttributes() > 0) {
                output.print("{");
                for (int i = 0; i < element.countAttributes(); ++ i) {
                    String name = element.getAttributeName(i);
                    String value = element.getAttributeValue(i);
                    if (i > 0) {
                        output.print(",");
                    }
                    output.print(name + "=" + value);
                }
                output.print("}");
            }
            _debug.println();
            
            if (child.size() > 0) {
                for (int x = child.size() - 1; x >= 0; x --) {                    
                    process.add(child.get(x));
                }
            }
        }
        _debug.println("-------------------------");

        process.clear();
        process.add(e);

        while(process.size() > 0) {
            e = process.remove(0);
            _debug.println("-" + e.getNodeName());
            List<MXDOMElement> instList = e.getChildElements("InstrumentList");

            while(instList.size() > 0) {
                MXDOMElement inst = instList.remove(0);
                List<MXDOMElement> mapList = inst.getChildElements("Map");
                
                while(mapList.size() > 0) {
                    MXDOMElement map = mapList.remove(0);

                    List<MXDOMElement> listPC = map.getChildElements("PC");
                    while(listPC.size() > 0) {
                        MXDOMElement pc = listPC.remove(0);
                        _debug.println(pc.getNodeName() + "=[" + pc.getText() + "]" + pc.getAttributesMap());

                        List<MXDOMElement> listBank = pc.getChildElements("Bank");
                        while(listBank.size() > 0) {
                            MXDOMElement bank = listBank.remove(0);
                            _debug.println("    "  + bank.getNodeName() + "=[" + bank.getText() + "]" + bank.getAttributesMap());
                        }
                    }
                }
            }
        }
    }

    public String toString() {
        return _nodePath.toString();
    }
}
