package jp.sourceforge.masme.validation.impl;

import jp.sourceforge.masme.MasmenValidateException;
import jp.sourceforge.masme.target.MaintenanceProperty;
import jp.sourceforge.masme.target.MaintenanceWorkTarget;
import jp.sourceforge.masme.target.ReferenceProperty;
import jp.sourceforge.masme.target.ValidateProperty;
import jp.sourceforge.masme.validation.MasmenValidation;
import jp.sourceforge.masme.validation.ValidationResult;


import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * MasmenValidationの実装クラス
 */
public class MasmenValidationImpl implements MasmenValidation {

    static Log log = LogFactory.getLog(MasmenValidationImpl.class);

    public void edit(MaintenanceWorkTarget target) {
        log.debug("validate [" + target + "] (edit)");
        workValidation(target);
    }

    public void cancelEdit(MaintenanceWorkTarget target) {
        // TODO 物理削除の際の検討
        log.debug("validate [" + target + "] (cancel edit)");
        workValidation(target);
    }

    /**
     * エンティティの反映状態に関して整合性を保っているかどうかを検証します。
     * この検証はワークエンティティ確認時や確認取消時に必要な検証です。
     * 検証内容は以下の通りです。
     * 
     * <h5>参照先のエンティティの整合性検証</h5>
     * （確認対象である）エンティティAが別のエンティティBを参照している場合には、
     * エンティティAの反映時にはエンティティBの本番情報が存在し
     * 特定の整合性条件を満たしている必要があります。
     * そのような状態を実現するためには、
     * エンティティAを確認時に、エンティティBが
     * 以下のいずれかの状態である必要があります。
     * <ul>
     * <li>エンティティBのワークが確認済状態にあり、
     * エンティティBの反映日 < エンティティAの反映日
     * （確認処理においてエンティティBのワークが条件を満たしていないことはありえない）
     * <li>エンティティBの本番が、エンティティAに必要な条件を満たしている 
     * </ul>
     * この検証はエンティティAが削除対象でない場合のみ行います。
     * 
     * <h5>被参照先のエンティティの整合性検証</h5>
     * （確認対象である）エンティティAが別のエンティティBを参照されている場合には、
     * エンティティBの反映時にはエンティティAの本番情報が存在し
     * 特定の整合性条件を満たしている必要があります。
     * そのような上状態を実現するためには、
     * エンティティAが削除されるまたは条件を満たさなくなる修正の場合の確認時に、
     * 以下のいずれかの状態である必要があります。
     * <ul>
     * <li>エンティティAを参照するエンティティBの本番が存在しないこと
     * （確認処理においてエンティティBのワークがエンティティAを参照することはありえない）
     * <li>エンティティAを参照するエンティティBの本番がある場合には、
     * （修正または削除されたエンティティBのワークが）
     * 確認済状態でエンティティBの反映日 < エンティティAの反映日
     * </ul>
     * 
     * @param target
     */
    public void confirm(MaintenanceWorkTarget target) {
        // 削除対象でない場合には、このオブジェクトの関連先があるかどうかのチェック
        // if (!target.getWorkInfo().getDeleteFlag()) {
        waitForValidation(target);
        // }

    }

    public void cancelConfirm(MaintenanceWorkTarget target) {
        // 削除対象であった場合には、このオブジェクトの関連先があるかどうかのチェック
        // if (target.getWorkInfo().getDeleteFlag()) {
        waitForValidation(target);
        // }

    }

    protected void waitForValidation(MaintenanceWorkTarget target) {
        for (MaintenanceProperty mp : target.getMaintenanceProperties()) {
            if (mp instanceof ReferenceProperty) {
                ReferenceProperty vmp = (ReferenceProperty) mp;
                // invoke
                ValidationResult resultWork = vmp.invokeWorkValidateMethod();
                ValidationResult resultReal = vmp.invokeRealValidateMethod();
                if (log.isDebugEnabled()) {
                    log.debug("Work validation result:" + resultWork);
                    log.debug("Real validation result:" + resultReal);
                }
                if (resultReal.validationSuccess()
                        && resultWork.validationSuccess()) {
                    // real ok and work ok -> ok
                    continue;
                } else if (resultReal.validationSuccess()) {
                    // real ok and work ng -> then check work is not wair
                    // reflect
                    if (!resultWork.validateNotWaitReflect()) {
                        String errorMessage = "error is occured ";
                        errorMessage += "with work message ["
                                + resultWork.getErrorMessage() + "]";
                        throw new MasmenValidateException(errorMessage);
                    }
                } else if (resultWork.validationSuccess()) {
                    // real ng and work ok -> then check work will be reflect
                    if (!resultReal.validateReflectFirst()) {
                        String errorMessage = "error is occured ";
                        errorMessage += "with real message ["
                                + resultReal.getErrorMessage() + "]";
                        throw new MasmenValidateException(errorMessage);
                    }
                } else {
                    // real ng and work ng -> ng
                    String errorMessage = "error is occured ";
                    errorMessage += "with work message ["
                            + resultWork.getErrorMessage() + "] ";
                    errorMessage += "with real message ["
                            + resultReal.getErrorMessage() + "]";
                    throw new MasmenValidateException(errorMessage);
                }

            }
        }
        // このオブジェクトが保持するValidateMethodが条件を満たすか確認する
        for (ValidateProperty vp : target.getValidateProperties()) {
            // invoke
            ValidationResult resultWork = vp.invokeWorkValidateMethod(target);
            ValidationResult resultReal = vp.invokeRealValidateMethod(target);

            if (log.isDebugEnabled()) {
                log.debug("Work validation result:" + resultWork);
                log.debug("Real validation result:" + resultReal);
            }
            if (resultWork.validationSuccess()
                    && resultReal.validationSuccess()) {
                // real ok and work ok -> ok
                continue;
            } else if (resultReal.validationSuccess()) {
                if (!resultWork.validateReflectFirst()) {
                    String errorMessage = "error is occured ";
                    errorMessage += "with work message ["
                            + resultWork.getErrorMessage() + "]";
                    throw new MasmenValidateException(errorMessage);
                }
            } else if (resultWork.validationSuccess()) {
                if (!resultReal.validateNotWaitReflect()) {
                    String errorMessage = "error is occured ";
                    errorMessage += "with real message ["
                            + resultReal.getErrorMessage() + "]";
                    throw new MasmenValidateException(errorMessage);
                }
            } else {
                // real ng and work ng -> ng
                String errorMessage = "error is occured ";
                errorMessage += "with work message ["
                        + resultWork.getErrorMessage() + "] ";
                errorMessage += "with real message ["
                        + resultReal.getErrorMessage() + "]";
                throw new MasmenValidateException(errorMessage);
            }
        }
    }

    /**
     * ワークエンティティが整合性を保っているかどうか検証します。
     * この検証はワークエンティティ修正時や修正取消時に必要な検証です。
     * 検証内容は以下の通りです。
     * <h5>参照先のエンティティの整合性検証</h5>
     * （変更後の）ワークエンティティが参照するエンティティに対して
     * 整合性を保っている状態かどうかを検証します。
     * <p>
     * これは従業員の情報を変更し、特定の店舗を割り当てる際の以下の検証になります。
     * <ul>
     * <li>割り当てる店舗が存在するか
     * <li>割り当てる店舗の営業開始日 < 従業員の就業開始日
     * </ul>
     * この検証はワークエンティティが削除対象でない場合に
     * （削除フラグをON時、登録取消時）にのみ行います。
     * <h5>被参照先のエンティティの整合性検証</h5>
     * （変更後の）ワークエンティティが参照されているエンティティに対して
     * 整合性を保っている状態かどうかを検証します。
     * <p>
     * これは店舗の情報を変更した際に、その店舗を参照している全ての従業員を対象に
     * 以下の検証を行うことになります。
     * <ul>
     * <li>店舗が削除される場合には、参照しているすべての従業員も削除対象になっているか
     * <li>変更後の店舗の営業開始日 < 全ての従業員の就業開始日
     * </ul>
     * @param target ワークエンティティ
     */
    protected void workValidation(MaintenanceWorkTarget target) {
        // 削除対象でない場合には、このオブジェクトの関連先があるかどうかのチェック
        if (!target.getWorkInfo().getDeleteFlag()) {
            for (MaintenanceProperty mp : target.getMaintenanceProperties()) {
                if (mp instanceof ReferenceProperty) {
                    ReferenceProperty vmp = (ReferenceProperty) mp;
                    ValidationResult result = vmp.invokeWorkValidateMethod();
                    log.debug(result);
                    if (!result.validationSuccess()) {
                        throw new MasmenValidateException(result
                                .getErrorMessage());
                    }
                }
            }
        }
        // このオブジェクトが保持するValidateMethodが条件を満たすか確認する
        for (ValidateProperty vp : target.getValidateProperties()) {
            ValidationResult result = vp.invokeWorkValidateMethod(target);
            // 条件を満たさなくなった場合には、参照するオブジェクトがあるか確認する
            if (!result.validationSuccess() && vp.isNonDeletedProperty()) {
                throw new MasmenValidateException("関連先" + vp + "があるため"
                        + result.getErrorMessage() + "を満たせない状態は不正です");
            }
        }
    }
}
