package jp.sourceforge.masme.util.impl;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import jp.sourceforge.masme.entity.MaintenanceEntity;
import jp.sourceforge.masme.entity.RealEntity;
import jp.sourceforge.masme.entity.WorkEntity;
import jp.sourceforge.masme.entity.WorkInfo;
import jp.sourceforge.masme.meta.MetaClass;
import jp.sourceforge.masme.meta.MetaProperty;
import jp.sourceforge.masme.meta.ReferenceMetaProperty;

import org.apache.commons.beanutils.PropertyUtils;

/**
 * マスタメンテナンスに関連するエンティティの操作を行うユーティリティクラスです
 */
public class MasmenEntityUtils {

    /**
     * プロパティを取得します.
     */
    public static Object getProperty(MaintenanceEntity entity,
            String propertyName) {
        try {
            return PropertyUtils.getProperty(entity, propertyName);
        } catch (Exception e) {
            throw new IllegalArgumentException("getting property ["
                    + propertyName + "] from [" + entity + "]", e);
        }
    }

    /**
     * プロパティを設定します
     */
    public static void setProperty(MaintenanceEntity entity,
            String propertyName, Object value) {
        try {
            PropertyUtils.setProperty(entity, propertyName, value);
        } catch (Exception e) {
            throw new IllegalArgumentException(e);
        }

    }

    /**
     * メンテナンス対象のプロパティのコピーをします
     */
    public static void replaceProperties(MaintenanceEntity from,
            MaintenanceEntity to, MetaClass metaClass) {
        for (MetaProperty mp : metaClass.getMetaProperties()) {
            if (mp instanceof ReferenceMetaProperty) {
                ReferenceMetaProperty rmp = (ReferenceMetaProperty) mp;
                // 本番についての入れ替え
                Object value = getProperty(from, rmp.getPropertyName());
                setProperty(to, rmp.getPropertyName(), value);
                // ワークについての入れ替え
                if (to instanceof WorkEntity) {
                    if (from instanceof RealEntity) {
                        RealEntity realValue = (RealEntity) getProperty(from,
                                rmp.getPropertyName());
                        value = realValue.getWorkEntity();
                    } else {
                        value = getProperty(from, rmp.getPropertyName());
                    }
                    setProperty(to, rmp.getWorkPropertyName(), value);
                }
            } else {
                Object value = getProperty(from, mp.getPropertyName());
                setProperty(to, mp.getPropertyName(), value);
            }
        }
    }

    /**
     * 検証用メソッドを検索するメソッド
     * <p>
     * 対象クラスのメソッド名に合致するMethodを探し返します。
     * Methodを探すために、引数としてparameterClazz型があればそのMethodを 無ければ引数なしのMethodを返します。
     * それでもなければ例外をスローします。
     * 
     * @param clazz
     * @param methodName
     * @param parameterClazz
     * @return
     */
    public static Method findValidationMethod(Class clazz, String methodName,
            Class parameterClazz) {

        Method method = null;
        for (Method m : clazz.getMethods()) {
            if (m.getName().equals(methodName)) {
                if (m.getParameterTypes().length == 0) {
                    if (method == null) {
                        method = m;
                    }
                } else if (m.getParameterTypes().length == 1) {
                    if (m.getParameterTypes()[0]
                            .isAssignableFrom(parameterClazz)) {
                        method = m;
                    }
                }
            }
        }
        if (method == null) {
            throw new IllegalArgumentException(new NoSuchMethodException(clazz
                    .getName()
                    + "." + methodName));
        }
        return method;
    }

    /**
     * 検証用メソッドを実行します。
     * <p>
     * Methodの引数がなければ、引数なしで実行します。 そうでなければ、parameterを引数として実行します。
     * 
     * @param method
     * @param target
     * @param parameter
     */
    public static void invokeValidationMethod(Method method, Object target,
            Object parameter) {
        try {
            if (method.getParameterTypes().length == 0) {
                method.invoke(target);
            } else {
                method.invoke(target, new Object[]{parameter});
            }
        } catch (IllegalArgumentException e) {
            throw new IllegalArgumentException("method Error Method[" + method
                    + "],target[" + target + "],parameter[" + parameter + "]",
                    e);
        } catch (IllegalAccessException e) {
            throw new IllegalArgumentException(e);
        } catch (InvocationTargetException e) {
            if (e.getTargetException() instanceof RuntimeException) {
                RuntimeException re = (RuntimeException) e.getTargetException();
                throw re;
            }
            throw new IllegalArgumentException(e);
        }
    }

    public static WorkInfo getWorkInfo(MaintenanceEntity entity) {
        if (entity == null) {
            return null;

        } else if (entity instanceof WorkEntity) {
            WorkEntity workEntity = (WorkEntity) entity;
            return workEntity.getWorkInfo();
        } else {
            RealEntity realEntity = (RealEntity) entity;
            return realEntity.getWorkEntity().getWorkInfo();
        }
    }
}
