package jp.sourceforge.masme.meta;

import java.util.List;

import jp.sourceforge.masme.annotation.MaintenanceTargetClass;
import jp.sourceforge.masme.validation.ValidateFunction;
import jp.sourceforge.masme.validation.ValidateFunctionFactory;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.InitializingBean;

/**
 * アノテーションをもとにメタ情報格納クラスを作成するFactoryBean
 */
public class AnnotationMetaInfoFactoryBean implements FactoryBean,
        InitializingBean {
    Log log = LogFactory.getLog(AnnotationMetaInfoFactoryBean.class);

    private List<String> annotatedMaintenanceClassNames;

    private MasmenMetaInfo metaInfo;

    private ValidateFunctionFactory validateFunctionFactory;

    /**
     * MasmenMetaInfoを作成します
     * <p>
     * MaintenanceTargetClassをメンテナンス対象クラスとします
     * 
     * @see org.springframework.beans.factory.InitializingBean#afterPropertiesSet()
     */
    public void afterPropertiesSet() throws Exception {
        log.debug("create MasmenMetaInfo using annotation.");
        metaInfo = new MasmenMetaInfo();
        for (String className : annotatedMaintenanceClassNames) {
            log.debug("create meta info for" + className);
            // MetaClassの作成
            Class<?> realClass = Class.forName(className);
            MaintenanceTargetClass anno = realClass
                    .getAnnotation(MaintenanceTargetClass.class);
            if (anno == null) {
                throw new IllegalArgumentException(className
                        + "はMaintenanceTargetClassが注釈されてません");
            }
            Class workClass = anno.workClass();
            String idName = MasmenAnnotationUtils.getIdPropertyName(realClass);
            if (idName == null) {
            	throw new IllegalArgumentException(className + "はIdが指定されていません");
            }
            MetaClass metaClass = new MetaClass(workClass, realClass, idName);
            metaInfo.addMetaClass(metaClass);
            // MetaPropertyの作成
            List<MetaProperty> metaProperties = MasmenAnnotationUtils
                    .getMaintenanceProperties(metaClass);
            for (MetaProperty metaProperty : metaProperties) {
                metaClass.addMetaProperty(metaProperty);
            }
        }
        // 参照情報の設定
        for (MetaClass metaClass : metaInfo.getMetaClasses()) {
            for (MetaProperty metaProperty : metaClass.getMetaProperties()) {
                if (metaProperty instanceof ReferenceMetaProperty) {
                    log.debug("create ref info " + metaProperty);
                    ReferenceMetaProperty refMetaProperty = (ReferenceMetaProperty) metaProperty;
                    MetaClass refMetaClass = metaInfo
                            .findMetaClassByClass(metaProperty
                                    .getPropertyType());
                    if (refMetaClass == null) {
                        throw new IllegalArgumentException(metaProperty
                                .getPropertyType()
                                + "は参照先として正しくありません");
                    }
                    refMetaProperty.setReferenceMetaClass(refMetaClass);
                    ValidateFunction validateFunction = validateFunctionFactory
                            .createValidateFunction(refMetaProperty);
                    refMetaProperty.setValidateFunction(validateFunction);
                }
            }
        }
        log.debug("end creating meta info.");
    }

    /**
     * @see org.springframework.beans.factory.FactoryBean#getObject()
     */
    public Object getObject() throws Exception {
        return metaInfo;
    }

    /**
     * @see org.springframework.beans.factory.FactoryBean#getObjectType()
     */
    public Class getObjectType() {
        return MasmenMetaInfo.class;
    }

    /**
     * @see org.springframework.beans.factory.FactoryBean#isSingleton()
     */
    public boolean isSingleton() {
        return false;
    }

    /**
     * @param annotatedMaintenanceClassNames annotatedMaintenanceClassNames を設定。
     */
    public void setAnnotatedMaintenanceClassNames(
            List<String> annotatedMaintenanceClassNames) {
        this.annotatedMaintenanceClassNames = annotatedMaintenanceClassNames;
    }

    /**
     * @param validateFunctionFactory validateFunctionFactory を設定。
     */
    public void setValidateFunctionFactory(
            ValidateFunctionFactory validateFunctionFactory) {
        this.validateFunctionFactory = validateFunctionFactory;
    }

}
