/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.m4hv.extensions.validator;

import javax.validation.ConstraintValidator;
import javax.validation.ConstraintValidatorContext;

import org.maru.m4hv.extensions.constraints.Hour;

import static org.maru.m4hv.extensions.util.Condition.isLessThan;
import static org.maru.m4hv.extensions.util.Condition.checkIncludedCorrectRange;
import static org.maru.m4hv.extensions.util.Condition.isIncludedInRange;

/**
 * Hour validator<br>
 *
 * This validator validate if given value is correct hour. Hour is represented 0 to 23.
 * By default maximum and minimum hours are defined on attributes of {@code Hour} annotation,
 * from,until. Both attributes are actually the integer type and for the from attribute you can
 * define 0 to 23 and until so on. If you define any number out of the hour range, the runtime
 * exception is thrown.
 *
 */
public class HourValidator implements ConstraintValidator<Hour, Object> {

    private int startHour;

    private int endHour;

    private static final int MIN_HOUR = 0;

    private static final int MAX_HOUR = 23;

    public void initialize(Hour hour) {
        startHour = checkIncludedCorrectRange(hour.from(), MIN_HOUR, MAX_HOUR);
        endHour = checkIncludedCorrectRange(hour.until(), MIN_HOUR, MAX_HOUR);
        if (!isLessThan(startHour, endHour)) {
            throw new IllegalArgumentException("The start hour [" + startHour +
                    "] must be less than the end hour [" + endHour + "]");
        }

    }

    /**
     * Validates if the given object is actual hour and included in arbitarary defined ranges
     */
    public boolean isValid(Object object, ConstraintValidatorContext constraintContext) {
        if (object == null) {
            return true;
        }

        Integer value;
        if (object instanceof String) {
            value = Integer.valueOf((String)object);
        } else if (object instanceof Integer) {
            value = (Integer)object;
        } else if (object instanceof Byte) {
            Byte byteValue = ((Byte) object).byteValue();
            value = byteValue.intValue();
        } else {
            throw new IllegalArgumentException("Unexpected type : " + object.getClass().getName());
        }
        return isIncludedInRange(value, startHour, endHour);
    }

}
