/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.m4hv.extensions.validator;

import static org.maru.m4hv.extensions.util.DateUtil.canonicalize;

import java.util.Date;

import javax.validation.ConstraintValidator;
import javax.validation.ConstraintValidatorContext;

import org.maru.m4hv.extensions.constraints.ActualDate;


/**
 * Actual Date(Year Month Day) validator<br>
 *
 * This validator validates the date accuracy and the range of date.
 * You can define the date range using attributes of from and until on {@code ActualDate} annotation.
 * The date range defaults from 1970/01/01 GMT. The 'until' attribute on {@code ActualDate} annotation is
 * by default the empty string. There are four types of default date pattern that we have already defined.
 * See below discription about default time pattern.<br>
 *
 * <pre>
 *    1. yyyy/MM/dd
 *       The date string format is like 2011/06/06.
 *    2. yyyy-MM-dd
 *       The date string format is like 2011-06-06.
 *    3. yyyy.MM.dd
 *       The date string format is like 2011.06.06.
 *    4.yyyyMMdd
 *       The date string format is like 20110606.
 * </pre>
 *
 * If you would like to add or redefine time patterns yourselves, just override the attribute of
 * patterns on {@code ActualDate} annotation. You can define your own pattern by following the format
 * that simple date format class supports.
 */
public class ActualDateValidator implements ConstraintValidator<ActualDate, String> {

    private Date startDate;

    private Date endDate;

    private String[] patterns;

    private static final String EXCEPTION_MESSAGE = "Unexpected date : ";

    public void initialize(ActualDate actualDate) {
        patterns = actualDate.patterns();
        String start = actualDate.from();
        startDate = canonicalize(start, patterns, EXCEPTION_MESSAGE);
        String end = actualDate.until();
        if (end != null && !end.equals("")) {
            endDate = canonicalize(end, patterns, EXCEPTION_MESSAGE);
        } else {
            endDate = null;
        }

    }

    /**
     * Validate date passed as a first parameter whether to be included in start
     * and end date defined {@code ActualDate} annotaton.
     */
    public boolean isValid(String date, ConstraintValidatorContext constraintContext) {
        if (date == null) {
            return true;
        }

        Date inputDate;
        try {
            inputDate = canonicalize(date, patterns, EXCEPTION_MESSAGE);
        } catch (IllegalArgumentException e) {
            // If the string date is not successfully canonicalized, it is not the actual date.
            return false;
        }
        if (endDate != null) {
            if (inputDate.compareTo(startDate) < 0 || inputDate.compareTo(endDate) > 0) {
                return false;
            } else {
                return true;
            }
        } else {
            if (inputDate.compareTo(startDate) < 0) {
                return false;
            } else {
                return true;
            }
        }

    }

}
