package util;

import java.util.AbstractSet;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * 重複を許さないヒープの実装です。
 * @author ma38su
 * @param <E> ヒープの要素のクラス
 */
public class HeapSet<E> extends AbstractSet<E> {

	/**
	 * ソートされるオブジェクト
	 */
	private E[] entries;

	/**
	 * ヒープ長
	 */
	private int size;

	private Map<E, Integer> table;
	
	private final Comparator<E> comparator;
	
	/**
	 * コンストラクタ
	 *
	 */
	public HeapSet () {
		this(10, null);
	}
	/**
	 * コンストラクタ
	 * @param initialCapacity 初期容量
	 */
	public HeapSet (int initialCapacity) {
		this(initialCapacity, null);
	}
	/**
	 * コンストラクタ
	 * @param comparator
	 */
	public HeapSet (Comparator<E> comparator) {
		this(10, comparator);
	}
	/**
	 * コンストラクタ
	 * @param initialCapacity 初期容量
	 * @param comparator
	 */
	@SuppressWarnings("unchecked")
	public HeapSet (int initialCapacity, Comparator<E> comparator) {
		if (initialCapacity < 1) {
			throw new IllegalArgumentException();
		}
		this.size = 0;
		this.entries = (E[]) new Object[initialCapacity + 1];
		this.table = new HashMap<E, Integer>();
		this.comparator = comparator;
	}
	
	/**
	 * 挿入する
	 * @param key
	 * @param value
	 * @return 挿入すればtrue
	 */
	@Override
	@SuppressWarnings("unchecked")
	public boolean add(E key) {
		E entry = key;
		if (this.table.containsKey(key)) {
			int index = this.table.get(key);
			this.entries[index] = entry;
			if(this.comparator == null) {
				if(((Comparable<E>) this.entries[index]).compareTo(entry) > 0) {
					this.fixUp(index);
				} else {
					this.fixDown(index);
				}
			}else{
				if (this.comparator.compare(this.entries[index], entry) > 0) {
					this.fixUp(index);
				} else {
					this.fixDown(index);
				}
			}
		} else {
			this.grow(++this.size);
			this.table.put(key, this.size);
			this.entries[this.size] = entry;
			this.fixUp(this.size);
		}
		return true;
	}

	/**
	 * 入れ替える
	 * @param index1
	 * @param index2
	 */
	private void swap(int index1, int index2) {
		E tmp = this.entries[index1];
		this.entries[index1] = this.entries[index2];
		this.entries[index2] = tmp;
		this.table.put(this.entries[index1], index1);
		this.table.put(this.entries[index2], index2);
	}

	/**
	 * 根を取り出す
	 * @return 根
	 */
	public E poll() {
		if (this.size == 0) {
			return null;
		}
		E entry = this.entries[1];
		this.table.remove(entry);
		if(this.size > 1) {
			this.entries[1] = this.entries[this.size];
			this.table.put(this.entries[1], 1);
		}
		this.entries[this.size] = null; // Drop extra ref to prevent memory leak
		if(--this.size > 1) {
			this.fixDown(1);
		}
		return entry;
	}
	
	/**
	 * 根を削除せずに取り出す
	 * @return 根
	 */
	public E peek() {
		return this.entries[1];
	}
	
	/**
	 * キーから値を取得
	 * @param key 
	 * @return 値
	 */
	public E get(Object key){
		return this.entries[this.table.get(key)];
	}

	/**
	 * @param key 
	 * @return キーが含まれていればtrue
	 */
	public boolean containsKey(Object key) {
		return this.table.containsKey(key);
	}

	public Collection<E> entrySet() {
		Collection<E> collection = new ArrayList<E>();
		while(!this.isEmpty()) {
			collection.add(this.poll());
		}
		return collection;
	}

	/**
	 * 子との状態の比較
	 * @param index
	 */
	@SuppressWarnings("unchecked")
	private void fixDown(int index) {
		int son;
		if(this.comparator == null) {
			while((son = index << 1) <= this.size) {
				if (son < this.size && ((Comparable<E>) this.entries[son]).compareTo(this.entries[son+1]) > 0) {
					son++;
				}
				if(((Comparable<E>) this.entries[index]).compareTo(this.entries[son]) <= 0) {
					break;
				}
				this.swap(index, son);
				index = son;
			}
		}else{
			while((son = index << 1) <= this.size) {
				if (son < this.size && this.comparator.compare(this.entries[son], this.entries[son+1]) > 0) {
					son++;
				}
				if (this.comparator.compare(this.entries[index], this.entries[son]) <= 0) {
					break;
				}
				this.swap(index, son);
				index = son;
			}
		}
	}
	/**
	 * 親との状態を確認
	 * @param index
	 */
	@SuppressWarnings("unchecked")
	private void fixUp(int index) {
		int parent;
		if(this.comparator == null) {
			while((parent = index >> 1) > 0) {
				if(((Comparable<E>) this.entries[index]).compareTo(this.entries[parent]) >= 0) {
					break;
				}
				this.swap(index, parent);
				index = parent;
			}
		}else{
			while((parent = index >> 1) > 0) {
				if(this.comparator.compare(this.entries[index], this.entries[parent]) >= 0) {
					break;
				}
				this.swap(index, parent);
				index = parent;
			}
		}
	}

	/**
	 * Resize Arrays
	 * @param index
	 */
	@SuppressWarnings("unchecked")
	private void grow(int index) {
		int newLength = this.entries.length;
		if (index < newLength) {
			// don't need to grow
			return;
		}
		if (index == Integer.MAX_VALUE) {
			throw new OutOfMemoryError();
		}
		while (newLength <= index) {
			if (newLength >= Integer.MAX_VALUE / 2) {
				newLength = Integer.MAX_VALUE;
			} else {
				newLength <<= 2;
			}
		}
		E[] newEntrys = (E[]) new Object[newLength];
		System.arraycopy(this.entries, 0, newEntrys, 0, this.entries.length);

		this.entries = newEntrys;
	}
	
	@Override
	public String toString() {
		return Arrays.toString(this.entries);
	}

	@Override
	public Iterator<E> iterator() {
		return this.table.keySet().iterator();
	}

	@Override
	public int size() {
		return this.size;
	}
}