package control.move;

import java.awt.Point;
import java.util.HashMap;
import java.util.Map;

import util.CellDistance;
import util.Heap;
import util.HeuristicComparator;
import util.Log;
import view.Field;
import model.Cast;

/**
 * A* アルゴリズムによって追跡を行うアルゴリズム
 * @author Masayasu Fujiwara
 */
public class AstarTrackingMove implements InteligentMove {
	private final Field field;

	public AstarTrackingMove(Field field) {
		this.field = field;
	}

	public boolean run(Cast predator) {
		int direction = predator.getDirection();
		if (predator.isMove()) {
			if (direction > 0) {
				if (direction == Cast.LEFT) {
					predator.moveLeft();
					return true;
				} else if (direction == Cast.RIGHT) {
					predator.moveRight();
					return true;
				}
			} else if (direction < 0) {
				if (direction == Cast.UP) {
					predator.moveUp();
					return true;
				} else if (direction == Cast.DOWN) {
					predator.moveDown();
					return true;
				}
			}
		}
		Log.out(this, "predator turn");
		switch (this.search(this.field.getCastPoint(predator), this.field.getPreyPoint())) {
			case Cast.RIGHT:
				predator.turnRight();
				predator.moveRight();
				return true;
			case Cast.LEFT:
				predator.turnLeft();
				predator.moveLeft();
				return true;
			case Cast.DOWN:
				predator.turnDown();
				predator.moveDown();
				return true;
			case Cast.UP:
				predator.turnUp();
				predator.moveUp();
				return true;
		}
		return false;
	}

	/**
	 * A* アルゴリズムによって最短の追跡の方向を計算して返します。
	 * @param start 始点の座標（フロア単位）
	 * @param terminal 終点の座標（フロア単位）
	 * @return 追跡の方向
	 */
	public int search(Point start, Point terminal) {
		if (start.equals(terminal)) {
			return 0;
		}
		Heap heap = new Heap(new HeuristicComparator(new CellDistance(), terminal));
		Map<Point, Integer> closed = new HashMap<Point, Integer>();
		Map<Point, Point> path = new HashMap<Point, Point>();
		heap.put(start, 0);
		Point p;
		int count = 0;
		do {
			Heap.Entry entry = heap.poll();
			p = entry.getKey();
			int cost = entry.getValue();
			if (p.equals(terminal)) {
				return this.traceback(terminal, path);
			}

			// p を訪問済み頂点とする
			closed.put(p, cost);

			int px = p.x;
			int py = p.y;
			if (this.field.checkFloor(px - 1, py)) {
				Point open = new Point(px - 1, py);
				if (!closed.containsKey(open)) {
					if (heap.put(open, cost)) {
						path.put(open, p);
					}
				}
			}
			if (this.field.checkFloor(px + 1, py)) {
				Point open = new Point(px + 1, py);
				if (!closed.containsKey(open)) {
					if (heap.put(open, cost)) {
						path.put(open, p);
					}
				}
			}
			if (this.field.checkFloor(px, py - 1)) {
				Point open = new Point(px, py - 1);
				if (!closed.containsKey(open)) {
					if (heap.put(open, cost)) {
						path.put(open, p);
					}
				}
			}
			if (this.field.checkFloor(px, py + 1)) {
				Point open = new Point(px, py + 1);
				if (!closed.containsKey(open)) {
					if (heap.put(open, cost)) {
						path.put(open, p);
					}
				}
			}
		} while (!heap.isEmpty() && count++ < 500);
		System.out.println("search failure");
		return 0;
	}

	/**
	 * 探索した経路をトレースバックして追跡の方向を返します。
	 * @param terminal 探索の終点
	 * @param path 探索した経路
	 * @return 追跡の方向
	 */
	private int traceback(Point terminal, Map<Point, Point> path) {
		Point point = terminal;
		while (true) {
			Point next = path.get(point);
			if (!path.containsKey(next)) {
				if (next.x > point.x) {
					return Cast.LEFT;
				} else if (next.x < point.x) {
					return Cast.RIGHT;
				} else if (next.y > point.y) {
					return Cast.UP;
				} else if (next.y < point.y) {
					return Cast.DOWN;
				}
				break;
			}
			point = next;
		}
		return 0;
	}
}
