package control;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.util.Collection;

import control.move.InteligentMove;
import control.move.RandomMove;
import control.move.RunoutMove;
import control.move.SimpleTrackingMove;
import control.move.TrackingMove;
import control.move.ViewTrackingMove;

import util.Log;
import view.Field;

import model.Cast;

/**
 * コントローラ
 * @author Masayasu Fujiwara
 */
public class Controller implements KeyListener, ActionListener, Runnable {

	/**
	 * 方向
	 */
	private int direction;

	/**
	 * 操作するフィールド
	 */
	private final Field field;
	
	/**
	 * 操作するコントローラ
	 */
	private final Cast prey;

	/**
	 * 操作するコントローラ
	 */
	private final Collection<Cast> predators;

	/**
	 * 再描画の間隔
	 */
	private final int SLEEP = 15;

	public Controller(Field field, Cast player, Collection<Cast> predator) {
		this.field = field;
		this.prey = player;
		this.predators = predator;
		this.algorithm = new RandomMove(field);
	}

	public void keyPressed(KeyEvent e) {
		switch (e.getKeyCode()) {
			case KeyEvent.VK_UP :
				this.direction = Cast.UP;
				Log.out(this, "press UP");
				break;
			case KeyEvent.VK_DOWN :
				this.direction = Cast.DOWN;
				Log.out(this, "press DOWN");
				break;
			case KeyEvent.VK_LEFT :
				this.direction = Cast.LEFT;
				Log.out(this, "press LEFT");
				break;
			case KeyEvent.VK_RIGHT :
				this.direction = Cast.RIGHT;
				Log.out(this, "press RIGHT");
				break;
			case KeyEvent.VK_SPACE :
				this.isEnemyRun = true;
				Log.out(this, "press SPACE");				
		}
	}

	public void keyReleased(KeyEvent e) {
		switch (e.getKeyCode()) {
			case KeyEvent.VK_UP :
				if (this.direction == Cast.UP) {
					this.direction = 0;
				}
				Log.out(this, "release UP");
				break;
			case KeyEvent.VK_DOWN :
				if (this.direction == Cast.DOWN) {
					this.direction = 0;
				}
				Log.out(this, "release DOWN");
				break;
			case KeyEvent.VK_LEFT :
				if (this.direction == Cast.LEFT) {
					this.direction = 0;
				}
				Log.out(this, "release LEFT");
				break;
			case KeyEvent.VK_RIGHT :
				if (this.direction == Cast.RIGHT) {
					this.direction = 0;
				}
				Log.out(this, "release RIGHT");
				break;
			case KeyEvent.VK_SPACE :
				this.isEnemyRun = false;
				Log.out(this, "release SPACE");				
		}
	}

	public void keyTyped(KeyEvent e) {
	}
	
	public void run() {
		try {
			while (true) {
				int moveDirection = this.direction;
				if (moveDirection < 0) {
					if (moveDirection == Cast.UP) {
						this.prey.turnUp();
						if (this.field.checkUp(this.prey)) {
							do {
								this.prey.moveUp();
								this.task(true);
							} while (this.prey.isMove());
						} else {
							this.task(true);
						}
						Log.out(this, "player UP");
					} else if (moveDirection == Cast.DOWN) {
						this.prey.turnDown();
						if (this.field.checkDown(this.prey)) {
							do {
								this.prey.moveDown();
								this.task(true);
							} while (this.prey.isMove());
						} else {
							this.task(true);
						}
						Log.out(this, "player DOWN");
					}
				} else if (moveDirection > 0) {
					if (moveDirection == Cast.LEFT) {
						this.prey.turnLeft();
						if (this.field.checkLeft(this.prey)) {
							do {
								this.prey.moveLeft();
								this.task(true);
							} while (this.prey.isMove());
						} else {
							this.task(true);
						}
						Log.out(this, "player LEFT");
					} else if (moveDirection == Cast.RIGHT) {
						this.prey.turnRight();
						if (this.field.checkRight(this.prey)) {
							do {
								this.prey.moveRight();
								this.task(true);
							} while (this.prey.isMove());
						} else {
							this.task(true);
						}
						Log.out(this, "player RIGHT");
					}
				} else {
					this.task(false);
				}
			}
		} catch (Exception e) {
			Log.err(this, e);
		}
	}
	
	private boolean isEnemyRun;

	/**
	 * プレイヤの操作以外のタスクとスレッドのスリープを行います。
	 * @param isRepaint trueであれば、確実に再描画を行います。
	 * @throws InterruptedException
	 */
	public void task(boolean isRepaint) throws InterruptedException {
		boolean flag = false;
		if (this.isEnemyRun) {
			for (Cast predator : this.predators) {
				flag |= this.algorithm.run(predator);
			}
		}
		if (flag || isRepaint) {
			this.field.repaint();
		}
		Thread.sleep(this.SLEEP);
	}
	
	/**
	 * アルゴリズム
	 */
	private InteligentMove algorithm;

	public void add(InteligentMove algorithm) {
		this.algorithm = algorithm;
	}

	public void actionPerformed(ActionEvent e) {
		String command = e.getActionCommand();
		if (command.equals("exit")) {
			System.exit(0);
		} else if (command.startsWith("ai_")) {
			if (command.equals("ai_random")) {
				this.algorithm = new RandomMove(this.field);
				Log.out(this, "change algorihtm");
			} else if (command.equals("ai_simple_tracking")) {
				this.algorithm = new SimpleTrackingMove(this.field, this.prey);
				Log.out(this, "change algorihtm");
			} else if (command.equals("ai_tracking")) {
				this.algorithm = new TrackingMove(this.field, this.prey);
				Log.out(this, "change algorihtm");
			} else if (command.equals("ai_runout")) {
				this.algorithm = new RunoutMove(this.field, this.prey);
				Log.out(this, "change algorihtm");
			} else if (command.equals("ai_viewtrack")) {
				this.algorithm = new ViewTrackingMove(this.field);
				Log.out(this, "change algorihtm");
			}
		} else if (command.startsWith("field_")) {
			if (command.equals("field_delete")) {
				this.field.fieldClear();
				this.field.repaintBackground();
				Log.out(this, "change algorihtm");
			} else if (command.equals("field_bomb")) {
				this.field.fieldBomb();
				this.field.repaintBackground();
				Log.out(this, "change algorihtm");
			} else if (command.equals("field_maze")) {
				this.field.fieldMaze();
				this.field.repaintBackground();
			}
		}
	}
}
