<?php
require_once('sonots.class.php');

/**
 * Advanced Option Parser for PukiWiki Plugin
 *
 * Example1)
 *  function plugin_hoge_convert()
 *  {
 *      $conf_options = array(
 *          'num' => array('number', 100),
 *          'prefix' => array('string', 'Hoge/'),
 *      );
 *      $args = func_get_args();
 *      $line = csv_implode(',', $args);
 *      $options = PluginSonotsOption::parse_option_line($line);
 *      list($options, $unknowns) = PluginSonotsOption::evaluate_options($options, $conf_options);
 *  }
 * Example2)
 *  function plugin_hoge_inline()
 *  {
 *      $args = func_get_args();
 *      array_pop($args); // drop {}
 *      $line = csv_implode(',', $args);
 *      $options = PluginSonotsOption::parse_option_line($line);
 *      // no $conf_options is also useful
 *  }
 * Example3)
 *  function plugin_hoge_action()
 *  {
 *      global $vars;
 *      $conf_options = array(
 *          'num' => array('number', 100),
 *          'prefix' => array('string', 'Hoge/'),
 *      );
 *      $options = $vars;
 *      list($options, $unknowns) = PluginSonotsOption::evaluate_options($options, $conf_options);
 *  }
 *
 * @package    PluginSonots
 * @license    http://www.gnu.org/licenses/gpl.html GPL v2
 * @author     sonots <http://lsx.sourceforge.jp>
 * @version    $Id: option.class.php,v 1.4 2008-06-07 11:14:46 sonots $
 * @require    sonots    v 1.9
 */
class PluginSonotsOption {
    /**
     * Parse option line as followings:
     *
     * Rule)
     *  , is used to separate options
     *  = is used to separate option key (name) and option value
     *  () is used if element is an array
     *
     * Example)
     * <code>
     *  $line = 'prefix=Hoge/,num=1:5,contents=(num=1,headline)';
     *  $options = PluginSonotsOption::parse_option_line($line);
     *  var_export(); // array('prefix'=>'Hoge/','num'=>'1:5','contents'=>array('num'=>'1','headline'=>true));
     * </code>
     *
     * parse_option_line is upper version of the simple 
     * sonots::parse_options($args) that does not support array
     * arguments. Except array arguments, both can generate the same results. 
     *
     * @access static
     * @param string $line
     * @param boolean $trim trim option key/val
     * @param boolean $decode performe decode key/val
     * @return array options
     * @uses string_to_array
     * @see sonots::parse_options
     */
    function parse_option_line($line, $trim = false, $decode = true)
    {
        $array = sonots::string_to_array($line, '=', ',', '(', ')', $decode);
        $options = PluginSonotsOption::numeric_to_boolean($array);
        if ($trim) {
            $options = sonots::trim_array($options, true, true);
        }
        return $options;
    }

    /**
     * Recover option line. 
     *
     * @access static
     * @param array $options
     * @param boolean $encode performe encode key/val
     * @return string
     * @uses array_to_string
     */
    function glue_option_line($options, $encode = true)
    {
        $array = PluginSonotsOption::boolean_to_numeric($options);
        return sonots::array_to_string($array, '=', ',', '(', ')', $encode);
    }

    /**
     * By string_to_array,
     * $string = 'foo,bar' is array(0=>'foo',1=>'bar').
     * As options, 
     * $string = 'foo,bar' is array('foo'=>true,'bar'=>true).
     * Performe this conversion
     *
     * @access private
     * @param array $array array parsed by string_to_array()
     * @return array $options
     */
    function numeric_to_boolean($array)
    {
        $options = array();
        foreach ($array as $key => $val) {
            if (is_numeric($key)) {
                $options[$val] = true;
            } elseif (is_array($val)) {
                $options[$key] = PluginSonotsOption::numeric_to_boolean($val);
            } else {
                $options[$key] = $val;
            }
        }
        return $options;
    }

    /**
     * Reverse numeric_to_boolean
     *
     * @access private
     * @param array $options
     * @return array $array
     */
    function boolean_to_numeric($options)
    {
        $array = array();
        foreach ($options as $key => $val) {
            if ($val === true) {
                $array[] = $key;
            } elseif (is_array($val)) {
                $array[$key] = PluginSonotsOption::boolean_to_numeric($val);
            } else {
                $array[$key] = $val;
            }
        }
        return $array;
    }

    /**
     * Evaluate options
     *
     * Option values are evaluated (evalute_option()) if given,
     * otherwise, simply default value is set. The validity of 
     * the default value is not checked, for example,
     * it is possible to use 'whetever' default string for bool type 
     * although bool type should be either TRUE or FALSE. 
     * It would be good idea to use NULL as a default value
     * to know whether the option was given by users. 
     *
     * Example)
     * <code>
     *  $conf_options = array(
     *        'hierarchy' => array('bool', true),
     *        'num'       => array('interval', null),
     *        'filter'    => array('string', ''),
     *        'sort'      => array('enum', 'name', array('name', 'reading', 'date')),
     *  );
     *  $options = array('Hoge/'=>true,'filter'=>'AAA');
     *  list($options, $unknowns) = PluginSonotsOption::evaluate_options($options, $conf_options);
     *  var_export($options); // array('hierarchy'=>true,'num'=>null,'filter'=>'AAA','sort'=>'name')
     *  var_export($unknowns); // array('Hoge/'=>true)
     *  if (is_null($options['num'])) echo "the option num was not given.\n" ;
     * </code>
     *
     * Lists of Supported Types
     * - bool      : boolean true or false
     * - string    : string
     * - array     : array
     * - enum      : special string which take only one of collections
     * - enumarray : special array which take only some of collections
     * - number    : number
     * - interval  : interval string. See parse_interval for details
     * - options   : options
     *
     * @access static
     * @param array $options
     *    $options[$name] = $value
     * @param array $conf_options
     *    $conf_options[$name] = array(type, default, conf)
     * @return array array($options, $unknowns)
     *    $options[$name] = $evaluated_value
     *    $unknowns[$unknown_name] = $value
     * @uses evaluate_option
     */
    function evaluate_options($options, $conf_options)
    {
        $outoptions = array();
        $unknowns   = array();
        foreach ($conf_options as $key => $tmp) {
            $default = isset($conf_options[$key][1]) ? $conf_options[$key][1] : null;
            $outoptions[$key] = $default;
        }
        foreach ($options as $key => $val) {
            if (isset($conf_options[$key])) {
                $type = isset($conf_options[$key][0]) ? $conf_options[$key][0] : null;
                $conf = isset($conf_options[$key][2]) ? $conf_options[$key][2] : null;
                list($outoptions[$key], $unknowns[$key]) = PluginSonotsOption::evaluate_option($val, $type, $conf);
                if (is_null($unknowns[$key])) unset($unknowns[$key]);
            } else {
                $unknowns[$key] = $val;
            }
        }
        return array($outoptions, $unknowns);
    }

    /**
     * Evaluate option value
     *
     * Lists of Supported Types
     * - bool      : boolean true or false
     * - string    : string
     * - array     : array
     * - enum      : special string which take only one of collections
     * - enumarray : special array which take only some of collections
     * - number    : number
     * - interval  : interval string. See parse_interval for details
     * - options   : recursive options
     *
     * @access private
     * @param mixed $val option value
     * @param string $type option type
     * @param array $conf config (use for enum, enumarray, options)
     * @return array (evaluated value, invalid value (unknown))
     * @uses PluginSonotsOption::parse_interval
     * @uses PluginSonotsOption::evaluate_options
     */
    function evaluate_option($val, $type, $conf = null)
    {
        $retval = $unknown = null;
        switch ($type) {
        case 'bool':
            if ($val === false ||
                $val === '0' ||
                $val === 'off' ||
                $val === 'false' ||
                $val === 'FALSE') {
                $retval = false;
            } elseif ($val === true ||
                      $val === null ||
                      $val === '' ||
                      $val === '1' ||
                      $val === 'on' ||
                      $val === 'true' ||
                      $val === 'TRUE') {
                $retval = true;
            } else {
                $unknown = $val;
            }
            break;
        case 'string':
            if ($val === true) { // true is default set by parse_option_line
                $val = '';
            }
            if (is_string($val)) {
                $retval  = $val;
            } else {
                $unknown = $val;
            }
            break;
        case 'array':
            if ($val === true) { // true is default set by parse_option_line
                $val = array();
            }
            $retval = (array)$val;
            break;
        case 'enum': // special string type
            if ($val === true) {
                $retval = $conf[0];
            }
            if (is_string($val)) {
                if (in_array($val, $conf)) {
                    $retval = $val;
                }
            } else {
                $unknown = $val;
            }
            break;
        case 'enumarray': // special array type
            if ($val === true) {
                $val = $conf;
            }
            $retval = array();
            $unknown = array();
            foreach ((array)$val as $elem) {
                if (in_array($elem, $conf)) {
                    $retval[] = $elem;
                } else {
                    $unknown[] = $elem;
                }
            }
            if (empty($retval)) $retval = null;
            if (empty($unknown)) $unknown = null;
            break;
        case 'options':
            if ($val === true) {
                $val = array();
            }
            list($retval, $unknown) = PluginSonotsOption::evaluate_options($val, $conf);
            if (empty($retval)) $retval = null;
            if (empty($unknown)) $unknown = null;
            break;
        case 'number':
            if ($val === true) {
                $val = 0;
            }
            if (is_numeric($val)) {
                $retval = $val;
            } else {
                $unknown = $val;
            }
            break;
        case 'interval':
            if ($val === true) {
                $val = '0:';
            }
            $retval = PluginSonotsOption::parse_interval($val);
            if (is_null($retval)) {
                $unknown = $val;
            }
            break;
        default:
            $unknown = $val;
            break;
        }
        return array($retval, $unknown);
    }

    /**
     * Evaluate option value
     *
     * Lists of Types
     *
     * bool:     boolean true or false
     * string:   string
     * array:    array
     * enum:     special string which take only one of collections
     * enumarray special array which take only some of collections
     * number:   number
     * interval: interval string. @uses PluginSonotsOption::parse_interval
     * options:  options. @uses PluginSonotsOption::evaluate_options
     *
     * @access private
     * @param mixed $val option value
     * @param string $type option type
     * @param array $conf config (use for enum, enumarray, options)
     * @return array (evaluated options, unknowns)
     * @uses PluginSonotsOption::parse_interval
     * @uses PluginSonotsOption::evaluate_options
     */
    /*
    function evaluate_option($val, $type, $conf = null)
    {
        switch ($type) {
        case 'bool':
            switch ($val) {
            case '0':
            case 'off':
            case 'false':
            case 'FALSE':
                return false;
                break;
            case null:
            case '':
            case '1':
            case 'on':
            case 'true':
            case 'TRUE':
                return true;
                break;
            default:
                return null;
                break;
            }
            break;
        case 'string':
            if (is_string($val)) {
                return $val;
            } else {
                return null;
            }
            break;
        case 'array':
            return (array)$val;
            break;
        case 'enum': // special string type
            if ($val === '') {
                return $conf[0];
            } elseif (is_string($val)) {
                if (in_array($val, $conf)) {
                    return $val;
                }
            } else {
                return null;
            }
            break;
        case 'enumarray': // special array type
            $val = (array)$val;
            foreach ($val as $elem) {
                if (! in_array($elem, $conf)) {
                    return null;
                }
            }
            return $val;
            break;
        case 'options':
            list($val, $unknowns) = PluginSonotsOption::evaluate_options($val, $conf);
            return $val;
            break;
        case 'number':
            if (is_numeric($val)) {
                return $val;
            } else {
                return null;
            }
            break;
        case 'interval':
            return PluginSonotsOption::parse_interval($val);
            break;
        default:
            return $val;
        }
    }*/

    /**
     * Parse an interval num string
     *
     * Example)
     *  1:5   means 1st to 5th returns array(0, 5)
     *  2:3   means 2nd to 3rd returns array(1, 2)
     *  2:    means 2nd to end returns array(1, null)
     *  :3    means 1st to 2rd returns array(0, 3)
     *  4     means 4th returns array(3, 1)
     *  -5:   means last 5th to end returns array(-5, null)
     *  :-5   means 1st to last 5th returns array(0, -4)
     *  1+2   means 1st to 3rd returns array(0, 3)
     *
     * @access static
     * @param string $interval
     * @return mixed array($offset, $length) or null
     * @see array_slice, array_splice
     */
    function parse_interval($interval)
    {
        $mini = 1; 
        if (strpos($interval, ':') !== false) {
            list($min, $max) = explode(':', $interval, 2);
            if (is_numeric($min)) {
                $min = (int)$min;
            } else {
                $min = $mini;
            }
            if (is_numeric($max)) {
                $max = (int)$max;
                $len = $max - $min + 1;
                if ($len == -1) $len = null;
                if ($len < 0) $len++;
            } else {
                $len = null;
            }
        } elseif (strpos($interval, '+') !== false) {
            list($min, $len) = explode("+", $interval, 2);
            if (is_numeric($min)) {
                $min = (int)$min;
            } else {
                $min = $mini;
            }
            if (is_numeric($len)) {
                $len = (int)$len + 1;
            } else {
                $len = null;
            }
        } else {
            if (is_numeric($interval)) {
                $min = (int)$interval;
                $len = 1;
            } else {
                return null;
            }
        }
        if ($min > 0) $min--;
        return array($min, $len);
    }


    /**
     * Convert ($offset, $length) interval form
     *   to ($start, $end) interval form.
     *
     * Example)
     *  Assume min = 1, max = 10
     *  array(0, 5) to array(1, 5)
     *  array(1, null) to array(2, 10)
     *  array(3, 1) to array(4, 4)
     *  array(-5, null) to array(6, 10)
     *  array(0, -4) to array(1, 6)
     *
     * @access static
     * @param int $offset
     * @param int $length
     * @param int $min
     * @param int $max
     * @return array array($start, $end)
     * @see range
     */
    function conv_interval($offset, $length, $min, $max)
    {
        // minus means index from back
        if ($offset < 0) {
            $start = $offset + $max + 1;
        } else {
            $start = $offset + $min;
        }
        // minus means length from back
        if ($length < 0) {
            $end = $length + $max;
        } elseif ($length > 0) {
            $end = $length + $start - 1;
        } else {
            $end = $max;
        }
        // make sure
        if (! isset($start) || $start < $min) {
            $start = $min;
        }
        if (! isset($end) || $end > $max) {
            $end = $max;
        }
        return array($start, $end);
    }
}

?>
