package org.minimalistbattery.impl.modelizer;

import java.io.InputStream;
import java.util.Hashtable;

import org.lixm.core.common.LIXMPhaseException;
import org.lixm.core.common.XMLModelizer;
import org.lixm.core.list.AttributesList;
import org.lixm.core.list.XMLModelList;
import org.lixm.core.model.AttributeModel;
import org.lixm.core.model.CharactersModel;
import org.lixm.core.model.ElementModel;
import org.lixm.core.model.EndDocumentModel;
import org.lixm.core.model.EndTagModel;
import org.lixm.core.model.StartDocumentModel;
import org.lixm.core.model.StartTagModel;
import org.lixm.optional.v11.namespace.NamespaceBinding;
import org.xml.sax.Attributes;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.SAXNotRecognizedException;
import org.xml.sax.SAXNotSupportedException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;
import org.xmlpull.mxp1.MXParser;
import org.xmlpull.v1.XmlPullParserException;

/**
 * <p>モデライザのJSR-172を利用しない実装です。</p>
 * <p>実装にはXPPの実装であるMXPをSAXドライバを利用して実装してあります。
 * このため、SAXに渡すハンドラを変更すればモデライザの動作ををカスタマイズ出来ます。</p>
 * 
 * @author tasogare
 * 
 * @see #setHandler(DefaultHandler)
 */
public class MinimalistXMLModelizer implements XMLModelizer, Locator, Attributes {

    public class ModelizerHandler extends DefaultHandler {

	private MinimalistNamespaceBinding namespaces;

	public ModelizerHandler() {
	    namespaces = new MinimalistNamespaceBinding();
	}

	public void startDocument() throws SAXException {
	    list.add(new StartDocumentModel(null));
	}

	public void endDocument() throws SAXException {
	    list.add(new EndDocumentModel(null));
	}

	public void characters(char[] ch, int start, int length)
		throws SAXException {
	    list.add(new CharactersModel(new String(ch, start, length)));
	}

	public void startElement(String uri, String localName, String qName,
		Attributes attributes) throws SAXException {

	    String elemPrefix = (String)namespaces.getPrefix(uri);

	    AttributesList attList = new AttributesList(attributes.getLength());
	    for (int i = 0; i < attributes.getLength(); i++) {

		String pref = (String)namespaces.getPrefix(attributes.getURI(i));
		attList.add(new AttributeModel(attributes.getLocalName(i),
			attributes.getValue(i), pref, attributes.getURI(i)));
	    }

	    list.add(new StartTagModel(new ElementModel(
		    localName, elemPrefix, uri), attList));
	}

	public void endElement(String uri, String localName, String qName)
		throws SAXException {

	    String elemPrefix = (String)namespaces.getPrefix(uri);
	    list.add(new EndTagModel(new ElementModel(
		    localName, elemPrefix, uri)));
	}

	public void startPrefixMapping(String prefix, String uri)
		throws SAXException {
	    namespaces.put(prefix, uri);
	}

	public void endPrefixMapping(String prefix) throws SAXException {
	}

	/**
	 * 名前空間バインディングを返す
	 * @return 名前空間バインディング
	 */
	public NamespaceBinding getNamespaceBinding(){
	    return namespaces;
	}
    }

    protected static final String DECLARATION_HANDLER_PROPERTY = "http://xml.org/sax/properties/declaration-handler";

    protected static final String LEXICAL_HANDLER_PROPERTY = "http://xml.org/sax/properties/lexical-handler";

    protected static final String NAMESPACES_FEATURE = "http://xml.org/sax/features/namespaces";

    protected static final String NAMESPACE_PREFIXES_FEATURE = "http://xml.org/sax/features/namespace-prefixes";

    protected static final String VALIDATION_FEATURE = "http://xml.org/sax/features/validation";

    protected static final String APACHE_SCHEMA_VALIDATION_FEATURE = "http://apache.org/xml/features/validation/schema";

    protected static final String APACHE_DYNAMIC_VALIDATION_FEATURE = "http://apache.org/xml/features/validation/dynamic";

    protected String systemId;

    protected DefaultHandler defaultHandler;

    protected XMLModelList list;

    protected MXParser pp;

    protected MinimalistXMLModelizer(){
    }

    /**
     * <p>新たなモデルリストを関連付けて初期化するコンストラクタです。</p>
     * @param list 関連付けるモデルリスト
     * @throws LIXMPhaseException
     */
    public MinimalistXMLModelizer(XMLModelList list) throws LIXMPhaseException {
	pp = new MXParser();
	setList(list);

	try {
	    setFeature(NAMESPACES_FEATURE, true);
	} catch (Exception e) {
	    throw new LIXMPhaseException(e, LIXMPhaseException.PHASE_PREPROCESS);
	}

	setHandler(new ModelizerHandler());
    }

    /**
     * <p>カスタムSAXハンドラを利用して初期化するコンストラクタです。</p>
     * 
     * @param list 関連付けるリスト
     * @param handler カスタムSAXハンドラ
     * 
     * @throws LIXMPhaseException
     */
    public MinimalistXMLModelizer(XMLModelList list, DefaultHandler handler) throws LIXMPhaseException {
	pp = new MXParser();
	setList(list);

	try {
	    setFeature(NAMESPACES_FEATURE, true);
	} catch (Exception e) {
	    throw new LIXMPhaseException(e, LIXMPhaseException.PHASE_MODELIZE);
	}

	setHandler(handler);
    }

    // -- Attributes interface

    public int getLength() {
	return pp.getAttributeCount();
    }

    public String getURI(int index) {
	return pp.getAttributeNamespace(index);
    }

    public String getLocalName(int index) {
	return pp.getAttributeName(index);
    }

    public String getQName(int index) {
	final String prefix = pp.getAttributePrefix(index);
	if (prefix != null) {
	    return prefix + ':' + pp.getAttributeName(index);
	} else {
	    return pp.getAttributeName(index);
	}
    }

    public String getType(int index) {
	return pp.getAttributeType(index);
    }

    public String getValue(int index) {
	return pp.getAttributeValue(index);
    }

    public int getIndex(String uri, String localName) {
	for (int i = 0; i < pp.getAttributeCount(); i++) {
	    if (pp.getAttributeNamespace(i).equals(uri)
		    && pp.getAttributeName(i).equals(localName)) {
		return i;
	    }

	}
	return -1;
    }

    public int getIndex(String qName) {
	for (int i = 0; i < pp.getAttributeCount(); i++) {
	    if (pp.getAttributeName(i).equals(qName)) {
		return i;
	    }

	}
	return -1;
    }

    public String getType(String uri, String localName) {
	for (int i = 0; i < pp.getAttributeCount(); i++) {
	    if (pp.getAttributeNamespace(i).equals(uri)
		    && pp.getAttributeName(i).equals(localName)) {
		return pp.getAttributeType(i);
	    }

	}
	return null;
    }

    public String getType(String qName) {
	for (int i = 0; i < pp.getAttributeCount(); i++) {
	    if (pp.getAttributeName(i).equals(qName)) {
		return pp.getAttributeType(i);
	    }

	}
	return null;
    }

    public String getValue(String uri, String localName) {
	return pp.getAttributeValue(uri, localName);
    }

    public String getValue(String qName) {
	return pp.getAttributeValue(null, qName);
    }

    // -- Locator interface

    public String getPublicId() {
	return null;
    }

    public String getSystemId() {
	return systemId;
    }

    public int getLineNumber() {
	return pp.getLineNumber();
    }

    public int getColumnNumber() {
	return pp.getColumnNumber();
    }

    // --- XMLReader interface

    /**
     * <p>機能フラグの値を検索します。</p>
     * <p>機能名は完全修飾 URI です。</p>
     * <p>XMLReader は機能名を認識することはできますが、同時にその値を返すことはできません。
     * 機能の値によっては、構文解析の実行前、実行中、
     * 実行後などの特定のコンテキストでのみ使用可能な場合があります。
     * また、プログラムではアクセスできない機能値もあります。
     * SAX1 Parser のアダプタの場合、基本となるパーサーが検証を実行しているかどうかや、
     * 外部エンティティーを拡張しているかどうかなどを表示する方法は実装に依存しません。</p>
     *
     * @throws SAXNotRecognizedException
     * 		機能の値を割り当てられない、 または取得できない場合
     * @throws SAXNotSupportedException
     * 		XMLReader が機能名を認識するだけで、 現時点ではその値を判断できない場合
     *  
     *  @return 機能の現在の値 (true または false) 
     */
    public boolean getFeature(String name) 
    	throws SAXNotRecognizedException, SAXNotSupportedException {
	if (NAMESPACES_FEATURE.equals(name)) {
	    return pp.getFeature(MXParser.FEATURE_PROCESS_NAMESPACES);
	} else if (NAMESPACE_PREFIXES_FEATURE.equals(name)) {
	    return pp.getFeature(MXParser.FEATURE_REPORT_NAMESPACE_ATTRIBUTES);
	} else if (VALIDATION_FEATURE.equals(name)) {
	    return pp.getFeature(MXParser.FEATURE_VALIDATION);
	} else {
	    return pp.getFeature(name);
	}
    }

    /**
     * <p>機能フラグの値を設定します。</p>
     * 
     * <p>機能名は完全修飾 URI です。</p>
     * 
     * <p>XMLReader は機能の値を表示できますが、現在の値を変更することはできません。
     * 機能の値によっては、構文解析の実行前、実行中、
     * 実行後などの特定のコンテキストでのみ不変または可変である場合があります。</p>
     * 
     * @param name 機能名。完全修飾 URI で表される
     * @param value 機能の要求された値 (true または false)
     *
     * @throws SAXNotRecognizedException
     * 		機能の値を割り当てられない、 または取得できない場合
     *
     * @throws SAXNotSupportedException
     * 		XMLReader が機能名を認識するだけで、 要求された値を設定できない場合
     * 
     * @link #getFeature(String)
     */
    public void setFeature(String name, boolean value)
	    throws SAXNotRecognizedException, SAXNotSupportedException {
	try {
	    if (NAMESPACES_FEATURE.equals(name)) {
		pp.setFeature(MXParser.FEATURE_PROCESS_NAMESPACES, value);
	    } else if (NAMESPACE_PREFIXES_FEATURE.equals(name)) {
		if (pp.getFeature(MXParser.FEATURE_REPORT_NAMESPACE_ATTRIBUTES) != value) {
		    pp.setFeature(MXParser.FEATURE_REPORT_NAMESPACE_ATTRIBUTES, value);
		}
	    } else if (VALIDATION_FEATURE.equals(name)) {
		pp.setFeature(MXParser.FEATURE_VALIDATION, value);
	    } else {
		pp.setFeature(name, value);
	    }
	} catch (Exception ex) {
	    throw new SAXNotSupportedException("problem with setting feature "
		    + name + ": " + ex);
	}
    }

    /**
     * <p>プロパティー値を検索します。</p>
     * <p>プロパティー名は完全修飾 URI です。</p>
     * <p>XMLReader はプロパティー名を認識できますが、一時的にその値を返すことはできません。
     * プロパティーの値によっては、構文解析の実行前、実行中、
     * 実行後などの特定のコンテキストでしか使用できない場合があります。
     * SAX2 には初期コアセットが文書化されていますが、
     * XMLReader は特定のプロパティー名を認識できなくてもかまいません。</p>
     * 
     * @param name プロパティー名。完全修飾 URI で表される
     * @return 現在のプロパティー値
     * 
     * @throws SAXNotRecognizedException
     * 		プロパティーの値を割り当てられない、 または取得できない場合
     *
     * @throws SAXNotSupportedException
     * 		XMLReader がプロパティー名を認識するだけで、 現時点ではその値を判断できない場合
     */
    public Object getProperty(String name) throws SAXNotRecognizedException,
	    SAXNotSupportedException {
	if (DECLARATION_HANDLER_PROPERTY.equals(name)) {
	    return null;
	} else if (LEXICAL_HANDLER_PROPERTY.equals(name)) {
	    return null;
	} else {
	    return pp.getProperty(name);
	}
    }

    /**
     * <p>プロパティー値を設定します。</p>
     * <p>プロパティー名は完全修飾 URI です。</p>
     * <p>XMLReader はプロパティー名を認識できますが、現在の値を変更することはできません。
     * プロパティーの値によっては、構文解析の実行前、実行中、
     * 実行後などの特定のコンテキストでのみ不変または可変である場合があります。
     * コアセットは SAX2 によって定義されていますが、
     * XMLReader は特定のプロパティー名を認識できなくてもかまいません。</p>
     * 
     * @param name プロパティー名。完全修飾 URI で表される
     * @param value 要求されたプロパティー値
     *
     * @throws SAXNotRecognizedException
     * 		プロパティーの値を割り当てられない、 または取得できない場合
     *
     * @throws SAXNotSupportedException
     * 		 XMLReader がプロパティー名を認識するだけで、 要求された値を設定できない場合
     */
    public void setProperty(String name, Object value)
	    throws SAXNotRecognizedException, SAXNotSupportedException {
	//
	if (DECLARATION_HANDLER_PROPERTY.equals(name)) {
	    throw new SAXNotSupportedException(
		    "not supported setting property " + name);
	} else if (LEXICAL_HANDLER_PROPERTY.equals(name)) {
	    throw new SAXNotSupportedException(
		    "not supported setting property " + name);
	} else {
	    try {
		pp.setProperty(name, value);
	    } catch (Exception ex) {
		throw new SAXNotSupportedException(
			"not supported set property " + name + ": " + ex);
	    }
	}
    }

    /**
     * <p>モデライザの内部SAXハンドラを変更します。</p>
     * @param handler カスタムハンドラ
     */
    public void setHandler(DefaultHandler handler) {
	this.defaultHandler = handler;
    }

    /**
     * <p>モデライザの内部SAXハンドラを取得します。</p>
     * @return 内部SAXハンドラ
     */
    public DefaultHandler getHandler() {
	return defaultHandler;
    }

    /**
     *      * <p>モデライズフェーズを開始します。
     * 先に何らかの方法でXMLModelList  を関連付けておく必要があります。</p>
     * <p>実装上の制限：XML文章のエンコーディングはUTF-8のみサポート</p>
     * 
     * @param ＸＭＬ文章のパス
     * 
     * @throws LIXMPhaseException モデライズ中に何らかの例外が 発生した場合投げられます。
     * 		他の例外をラップしている可能性があります。
     */
    public void modelize(String name) throws LIXMPhaseException {

	InputStream stream = getClass().getResourceAsStream(name);
	defaultHandler.setDocumentLocator(this);

	try {

	    if (stream == null) {
		throw new SAXParseException("null source", this);
	    }

	    systemId = name;

	    //UTF-8で固定
	    pp.setInput(stream, "UTF-8");

	} catch (SAXParseException e) {
	    throw new LIXMPhaseException("parsing initialization error ", e,
		    LIXMPhaseException.PHASE_MODELIZE);
	} catch (XmlPullParserException e) {
	    throw new LIXMPhaseException("not UTF-8 encoding streem error ", e,
		    LIXMPhaseException.PHASE_MODELIZE);
	}

	// start parsing - move to first start tag
	try {
	    defaultHandler.startDocument();

	    // get first event
	    pp.next();

	    // it should be start tag...
	    if (pp.getEventType() != MXParser.START_TAG) {
		throw new LIXMPhaseException("expected start tag not"
			+ pp.getPositionDescription(),
			LIXMPhaseException.PHASE_MODELIZE);
	    }
	} catch (Exception e) {
	    throw new LIXMPhaseException("parsing initialization error", e,
		    LIXMPhaseException.PHASE_MODELIZE);
	}

	// now real parsing can start!

	parseSubTree(pp);

	// and finished ...

	try {
	    defaultHandler.endDocument();
	} catch (SAXException e) {
	    throw new LIXMPhaseException("parsing initialization error", e,
		    LIXMPhaseException.PHASE_MODELIZE);
	}
    }

    protected void parseSubTree(MXParser pp) throws LIXMPhaseException {

	this.pp = pp;
	boolean namespaceAware = pp.getFeature(MXParser.FEATURE_PROCESS_NAMESPACES);

	try {
	    if (pp.getEventType() != MXParser.START_TAG) {
		throw new LIXMPhaseException("start tag must be read before skiping subtree"
			+ pp.getPositionDescription(),LIXMPhaseException.PHASE_MODELIZE);
	    }

	    final int[] holderForStartAndLength = new int[2];
	    final StringBuffer rawName = new StringBuffer(16);
	    String prefix = null;
	    String name = null;
	    int level = pp.getDepth() - 1;
	    int type = MXParser.START_TAG;

	    LOOP: do {
		switch (type) {
		case MXParser.START_TAG:
		    if (namespaceAware) {

			final int depth = pp.getDepth() - 1;
			final int countPrev = (level > depth) ? pp.getNamespaceCount(depth) : 0;

				final int count = pp.getNamespaceCount(depth + 1);
			for (int i = countPrev; i < count; i++) {
			    defaultHandler.startPrefixMapping(
				    pp.getNamespacePrefix(i),
				    pp.getNamespaceUri(i));
			}

			name = pp.getName();
			prefix = pp.getPrefix();

			if (prefix != null) {
			    rawName.setLength(0);
			    rawName.append(prefix);
			    rawName.append(':');
			    rawName.append(name);
			}

			startElement(pp.getNamespace(), name,
				prefix != null ? name : rawName.toString());
		    } else {
			startElement(pp.getNamespace(), pp.getName(), pp
				.getName());
		    }

		    break;
		case MXParser.TEXT:
		    final char[] chars = pp
			    .getTextCharacters(holderForStartAndLength);
		    defaultHandler.characters(chars,
			    holderForStartAndLength[0], // start
			    holderForStartAndLength[1] // len
			    );
		    break;
		case MXParser.END_TAG:
		    // --level;
		    if (namespaceAware) {
			name = pp.getName();
			prefix = pp.getPrefix();
			if (prefix != null) {
			    rawName.setLength(0);
			    rawName.append(prefix);
			    rawName.append(':');
			    rawName.append(name);
			}
			defaultHandler.endElement(pp.getNamespace(), name,
				prefix != null ? name : rawName.toString());
			// when entering show prefixes for all levels!!!!
			final int depth = pp.getDepth();
			final int countPrev = (level > depth) ? pp
				.getNamespaceCount(pp.getDepth()) : 0;
			int count = pp.getNamespaceCount(pp.getDepth() - 1);
			// undeclare them in reverse order
			for (int i = count - 1; i >= countPrev; i--) {
			    defaultHandler.endPrefixMapping(pp
				    .getNamespacePrefix(i));
			}
		    } else {
			defaultHandler.endElement(pp.getNamespace(), pp
				.getName(), pp.getName());

		    }
		    break;
		case MXParser.END_DOCUMENT:
		    break LOOP;
		}
		type = pp.next();
	    } while (pp.getDepth() > level);
	} catch (Exception e) {
	    throw new LIXMPhaseException("parsing error", e,
		    LIXMPhaseException.PHASE_MODELIZE);
	}
    }

    protected void startElement(String namespace, String localName, String qName)
	    throws SAXException {
	defaultHandler.startElement(namespace, localName, qName, this);
    }

    /**
     * <p>このモデライザに関連付けられた XMLModelListを取得します。
     * モデルリストを関連付ける方法は任意ですがモデライズ前には必ず
     * 関連付けておく必要があります。</p>
     * 
     * @return 関連付けられたモデルリスト
     */
    public XMLModelList getList() {
	return list;
    }

    /**
     * <p>モデライザを関連付けます。</p>
     * 
     * @param list 関連付けるモデライザ
     */
    public void setList(XMLModelList list) {
	this.list = list;
    }
}
