/*
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package handlers.telnethandlers;

import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.lang.management.ManagementFactory;
import java.lang.management.ThreadInfo;
import java.lang.management.ThreadMXBean;
import java.net.Socket;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Map;
import java.util.Map.Entry;
import java.util.StringTokenizer;

import javolution.util.FastComparator;
import javolution.util.FastTable;

import com.l2jserver.Config;
import com.l2jserver.gameserver.GameTimeController;
import com.l2jserver.gameserver.GmListTable;
import com.l2jserver.gameserver.LoginServerThread;
import com.l2jserver.gameserver.ThreadPoolManager;
import com.l2jserver.gameserver.handler.ITelnetHandler;
import com.l2jserver.gameserver.model.L2Object;
import com.l2jserver.gameserver.model.L2World;
import com.l2jserver.gameserver.model.actor.L2Character;
import com.l2jserver.gameserver.model.actor.L2Npc;
import com.l2jserver.gameserver.model.actor.L2Summon;
import com.l2jserver.gameserver.model.actor.instance.L2DoorInstance;
import com.l2jserver.gameserver.model.actor.instance.L2MonsterInstance;
import com.l2jserver.gameserver.model.actor.instance.L2PcInstance;
import com.l2jserver.gameserver.model.item.instance.L2ItemInstance;
import com.l2jserver.gameserver.taskmanager.DecayTaskManager;

/**
 * @author UnAfraid
 */
public class DebugHandler implements ITelnetHandler
{
	private final String[] _commands =
	{
		"debug"
	};
	
	private int uptime = 0;
	
	@Override
	public boolean useCommand(String command, PrintWriter _print, Socket _cSocket, int _uptime)
	{
		if (command.startsWith("debug") && command.length() > 6)
		{
			StringTokenizer st = new StringTokenizer(command.substring(6));
			FileOutputStream fos = null;
			OutputStreamWriter out = null;
			try
			{
				String dbg = st.nextToken();
				
				if (dbg.equals("decay"))
				{
					_print.print(DecayTaskManager.getInstance().toString());
				}
				else if (dbg.equals("PacketTP"))
				{
					String str = ThreadPoolManager.getInstance().getPacketStats();
					_print.println(str);
					int i = 0;
					File f = new File("./log/StackTrace-PacketTP-" + i + ".txt");
					while (f.exists())
					{
						i++;
						f = new File("./log/StackTrace-PacketTP-" + i + ".txt");
					}
					f.getParentFile().mkdirs();
					fos = new FileOutputStream(f);
					out = new OutputStreamWriter(fos, "UTF-8");
					out.write(str);
				}
				else if (dbg.equals("IOPacketTP"))
				{
					String str = ThreadPoolManager.getInstance().getIOPacketStats();
					_print.println(str);
					int i = 0;
					File f = new File("./log/StackTrace-IOPacketTP-" + i + ".txt");
					while (f.exists())
					{
						i++;
						f = new File("./log/StackTrace-IOPacketTP-" + i + ".txt");
					}
					f.getParentFile().mkdirs();
					fos = new FileOutputStream(f);
					out = new OutputStreamWriter(fos, "UTF-8");
					out.write(str);
				}
				else if (dbg.equals("GeneralTP"))
				{
					String str = ThreadPoolManager.getInstance().getGeneralStats();
					_print.println(str);
					int i = 0;
					File f = new File("./log/StackTrace-GeneralTP-" + i + ".txt");
					while (f.exists())
					{
						i++;
						f = new File("./log/StackTrace-GeneralTP-" + i + ".txt");
					}
					f.getParentFile().mkdirs();
					fos = new FileOutputStream(f);
					out = new OutputStreamWriter(fos, "UTF-8");
					out.write(str);
				}
				else if (dbg.equals("full"))
				{
					Calendar cal = Calendar.getInstance();
					SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss Z");
					
					StringBuilder sb = new StringBuilder(256);
					sb.append(sdf.format(cal.getTime()));
					sb.append("\n\nL2J Server Version: ").append(Config.SERVER_VERSION);
					sb.append("\nDP Revision: ").append(Config.DATAPACK_VERSION);
					sb.append("\n\n");
					uptime = _uptime;
					sb.append(getServerStatus());
					sb.append("\n\n"
						+ "\n## Java Platform Information ##"
						+ "\nJava Runtime Name: ").append(System.getProperty("java.runtime.name"));
					sb.append("\nJava Version: ").append(System.getProperty("java.version"));
					sb.append("\nJava Class Version: ").append(System.getProperty("java.class.version"));
					sb.append("\n"
						+ "\n## Virtual Machine Information ##"
						+ "\nVM Name: ").append(System.getProperty("java.vm.name"));
					sb.append("\nVM Version: ").append(System.getProperty("java.vm.version"));
					sb.append("\nVM Vendor: ").append(System.getProperty("java.vm.vendor"));
					sb.append("\nVM Info: ").append(System.getProperty("java.vm.info"));
					sb.append("\n"
						+ "\n## OS Information ##"
						+ "\nName: ").append(System.getProperty("os.name"));
					sb.append("\nArchiteture: ").append(System.getProperty("os.arch"));
					sb.append("\nVersion: ").append(System.getProperty("os.version"));
					sb.append("\n"
						+ "\n## Runtime Information ##"
						+ "\nCPU Count: ").append(Runtime.getRuntime().availableProcessors());
					sb.append("\nCurrent Free Heap Size: ").append((Runtime.getRuntime().freeMemory() / 1024 / 1024)).append(" mb"
						+ "\nCurrent Heap Size: ").append((Runtime.getRuntime().totalMemory() / 1024 / 1024)).append(" mb"
						+ "\nMaximum Heap Size: ").append((Runtime.getRuntime().maxMemory() / 1024 / 1024)).append(" mb"
					
						+ "\n"
						+ "\n## Class Path Information ##\n");
					String cp = System.getProperty("java.class.path");
					String[] libs = cp.split(File.pathSeparator);
					for (String lib : libs)
					{
						sb.append(lib);
						sb.append('\n');
					}
					
					sb.append("\n"
						+ "## Threads Information ##\n");
					Map<Thread, StackTraceElement[]> allThread = Thread.getAllStackTraces();
					
					FastTable<Entry<Thread, StackTraceElement[]>> entries = new FastTable<Entry<Thread, StackTraceElement[]>>();
					entries.setValueComparator(new FastComparator<Entry<Thread, StackTraceElement[]>>()
					{
						
						/**
						 * 
						 */
						private static final long serialVersionUID = 1L;
						
						@Override
						public boolean areEqual(Entry<Thread, StackTraceElement[]> e1, Entry<Thread, StackTraceElement[]> e2)
						{
							return e1.getKey().getName().equals(e2.getKey().getName());
						}
						
						@Override
						public int compare(Entry<Thread, StackTraceElement[]> e1, Entry<Thread, StackTraceElement[]> e2)
						{
							return e1.getKey().getName().compareTo(e2.getKey().getName());
						}
						
						@Override
						public int hashCodeOf(Entry<Thread, StackTraceElement[]> e)
						{
							return e.hashCode();
						}
						
					});
					entries.addAll(allThread.entrySet());
					entries.sort();
					for (Entry<Thread, StackTraceElement[]> entry : entries)
					{
						StackTraceElement[] stes = entry.getValue();
						Thread t = entry.getKey();
						sb.append("--------------\n");
						sb.append(t.toString()).append(" (").append(t.getId()).append(")\n"
							+ "State: ").append(t.getState()).append("\n"
							+ "isAlive: ").append(t.isAlive()).append(" | isDaemon: ").append(t.isDaemon()).append(" | isInterrupted: ").append(t.isInterrupted()).append("\n"
							+ "\n");
						for (StackTraceElement ste : stes)
						{
							sb.append(ste.toString());
							sb.append('\n');
						}
						sb.append('\n');
					}
					
					sb.append('\n');
					ThreadMXBean mbean = ManagementFactory.getThreadMXBean();
					long[] ids = findDeadlockedThreads(mbean);
					if (ids != null && ids.length > 0)
					{
						Thread[] threads = new Thread[ids.length];
						for (int i = 0; i < threads.length; i++)
						{
							threads[i] = findMatchingThread(mbean.getThreadInfo(ids[i]));
						}
						sb.append("Deadlocked Threads:\n"
							+ "-------------------\n");
						for (Thread thread : threads)
						{
							System.err.println(thread);
							for (StackTraceElement ste : thread.getStackTrace())
							{
								sb.append('\t').append(ste);
								sb.append('\n');
							}
						}
					}
					
					sb.append("\n\n## Thread Pool Manager Statistics ##\n");
					for (String line : ThreadPoolManager.getInstance().getStats())
					{
						sb.append(line);
						sb.append('\n');
					}
					
					int i = 0;
					File f = new File("./log/Debug-" + i + ".txt");
					while (f.exists())
					{
						i++;
						f = new File("./log/Debug-" + i + ".txt");
					}
					f.getParentFile().mkdirs();
					fos = new FileOutputStream(f);
					out = new OutputStreamWriter(fos, "UTF-8");
					out.write(sb.toString());
					out.flush();
					out.close();
					fos.close();
					
					_print.println("Debug output saved to log/" + f.getName());
					_print.flush();
				}
			}
			catch (Exception e)
			{
			}
			finally
			{
				try
				{
					out.close();
				}
				catch (Exception e)
				{
				}
				
				try
				{
					fos.close();
				}
				catch (Exception e)
				{
				}
			}
			
		}
		return false;
	}
	
	private long[] findDeadlockedThreads(ThreadMXBean mbean)
	{
		// JDK 1.5 only supports the findMonitorDeadlockedThreads()
		// method, so you need to comment out the following three lines
		if (mbean.isSynchronizerUsageSupported())
		{
			return mbean.findDeadlockedThreads();
		}
		else
		{
			return mbean.findMonitorDeadlockedThreads();
		}
	}
	
	private Thread findMatchingThread(ThreadInfo inf)
	{
		for (Thread thread : Thread.getAllStackTraces().keySet())
		{
			if (thread.getId() == inf.getThreadId())
			{
				return thread;
			}
		}
		throw new IllegalStateException("Deadlocked Thread not found");
	}
	
	public String getServerStatus()
	{
		int playerCount = 0, objectCount = 0;
		int max = LoginServerThread.getInstance().getMaxPlayer();
		
		playerCount = L2World.getInstance().getAllPlayersCount();
		objectCount = L2World.getInstance().getAllVisibleObjectsCount();
		
		int itemCount = 0;
		int itemVoidCount = 0;
		int monsterCount = 0;
		int minionCount = 0;
		int minionsGroupCount = 0;
		int npcCount = 0;
		int charCount = 0;
		int pcCount = 0;
		int detachedCount = 0;
		int doorCount = 0;
		int summonCount = 0;
		int AICount = 0;
		
		L2Object[] objs = L2World.getInstance().getAllVisibleObjectsArray();
		for (L2Object obj : objs)
		{
			if (obj == null)
				continue;
			if (obj instanceof L2Character)
				if (((L2Character) obj).hasAI())
					AICount++;
			if (obj instanceof L2ItemInstance)
				if (((L2ItemInstance) obj).getLocation() == L2ItemInstance.ItemLocation.VOID)
					itemVoidCount++;
				else
					itemCount++;
			
			else if (obj instanceof L2MonsterInstance)
			{
				monsterCount++;
				if (((L2MonsterInstance) obj).hasMinions())
				{
					minionCount += ((L2MonsterInstance) obj).getMinionList().countSpawnedMinions();
					minionsGroupCount += ((L2MonsterInstance) obj).getMinionList().lazyCountSpawnedMinionsGroups();
				}
			}
			else if (obj instanceof L2Npc)
				npcCount++;
			else if (obj instanceof L2PcInstance)
			{
				pcCount++;
				if (((L2PcInstance) obj).getClient() != null && ((L2PcInstance) obj).getClient().isDetached())
					detachedCount++;
			}
			else if (obj instanceof L2Summon)
				summonCount++;
			else if (obj instanceof L2DoorInstance)
				doorCount++;
			else if (obj instanceof L2Character)
				charCount++;
		}
		StringBuilder sb = new StringBuilder(256);
		sb.append("Server Status: "
			+ "\r\n  --->  Player Count: ").append(playerCount).append('/').append(max);
		sb.append("\r\n  ---> Offline Count: ").append(detachedCount).append('/').append(playerCount);
		sb.append("\r\n  +-->  Object Count: ").append(objectCount);
		sb.append("\r\n  +-->      AI Count: ").append(AICount);
		sb.append("\r\n  +.... L2Item(Void): ").append(itemVoidCount);
		sb.append("\r\n  +.......... L2Item: ").append(itemCount);
		sb.append("\r\n  +....... L2Monster: ").append(monsterCount);
		sb.append("\r\n  +......... Minions: ").append(minionCount);
		sb.append("\r\n  +.. Minions Groups: ").append(minionsGroupCount);
		sb.append("\r\n  +........... L2Npc: ").append(npcCount);
		sb.append("\r\n  +............ L2Pc: ").append(pcCount);
		sb.append("\r\n  +........ L2Summon: ").append(summonCount);
		sb.append("\r\n  +.......... L2Door: ").append(doorCount);
		sb.append("\r\n  +.......... L2Char: ").append(charCount);
		sb.append("\r\n  --->   Ingame Time: ").append(gameTime());
		sb.append("\r\n  ---> Server Uptime: ").append(getUptime(uptime));
		sb.append("\r\n  --->      GM Count: ").append(getOnlineGMS());
		sb.append("\r\n  --->       Threads: ").append(Thread.activeCount());
		sb.append("\r\n  RAM Used: ").append(((Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()) / 1048576)); // 1024 * 1024 = 1048576
		sb.append("\r\n");
		
		return sb.toString();
	}
	
	private int getOnlineGMS()
	{
		return GmListTable.getInstance().getAllGms(true).size();
	}
	
	private String getUptime(int time)
	{
		int uptime = (int) System.currentTimeMillis() - time;
		uptime = uptime / 1000;
		int h = uptime / 3600;
		int m = (uptime - (h * 3600)) / 60;
		int s = ((uptime - (h * 3600)) - (m * 60));
		return h + "hrs " + m + "mins " + s + "secs";
	}
	
	private String gameTime()
	{
		int t = GameTimeController.getInstance().getGameTime();
		int h = t / 60;
		int m = t % 60;
		SimpleDateFormat format = new SimpleDateFormat("H:mm");
		Calendar cal = Calendar.getInstance();
		cal.set(Calendar.HOUR_OF_DAY, h);
		cal.set(Calendar.MINUTE, m);
		return format.format(cal.getTime());
	}
	
	@Override
	public String[] getCommandList()
	{
		return _commands;
	}
}
