/* **************************************************************************
 * Copyright (C) 2008 BJoRFUAN. All Right Reserved
 * **************************************************************************
 * This module, contains source code, binary and documentation, is in the
 * BSD License, and comes with NO WARRANTY.
 * 
 *                                                 torao <torao@bjorfuan.com>
 *                                                       http://www.moyo.biz/
 * $Id: Rational.java,v 1.3 2008/08/11 17:29:43 torao Exp $
*/
package org.koiroha.fixez;

import java.io.Serializable;



// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// Rational: 有理数クラス
// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
/**
 * int 型の分子と分母で構成される有理数を表す不変クラスです。無符号 4 バイト
 * 整数を保持するために分子/分母共に long 値を取ります。符号は分子のみが持ち
 * ます。
 * <p>
 * @version fixez 1.0 - $Revision: 1.3 $ $Date: 2008/08/11 17:29:43 $
 * @author <a href="mailto:torao@mars.dti.ne.jp">torao</a>
 * @since fixez 1.0 - 2008/08/11
 */
public final class Rational extends Number implements Serializable {

	// ======================================================================
	// シリアルバージョン
	// ======================================================================
	/**
	 * このクラスのシリアルバージョンです。
	 * <p>
	*/
	private static final long serialVersionUID = 1L;

	// ======================================================================
	// NaN
	// ======================================================================
	/**
	 * NaN を表す定数です。0/0 を示します。
	 * <p>
	 */
	public static final Rational NaN = new Rational(0, 0);

	// ======================================================================
	// ゼロ
	// ======================================================================
	/**
	 * ゼロを表す定数です。0/1 を示します。
	 * <p>
	 */
	public static final Rational ZERO = new Rational(0, 1);

	// ======================================================================
	// 正の無限大
	// ======================================================================
	/**
	 * 正の無限大を表す定数です。インスタンス節約のためにクラス内で使用します。
	 * <p>
	 */
	private static final Rational POSITIVE_INFINITY = new Rational(1, 0);

	// ======================================================================
	// 負の無限大
	// ======================================================================
	/**
	 * 負の無限大を表す定数です。インスタンス節約のためにクラス内で使用します。
	 * <p>
	 */
	private static final Rational NEGATIVE_INFINITY = new Rational(-1, 0);

	// ======================================================================
	// 分子
	// ======================================================================
	/**
	 * 分数の分子です。
	 * <p>
	 */
	private final long numerator;

	// ======================================================================
	// 分母
	// ======================================================================
	/**
	 * 分数の分母です。
	 * <p>
	 */
	private final long denominator;

	// ======================================================================
	// コンストラクタ
	// ======================================================================
	/**
	 * 指定された分子/分母で有理数を作成します。
	 * <p>
	 * 分子/分母共に負の値を指定した場合は双方とも正の値に変換されます。また
	 * 分母のみに負の値を指定した場合、分母を正、分子を負となるよう設定され
	 * ます。
	 * <p>
	 * @param numerator 分子
	 * @param denominator 分母
	 */
	public Rational(long numerator, long denominator) {

		// 符号の調整
		if(numerator < 0 && denominator < 0){
			numerator = Math.abs(numerator);
			denominator = Math.abs(denominator);
		} else if(denominator < 0){
			numerator = - numerator;		// 分子のみが符号を持つように
			denominator = - denominator;
		}

		this.numerator = numerator;
		this.denominator = denominator;
		return;
	}

	// ======================================================================
	// 分子の参照
	// ======================================================================
	/**
	 * 分子を参照します。
	 * <p>
	 * @return 分子
	 */
	public long getNumerator() {
		return numerator;
	}

	// ======================================================================
	// 分母の参照
	// ======================================================================
	/**
	 * 分母を参照します。
	 * <p>
	 * @return 分母
	 */
	public long getDenominator() {
		return denominator;
	}

	// ======================================================================
	// 約分
	// ======================================================================
	/**
	 * この分数を約分します。
	 * <p>
	 * @return 約分した有理数
	 */
	public Rational reduction(){
		long n = getNumerator();
		long d = getDenominator();

		// 分子が 0 の場合
		if(n == 0){
			return (d == 0)? NaN: ZERO;
		}

		// 分母が 0 の場合
		if(d == 0){
			return (n < 0)? NEGATIVE_INFINITY: POSITIVE_INFINITY;
		}

		// 偶数なら 2 で割る
		while((n & 0x01) == 0 && (d & 0x01) == 0){
			n >>>= 1;
			d >>>= 1;
		}

		// 3 から奇数のみを対象に約分 (65535 までに制限)
		long min = Math.max(Math.min(Math.abs(n), Math.abs(d)), 65536);
		for(int i=3; i<=min && n > 1 && d > 1; i+=2){
			if((n % i) == 0 && (d % i) == 0){
				n /= i;
				d /= i;
				i -= 2;
			}
		}

		// インスタンス節約のため約分が行われていなければ自身を返す
		if(n == getNumerator()){
			return this;
		}

		// 約分された分数を返す
		return new Rational(n, d);
	}

	// ======================================================================
	// 絶対値の参照
	// ======================================================================
	/**
	 * この有理数の絶対値を参照します。
	 * <p>
	 * @return 有理数の絶対値
	 */
	public Rational abs(){
		assert(denominator >= 0);

		// 正の値を取っている場合
		if(getNumerator() >= 0){
			return this;
		}

		// 負の値を取っている場合
		return new Rational(- getNumerator(), getDenominator());
	}

	// ======================================================================
	// NaN の判定
	// ======================================================================
	/**
	 * この有理数の分子/分母が共に 0 かを判定します。
	 * <p>
	 * @return NaN の場合 true
	 */
	public boolean isNaN(){
		return getNumerator() == 0 && getDenominator() == 0;
	}

	// ======================================================================
	// 負値の判定
	// ======================================================================
	/**
	 * この有理数が負かどうかを判定します。
	 * <p>
	 * @return 負の場合 true
	 */
	public boolean isNegative(){
		assert(denominator >= 0);
		return getNumerator() < 0;
	}

	// ======================================================================
	// 正の無限大判定
	// ======================================================================
	/**
	 * この有理数が正の無限大かどうかを判定します。
	 * <p>
	 * @return 正の無限大の場合 true
	 */
	public boolean isPositiveInfinity(){
		return getDenominator() == 0 && getNumerator() > 0;
	}

	// ======================================================================
	// 負の無限大判定
	// ======================================================================
	/**
	 * この有理数が負の無限大かどうかを判定します。
	 * <p>
	 * @return 負の無限大の場合 true
	 */
	public boolean isNegativeInfinity(){
		return getDenominator() == 0 && getNumerator() < 0;
	}

	// ======================================================================
	// 整数値の参照
	// ======================================================================
	/**
	 * この有理数を実数化した時の整数値を参照します。
	 * <p>
	 * @return 整数値
	*/
	@Override
	public int intValue() {
		return (int)doubleValue();
	}

	// ======================================================================
	// 整数値の参照
	// ======================================================================
	/**
	 * この有理数を実数化した時の整数値を参照します。
	 * <p>
	 * @return 整数値
	*/
	@Override
	public long longValue() {
		return (long)doubleValue();
	}

	// ======================================================================
	// 単精度浮動小数点の参照
	// ======================================================================
	/**
	 * この有理数を実数化した時の単精度浮動小数点値を参照します。
	 * <p>
	 * @return 単精度浮動小数点の値
	*/
	@Override
	public float floatValue() {
		return (float)doubleValue();
	}

	// ======================================================================
	// 実数値の変換
	// ======================================================================
	/**
	 * この分数を実数値に変換します。分母が 0 の場合は NaN を返します。
	 * <p>
	 * @return 実数値
	 */
	@Override
	public double doubleValue(){

		// 割り算可能な場合
		if(denominator != 0){
			return (double)numerator / (double)denominator;
		}

		// 0/0 (NaN) の場合
		if(numerator == 0){
			return Double.NaN;
		}

		// 負の無限大の場合
		if(numerator < 0){
			return Double.NEGATIVE_INFINITY;
		}

		// 正の無限大の場合
		return Double.POSITIVE_INFINITY;
	}

	// ======================================================================
	// ハッシュ値の参照
	// ======================================================================
	/**
	 * ハッシュ値を参照します。
	 * <p>
	 * @return ハッシュ値
	*/
	@Override
	public int hashCode(){
		return (int)(getNumerator() + getDenominator());
	}

	// ======================================================================
	// 等価性の評価
	// ======================================================================
	/**
	 * 指定されたインスタンスとこのインスタンスが等しいかどうかを評価します。
	 * <p>
	 * @param obj 比較するオブジェクト
	 * @return 等しい場合 true
	*/
	@Override
	public boolean equals(Object obj){
		if(! (obj instanceof Rational)){
			return false;
		}
		Rational other = (Rational)obj;
		return this.getNumerator() == other.getNumerator()
			&& this.getDenominator() == other.getDenominator();
	}

	// ======================================================================
	// インスタンスの文字列化
	// ======================================================================
	/**
	 * このインスタンスを文字列化します。
	 * <p>
	 * @return インスタンスの文字列
	*/
	@Override
	public String toString() {
		long n = getNumerator();
		long d = getDenominator();

		// 分母が 0 の場合
		if(d == 0){
			if(n == 0){
				return String.valueOf(Double.NaN);
			}
			if(n > 0){
				return String.valueOf(Double.POSITIVE_INFINITY);
			}
			return String.valueOf(Double.NEGATIVE_INFINITY);
		}

		// 分母が 1 の場合は分母を省略
		if(d == 1){
			return String.valueOf(n);
		}

		// 分子が 0 の場合は必ず 0
		if(n == 0){
			return "0";
		}

		// それ以外の場合
		return n + "/" + d;
	}

}
