/* **************************************************************************
 * Copyright (C) 2008 BJoRFUAN. All Right Reserved
 * **************************************************************************
 * This module, contains source code, binary and documentation, is in the
 * BSD License, and comes with NO WARRANTY.
 * 
 *                                                 torao <torao@bjorfuan.com>
 *                                                       http://www.moyo.biz/
 * $Id: Photograph.java,v 1.3 2008/08/26 05:17:20 torao Exp $
*/
package org.koiroha.fixez;

import java.text.*;
import java.util.*;
import java.util.logging.Logger;


// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// Photograph: 撮影情報アクセスユーティリティ
// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
/**
 * {@link Exif} から撮影関係の一般情報にアクセスするためのユーティリティクラスです。
 * <p>
 * @version fixez 1.0 - $Revision: 1.3 $ $Date: 2008/08/26 05:17:20 $
 * @author <a href="mailto:torao@mars.dti.ne.jp">torao</a>
 * @since fixez 1.0 - 2008/08/11
 */
public final class Photograph {

	// ======================================================================
	// ログ出力先
	// ======================================================================
	/**
	 * このクラスのログ出力先です。
	 * <p>
	 */
	private static final Logger logger = Logger.getLogger(Photograph.class.getName());

	// ======================================================================
	// Exif 情報
	// ======================================================================
	/**
	 * Exif 情報です。
	 * <p>
	 */
	private final Exif exif;

	// ======================================================================
	// コンストラクタ
	// ======================================================================
	/**
	 * Exif 情報を指定して構築を行います。
	 * <p>
	 * @param exif Exif 情報
	 */
	public Photograph(Exif exif) {
		this.exif = exif;
		return;
	}

	// ======================================================================
	// 撮影日時の参照
	// ======================================================================
	/**
	 * 撮影日時を参照するための簡易メソッドです。源画像データの作成日時または
	 * デジタルデータの作成日時を参照します。日時が不明な場合には null を返し
	 * ます。
	 * <p>
	 * @return 撮影日時
	*/
	public Date getDate(){
		Date date = null;
		DateFormat pf = new SimpleDateFormat("yyyy:MM:dd HH:mm:ss");
		ExifField field = exif.getField(IFD.EXIF, 0x9003);	// 源画像データの作成日時
		if(field == null){
			field = exif.getField(IFD.EXIF, 0x9004);			// デジタルデータの作成日時
		}
		if(field == null){
			field = exif.getField(IFD.I0TH, 0x0132);			// ファイル更新日時
		}
		if(field != null){
			try{
				date = pf.parse(field.getString());
			} catch(ParseException ex){/* */}
		}
		return date;
	}

	// ======================================================================
	// メーカー名の参照
	// ======================================================================
	/**
	 * 撮影に使用された機器のメーカー名を参照します。メーカー名が不明な場合は
	 * nullを返します。
	 * <p>
	 * @return 機器のメーカー名
	*/
	public String getMake(){

		// メーカーの参照
		ExifField maker = exif.getField(IFD.I0TH, Tag.I0TH_MAKE);
		if(maker == null){
			return null;
		}

		return maker.getString().trim();
	}

	// ======================================================================
	// モデル名の参照
	// ======================================================================
	/**
	 * 撮影に使用された機器のモデル名を参照します。モデルが不明な場合は null
	 * を返します。
	 * <p>
	 * @return 機器のモデル名
	*/
	public String getModel(){

		// モデルの参照
		ExifField model = exif.getField(IFD.I0TH, Tag.I0TH_MODEL);
		if(model == null){
			return null;
		}

		return model.getString().trim();
	}

	// ======================================================================
	// 露光時間の参照
	// ======================================================================
	/**
	 * 撮影時の露光時間を参照します。露光時間が不明な場合は null を返します。
	 * <p>
	 * @return 露光時間
	*/
	public Rational getExposureTime(){

		// 露光時間の参照
		ExifField exposure = exif.getField(IFD.EXIF, Tag.EXIF_EXPOSURE_TIME);
		if(exposure == null){
			return null;
		}

		return exposure.getRational();
	}

	// ======================================================================
	// 露光補正量の参照
	// ======================================================================
	/**
	 * 撮影時の露光補正量を APEX 値 (EV) で参照します。露光補正量が不明な場合
	 * は null を返します。
	 * <p>
	 * @return 露光補正量
	*/
	public Rational getExposureBiasValue(){

		// 露光補正量の参照
		ExifField bias = exif.getField(IFD.EXIF, Tag.EXIF_EXPOSURE_BIAS_VALUE);
		if(bias == null){
			return null;
		}

		return bias.getRational();
	}

	// ======================================================================
	// F 値の参照
	// ======================================================================
	/**
	 * 撮影時の F 値 (絞り) を参照します。F 値が定義されていない場合は NaN を
	 * 返します。
	 * <p>
	 * @return F値
	*/
	public double getFNumber(){

		// Ｆ値の参照
		ExifField f = exif.getField(IFD.EXIF, Tag.EXIF_F_NUMBER);
		if(f == null){
			return Double.NaN;
		}
		
		return f.getDouble();
	}

	// ======================================================================
	// 焦点距離の参照
	// ======================================================================
	/**
	 * 撮影時の焦点距離をミリメートル単位の数値として参照します。焦点距離が
	 * 定義されていない場合は NaN を返します。
	 * <p>
	 * @return 焦点距離 (ミリメートル)
	*/
	public double getFocalLength(){

		// 焦点距離の参照
		ExifField focus = exif.getField(IFD.EXIF, Tag.EXIF_FOCAL_LENGTH);
		if(focus == null){
			return Double.NaN;
		}

		return focus.getDouble();
	}

	// ======================================================================
	// ISO 感度の参照
	// ======================================================================
	/**
	 * 撮影時の ISO 感度を参照します。不明な場合は負の値を返します。
	 * <p>
	 * @return ISO感度
	*/
	public int getISOSpeedRatings(){

		// ISO感度の参照
		ExifField iso = exif.getField(IFD.EXIF, Tag.EXIF_ISO_SPEED_RATINGS);
		if(iso == null)	{
			return -1;
		}

		return iso.getInt();
	}

	// ======================================================================
	// フラッシュの参照
	// ======================================================================
	/**
	 * 撮影時のフラッシュを参照します。フラッシュの状況が不明な場合は null を
	 * 返します。
	 * <p>
	 * @return フラッシュ
	*/
	public Flash getFlash(){

		// フラッシュの参照
		ExifField flash = exif.getField(IFD.EXIF, Tag.EXIF_FLASH);
		if(flash == null){
			return null;
		}


		// 列挙型の参照
		Flash fl = Flash.valueOf(flash.getInt() & 0x07);
		if(fl == null){
			logger.warning("unsupported flash: 0x" + Integer.toHexString(flash.getInt()) + ": " + flash + ": " + flash.getTagLabel());
			return null;
		}
		return fl;
	}

	// ======================================================================
	// 光源の参照
	// ======================================================================
	/**
	 * 撮影時の光源を参照します。
	 * <p>
	 * @return 光源
	*/
	public LightSource getLightSource(){

		// 光源の参照
		ExifField light = exif.getField(IFD.EXIF, Tag.EXIF_LIGHT_SOURCE);
		if(light == null){
			return LightSource.UNKNOWN;
		}

		// 列挙型の参照
		LightSource ls = LightSource.valueOf(light.getInt());
		if(ls == null){
			logger.warning("unsupported light-source: 0x" + Integer.toHexString(light.getInt()));
			return LightSource.UNKNOWN;
		}
		return ls;
	}

	// ======================================================================
	// 採光方式の参照
	// ======================================================================
	/**
	 * 撮影時の採光方式を参照します。
	 * <p>
	 * @return 採光方式
	*/
	public MeteringMode getMeteringMode(){

		// 採光方式の参照
		ExifField metering = exif.getField(IFD.EXIF, Tag.EXIF_METERING_MODE);
		if(metering == null){
			return MeteringMode.UNKNOWN;
		}

		// 列挙型の参照
		MeteringMode m = MeteringMode.valueOf(metering.getInt());
		if(m == null){
			logger.warning("unsupported metering-mode: 0x" + Integer.toHexString(metering.getInt()));
			return MeteringMode.UNKNOWN;
		}
		return m;
	}

	// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	// Flash: フラッシュ
	// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	/**
	 * フラッシュの列挙です。
	 * <p>
	 * @version $Revision: 1.3 $
	 * @author torao
	 * @since 2008/08/10 Java SE 6
	 */
	public enum Flash {

		// ==================================================================
		// [フラッシュ] 発光なし
		// ==================================================================
		/**
		 * ストロボが発光しなかったことを示す定数です。
		 * <p>
		*/
		NOT_FIRED(0, false, "なし"),

		// ==================================================================
		// [フラッシュ] 発光あり
		// ==================================================================
		/**
		 * ストロボが発光した事を示す (反射光検出機構なし) 定数です。
		 * <p>
		*/
		FIRED(1, true, "あり"),

		// ==================================================================
		// [フラッシュ] 反射光未検出
		// ==================================================================
		/**
		 * ストロボが発光したが反射光を検出できなかったことを示す定数です。
		 * <p>
		*/
		RETURN_LIGHT_NOT_DETECTED(5, true, "あり (反射光未検出)"),

		// ==================================================================
		// [フラッシュ] 反射光未検出
		// ==================================================================
		/**
		 * ストロボが発光し反射光を検出したことを示す定数です。
		 * <p>
		*/
		RETURN_LIGHT_DETECTED(7, true, "あり (反射光検出)"),

		;

		// ==================================================================
		// 数値
		// ==================================================================
		/**
		 * この採光方式の値です。
		 * <p>
		*/
		private final int value;

		// ==================================================================
		// 表示名
		// ==================================================================
		/**
		 * 採光方式の表示名です。
		 * <p>
		*/
		private final boolean fired;

		// ==================================================================
		// 表示名
		// ==================================================================
		/**
		 * 採光方式の表示名です。
		 * <p>
		*/
		private final String label;

		// ==================================================================
		// コンストラクタ
		// ==================================================================
		/**
		 * 数値を指定して構築を行います。
		 * <p>
		 * @param num タイプの値
		 * @param fired フラッシュが焚かれたか
		 * @param label 表示名
		*/
		private Flash(int num, boolean fired, String label){
			this.value = num;
			this.fired = fired;
			this.label = label;
			return;
		}

		// ==================================================================
		// フラッシュ判定
		// ==================================================================
		/**
		 * フラッシュが焚かれたかどうかを判定します。
		 * <p>
		 * @return フラッシュが焚かれた場合 true
		*/
		public boolean isFired(){
			return fired;
		}

		// ==================================================================
		// ラベルの参照
		// ==================================================================
		/**
		 * このフラッシュの表示用ラベルを参照します。
		 * <p>
		 * @return フラッシュのラベル
		*/
		public String getLabel(){
			return label;
		}

		// ==================================================================
		// インスタンスの参照
		// ==================================================================
		/**
		 * 指定されたフラッシュの値に対するインスタンスを参照します。該当する定数
		 * が定義されていない場合は null を返します。
		 * <p>
		 * @param num タイプの値
		 * @return タイプ
		*/
		public static Flash valueOf(int num){
			for(Flash mode: Flash.values()){
				if(mode.value == num){
					return mode;
				}
			}
			return null;
		}

	}

	// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	// LightSource: 光源
	// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	/**
	 * 光源の列挙です。
	 * <p>
	 * @version $Revision: 1.3 $
	 * @author torao
	 * @since 2008/08/10 Java SE 6
	 */
	public enum LightSource {

		// ==================================================================
		// [光源] 不明
		// ==================================================================
		/**
		 * 不明を表す採光方式です。
		 * <p>
		*/
		UNKNOWN(0, "不明"),

		// ==================================================================
		// [光源] 昼光
		// ==================================================================
		/**
		 * 昼光を表す採光方式です。
		 * <p>
		*/
		DAYLIGHT(1, "昼光"),

		// ==================================================================
		// [光源] 蛍光灯
		// ==================================================================
		/**
		 * 蛍光灯を表す採光方式です。
		 * <p>
		*/
		FLUORESCENT(2, "蛍光灯"),

		// ==================================================================
		// [光源] タングステン
		// ==================================================================
		/**
		 * タングステンを表す採光方式です。
		 * <p>
		*/
		TUNGSTEN(3, "タングステン"),

		// ==================================================================
		// [光源] フラッシュ
		// ==================================================================
		/**
		 * フラッシュを表す採光方式です。
		 * <p>
		*/
		FLASH(10, "フラッシュ"),

		// ==================================================================
		// [光源] 標準光A
		// ==================================================================
		/**
		 * 標準光Aを表す採光方式です。
		 * <p>
		*/
		STANdARD_LIGHT_A(17, "標準光 A"),

		// ==================================================================
		// [光源] 標準光B
		// ==================================================================
		/**
		 * 標準光Bを表す採光方式です。
		 * <p>
		*/
		STANDARD_LIGHT_B(18, "標準光 B"),

		// ==================================================================
		// [光源] 標準光C
		// ==================================================================
		/**
		 * 標準光Cを表す採光方式です。
		 * <p>
		*/
		STANDARD_LIGHT_C(19, "標準光 C"),

		// ==================================================================
		// [光源] D55
		// ==================================================================
		/**
		 * D55 を表す採光方式です。
		 * <p>
		*/
		D55(20, "D55"),

		// ==================================================================
		// [光源] D65
		// ==================================================================
		/**
		 * D65を表す採光方式です。
		 * <p>
		*/
		D65(21, "D65"),

		// ==================================================================
		// [光源] D75
		// ==================================================================
		/**
		 * D75を表す採光方式です。
		 * <p>
		*/
		D75(22, "D75"),

		// ==================================================================
		// [光源] その他
		// ==================================================================
		/**
		 * その他を表す採光方式です。
		 * <p>
		*/
		OTHER(255, "その他"),

		;

		// ==================================================================
		// 数値
		// ==================================================================
		/**
		 * この採光方式の値です。
		 * <p>
		*/
		private final int value;

		// ==================================================================
		// 表示名
		// ==================================================================
		/**
		 * 採光方式の表示名です。
		 * <p>
		*/
		private final String label;

		// ==================================================================
		// コンストラクタ
		// ==================================================================
		/**
		 * 数値を指定して構築を行います。
		 * <p>
		 * @param num タイプの値
		 * @param label 表示名
		*/
		private LightSource(int num, String label){
			this.value = num;
			this.label = label;
			return;
		}

		// ==================================================================
		// ラベルの参照
		// ==================================================================
		/**
		 * この採光方式の表示用ラベルを参照します。
		 * <p>
		 * @return 採光方式のラベル
		*/
		public String getLabel(){
			return label;
		}

		// ==================================================================
		// インスタンスの参照
		// ==================================================================
		/**
		 * 指定されたタイプの値に対するインスタンスを参照します。該当する定数が
		 * 定義されていない場合は null を返します。
		 * <p>
		 * @param num タイプの値
		 * @return タイプ
		*/
		public static LightSource valueOf(int num){
			for(LightSource mode: LightSource.values()){
				if(mode.value == num){
					return mode;
				}
			}
			return null;
		}

	}

	// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	// MeteringMode: 採光方式
	// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	/**
	 * 採光方式の列挙です。
	 * <p>
	 * @version $Revision: 1.3 $
	 * @author torao
	 * @since 2008/08/10 Java SE 6
	 */
	public enum MeteringMode {

		// ==================================================================
		// [採光方式] 不明
		// ==================================================================
		/**
		 * 不明を表す採光方式です。
		 * <p>
		*/
		UNKNOWN(0, "不明"),

		// ==================================================================
		// [採光方式] 平均
		// ==================================================================
		/**
		 * 平均を表す採光方式です。
		 * <p>
		*/
		AVERAGE(1, "平均"),

		// ==================================================================
		// [採光方式] 中央重点
		// ==================================================================
		/**
		 * 中央重点を表す採光方式です。
		 * <p>
		*/
		CENTER_WEIGHTED_AVERAGE(2, "中央重点"),

		// ==================================================================
		// [採光方式] スポット
		// ==================================================================
		/**
		 * スポットを表す採光方式です。
		 * <p>
		*/
		SPOT(3, "スポット"),

		// ==================================================================
		// [採光方式] マルチスポット
		// ==================================================================
		/**
		 * マルチスポットを表す採光方式です。
		 * <p>
		*/
		MULTI_SPOT(4, "マルチスポット"),

		// ==================================================================
		// [採光方式] 分割測光
		// ==================================================================
		/**
		 * 分割測光を表す採光方式です。
		 * <p>
		*/
		MULTI_SEGMENT(5, "分割測光"),

		// ==================================================================
		// [採光方式] 部分測光
		// ==================================================================
		/**
		 * 部分測光を表す採光方式です。
		 * <p>
		*/
		PARTIAL(6, "部分測光"),

		// ==================================================================
		// [採光方式] その他
		// ==================================================================
		/**
		 * その他を表す採光方式です。
		 * <p>
		*/
		OTHER(255, "その他"),

		;

		// ==================================================================
		// 数値
		// ==================================================================
		/**
		 * この採光方式の値です。
		 * <p>
		*/
		private final int value;

		// ==================================================================
		// 表示名
		// ==================================================================
		/**
		 * 採光方式の表示名です。
		 * <p>
		*/
		private final String label;

		// ==================================================================
		// コンストラクタ
		// ==================================================================
		/**
		 * 数値を指定して構築を行います。
		 * <p>
		 * @param num タイプの値
		 * @param label 表示名
		*/
		private MeteringMode(int num, String label){
			this.value = num;
			this.label = label;
			return;
		}

		// ==================================================================
		// ラベルの参照
		// ==================================================================
		/**
		 * この採光方式の表示用ラベルを参照します。
		 * <p>
		 * @return 採光方式のラベル
		*/
		public String getLabel(){
			return label;
		}

		// ==================================================================
		// インスタンスの参照
		// ==================================================================
		/**
		 * 指定されたタイプの値に対するインスタンスを参照します。該当する定数が
		 * 定義されていない場合は null を返します。
		 * <p>
		 * @param num タイプの値
		 * @return タイプ
		*/
		public static MeteringMode valueOf(int num){
			for(MeteringMode mode: MeteringMode.values()){
				if(mode.value == num){
					return mode;
				}
			}
			return null;
		}

	}

}
