/* **************************************************************************
 * Copyright (C) 2008 BJoRFUAN. All Right Reserved
 * **************************************************************************
 * This module, contains source code, binary and documentation, is in the
 * BSD License, and comes with NO WARRANTY.
 * 
 *                                                 torao <torao@bjorfuan.com>
 *                                                       http://www.moyo.biz/
 * $Id: GPS.java,v 1.5 2009/03/28 08:27:46 torao Exp $
*/
package org.koiroha.fixez;

import java.text.MessageFormat;


// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// GPS: GPS 情報アクセスユーティリティ
// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
/**
 * {@link Exif} から GPS 関係の情報にアクセスするためのユーティリティクラスで
 * す。
 * <p>
 * @version fixez 1.0 - $Revision: 1.5 $ $Date: 2009/03/28 08:27:46 $
 * @author <a href="mailto:torao@mars.dti.ne.jp">torao</a>
 * @since fixez 1.0 - 2008/08/11
 */
public final class GPS {

	// ======================================================================
	// Exif 情報
	// ======================================================================
	/**
	 * Exif 情報です。
	 * <p>
	 */
	private final Exif exif;

	// ======================================================================
	// コンストラクタ
	// ======================================================================
	/**
	 * Exif 情報を指定して構築を行います。
	 * <p>
	 * @param exif Exif 情報
	 */
	public GPS(Exif exif) {
		this.exif = exif;
		return;
	}

	// ======================================================================
	// GPS 緯度の参照
	// ======================================================================
	/**
	 * 撮影時の GPS 緯度を返します。返値の緯度は赤道を 0°として北に +、
	 * 南に - をとります。緯度が不明な場合には null を返します。
	 * <p>
	 * @return GPS 緯度
	*/
	public Degree getLatitude(){

		// 緯度の取得
		ExifField lat1 = exif.getField(IFD.GPS, Tag.GPS_LATITUDE_REF);	// 北緯/南緯記号
		ExifField lat2 = exif.getField(IFD.GPS, Tag.GPS_LATITUDE);		// 北緯/南緯
		if(lat1 == null || lat2 == null){
			return null;
		}

		double degree = getDegree(lat2.getRationals());
		if(lat1.getString().equalsIgnoreCase("N")){
			degree = Math.abs(degree);
		} else if(lat1.getString().equalsIgnoreCase("S")){
			degree = - Math.abs(degree);
		} else {
			return null;
		}
		return new Degree(degree);
	}

	// ======================================================================
	// GPS 経度の参照
	// ======================================================================
	/**
	 * 撮影時の GPS 経度を取得します。返値の経度は本初子午線を 0°として
	 * 東に +、西に - をとります。経度が不明な場合には null を返します。
	 * <p>
	 * @return GPS 経度
	*/
	public Degree getLongitude(){

		// 経度の取得
		ExifField lon1 = exif.getField(IFD.GPS, Tag.GPS_LONGITUDE_REF);	// 東経/西経記号
		ExifField lon2 = exif.getField(IFD.GPS, Tag.GPS_LONGITUDE);	// 東経/西経
		if(lon1 == null || lon2 == null){
			return null;
		}

		double degree = getDegree(lon2.getRationals());
		if(lon1.getString().equalsIgnoreCase("E")){
			degree = Math.abs(degree);
		} else if(lon1.getString().equalsIgnoreCase("W")){
			degree = - Math.abs(degree);
		} else {
			return null;
		}
		return new Degree(degree);
	}

	// ======================================================================
	// GPS測地系の参照
	// ======================================================================
	/**
	 * 撮影時のGPS測地系を参照します。測地系が不明な場合には null を返し
	 * ます。
	 * <p>
	 * @return 測地系
	*/
	public String getDatum(){

		// 測地系の参照
		ExifField datum = exif.getField(IFD.GPS, Tag.GPS_MAP_DATUM);
		if(datum == null){
			return null;
		}

		return datum.getString();
	}

	// ======================================================================
	// 度分秒の参照
	// ======================================================================
	/**
	 * 指定された分数を度分秒に変換します。
	 * <p>
	 * @param r 分数の配列
	 * @return 度分秒を表す文字列
	*/
	private static double getDegree(Rational[] r){
		assert(r.length == 3);
		return r[0].doubleValue()
			+ (r[1].doubleValue() / 60.0)
			+ (r[2].doubleValue() / 60.0 / 60.0);
	}

	// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	// Degree: 度分秒クラス
	// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	/**
	 * 度分秒を表すクラスです。
	 * <p>
	 * @version $Revision: 1.5 $
	 * @author torao
	 * @since 2008/08/10 Java SE 6
	 */
	public static final class Degree {

		// ==================================================================
		// 度分秒
		// ==================================================================
		/**
		 * 度分秒です。
		 * <p>
		 */
		private final double degree;

		// ==================================================================
		// コンストラクタ
		// ==================================================================
		/**
		 * 実数値での度数を指定して構築を行います。
		 * <p>
		 * @param degree 度数
		 */
		public Degree(double degree) {
			this.degree = degree;
			return;
		}

		// ==================================================================
		// 度の参照
		// ==================================================================
		/**
		 * 符号付の度数を参照します。
		 * <p>
		 * @return 度数
		 */
		public int getDegree(){
			return (int)degree;
		}

		// ==================================================================
		// 分の参照
		// ==================================================================
		/**
		 * 分数を参照します。符号は付きません。
		 * <p>
		 * @return 分数
		 */
		public int getMinute(){
			return (int)((Math.abs(degree) * 60.0) % 60.0);
		}

		// ==================================================================
		// 秒の参照
		// ==================================================================
		/**
		 * 秒と小数点以下を参照します。符号は付きません。
		 * <p>
		 * @return 秒数
		 */
		public double getSecond(){
			return (Math.abs(degree) * 60.0 * 60.0) % 60.0;
		}

		// ==================================================================
		// 度分秒の参照
		// ==================================================================
		/**
		 * 度数を実数で返します。分秒は返値の小数点以下に含まれています。
		 * <p>
		 * @return 度分秒
		 */
		public double toDouble(){
			return degree;
		}

		// ==================================================================
		// ハッシュ値の参照
		// ==================================================================
		/**
		 * ハッシュ値を参照します。
		 * <p>
		 * @return ハッシュ値
		 */
		@Override
		public int hashCode(){
			return (int)(degree * 60 * 60);
		}

		// ==================================================================
		// 等価性の比較
		// ==================================================================
		/**
		 * 指定されたオブジェクトとこのオブジェクトが等しいかを判定します。
		 * <p>
		 * @return 等しい場合 true
		 */
		@Override
		public boolean equals(Object obj){
			if(! (obj instanceof Degree)){
				return false;
			}
			return (this.toDouble() == ((Degree)obj).toDouble());
		}

		// ==================================================================
		// インスタンスの文字列化
		// ==================================================================
		/**
		 * このインスタンスを文字列化します。
		 * <p>
		 * @return インスタンスの文字列
		 */
		@Override
		public String toString(){
			return MessageFormat.format(
				"{0,number,0}:{1,number,00}:{2,number,00.00}",
				getDegree(), getMinute(), getSecond());
		}

	}

}
