/* **************************************************************************
 * Copyright (C) 2004 BJORFUAN. All Right Reserved
 * **************************************************************************
 *                                               ToRA. <torao@@mars.dti.ne.jp>
 *                                                       http://www.moyo.biz/
 * $Id: ExifField.java,v 1.3 2008/08/11 17:28:49 torao Exp $
*/
package org.koiroha.fixez;

import java.io.*;
import java.util.*;



// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// ExifField: Exif フィールドクラス
// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
/**
 * Exif フィールド情報を表す不変クラスです。
 * <p>
 * @version fixez 1.0 - $Revision: 1.3 $ $Date: 2008/08/11 17:28:49 $
 * @author <a href="mailto:torao@mars.dti.ne.jp">torao</a>
 * @since fixez 1.0 - 2005/06/02
*/
public class ExifField implements Serializable{

	// ======================================================================
	// シリアルバージョン
	// ======================================================================
	/**
	 * このクラスのシリアルバージョンです。
	 * <p>
	*/
	private static final long serialVersionUID = 1L;

	// ======================================================================
	// フィールド名
	// ======================================================================
	/**
	 * IFD 識別子、タグに対する名前です。(idf << 16 | tag) をキーとし、対応す
	 * るフィールド名文字列が保存されます。
	 * <p>
	*/
	private static final Map<Integer,String> FIELD_NAME = new HashMap<Integer,String>();

	// ======================================================================
	// フィールドラベル
	// ======================================================================
	/**
	 * IFD 識別子、タグに対するラベルです。(idf << 16 | tag) をキーとし、対応
	 * するラベル文字列が保存されます。
	 * <p>
	*/
	private static final Map<Integer,String> FIELD_LABEL = new HashMap<Integer,String>();

	// ======================================================================
	// IFD 識別子
	// ======================================================================
	/**
	 * このフィールドの IFD 識別子です。
	 * <p>
	*/
	private final IFD ifd;

	// ======================================================================
	// タグ
	// ======================================================================
	/**
	 * このフィールドのタグです。
	 * <p>
	*/
	private final int tag;

	// ======================================================================
	// タイプ
	// ======================================================================
	/**
	 * このフィールドのタイプです。
	 * <p>
	*/
	private final Type type;

	// ======================================================================
	// 繰り返し数
	// ======================================================================
	/**
	 * このフィールドの繰り返し数です。
	 * <p>
	*/
	private final int count;

	// ======================================================================
	// 文字列データ
	// ======================================================================
	/**
	 * このフィールドの文字列データです。
	 * <p>
	*/
	private final String text;

	// ======================================================================
	// バイナリデータ
	// ======================================================================
	/**
	 * このフィールドのバイナリデータです。
	 * <p>
	*/
	private final byte[] binary;

	// ======================================================================
	// 数値
	// ======================================================================
	/**
	 * このフィールドのバイナリデータです。
	 * <p>
	*/
	private final Number[] number;

	// ======================================================================
	// コンストラクタ
	// ======================================================================
	/**
	 * 指定された情報で Exif フィールドを構築します。
	 * <p>
	 * @param ifd IFD
	 * @param tag タグ値
	 * @param type タイプ
	 * @param count 繰り返し数
	 * @param text 文字列値
	 * @param binary バイナリ値
	 * @param number 数値配列 (null 可)
	 */
	ExifField(IFD ifd, int tag, Type type, int count, String text, byte[] binary, Number[] number) {
		this.ifd = ifd;
		this.tag = tag;
		this.type = type;
		this.count = count;
		this.text = text;
		this.binary = binary;
		this.number = number;
		return;
	}

	// ======================================================================
	// IFD 識別子の参照
	// ======================================================================
	/**
	 * このフィールドの IFD 識別子を参照します。
	 * <p>
	 * @return IFD 識別子
	*/
	public IFD getIFD(){
		return ifd;
	}

	// ======================================================================
	// タグの参照
	// ======================================================================
	/**
	 * このフィールドのタグを参照します。
	 * <p>
	 * @return タグ
	*/
	public int getTag(){
		return tag;
	}

	// ======================================================================
	// タグ名の参照
	// ======================================================================
	/**
	 * このフィールドのタグ名を参照します。タグ名はタグをユニークに識別する
	 * ものではありません。
	 * <p>
	 * @return タグ名
	*/
	public String getTagName(){
		int key = getKey(ifd, tag);
		String name = FIELD_NAME.get(key);
		if(name == null){
			return hex(getIFD().getDigit()) + "." + hex(getTag());
		}
		return name;
	}

	// ======================================================================
	// タグラベルの参照
	// ======================================================================
	/**
	 * このフィールドのタグラベルを参照します。タグ名はタグをユニークに識別す
	 * るものではありません。
	 * <p>
	 * @return タグラベル
	*/
	public String getTagLabel(){
		int key = getKey(ifd, tag);
		String label = FIELD_LABEL.get(key);
		if(label == null){
			return "";
		}
		return label;
	}

	// ======================================================================
	// タイプの参照
	// ======================================================================
	/**
	 * このフィールドのタイプを参照します。
	 * <p>
	 * @return タイプ
	*/
	public Type getType(){
		return type;
	}

	// ======================================================================
	// 繰り返し数の参照
	// ======================================================================
	/**
	 * このフィールドの繰り返し数を参照します。
	 * <p>
	 * @return 繰り返し数
	*/
	public int getCount(){
		return count;
	}

	// ======================================================================
	// 文字列値の参照
	// ======================================================================
	/**
	 * このフィールドの値を文字列として参照します。{@link Type#ASCII} 型に
	 * 最適化された値参照用のメソッドです。全ての数値型または
	 * {@link Type#UNDEFINED} 型に対しても表現的な文字列値を参照することが
	 * 出来ます。
	 * <p>
	 * @return フィールドの文字列値
	*/
	public String getString(){
		return text;
	}

	// ======================================================================
	// 整数値の参照
	// ======================================================================
	/**
	 * このフィールドの値を整数値として参照します。{@link Type#BYTE},
	 * {@link Type#USHORT}, {@link Type#SHORT}, {@link Type#ULONG},
	 * {@link Type#LONG} 型に最適化された値参照用のメソッドです。実数型および
	 * 有理数型に対しても値を整数として参照することが出来ます。非数値型に対し
	 * ては 0 を返します。
	 * <p>
	 * 型が繰り返しを持つ配列の場合はその先頭の値を返します。長さ 0 の配列の
	 * 場合は 0 を返します。
	 * <p>
	 * @return フィールドの整数値
	*/
	public int getInt(){
		if(number == null || number.length == 0){
			return 0;
		}
		return number[0].intValue();
	}

	// ======================================================================
	// 整数値配列の参照
	// ======================================================================
	/**
	 * 繰り返しを持つフィールドに対して全ての値を整数値配列として参照します。
	 * {@link Type#BYTE}, {@link Type#USHORT}, {@link Type#SHORT},
	 * {@link Type#ULONG}, {@link Type#LONG} 型に最適化された値参照用のメソッ
	 * ドです。実数型および有理数型に対しても値を整数配列として参照することが
	 * 出来ます。非数値型に対しては長さ 0 の配列を返します。
	 * <p>
	 * @return フィールドの整数値配列
	*/
	public int[] getInts(){
		if(number == null || number.length == 0){
			return new int[0];
		}
		int[] num = new int[number.length];
		for(int i=0; i<number.length; i++){
			num[i] = number[i].intValue();
		}
		return num;
	}

	// ======================================================================
	// 実数値の参照
	// ======================================================================
	/**
	 * このフィールドの値を実数値として参照します。{@link Type#SFLOAT},
	 * {@link Type#DFLOAT} 型に最適化された値参照用のメソッドです。整数型およ
	 * び有理数型に対しても値を実数として参照することが出来ます。非数値型に対し
	 * ては {@link Double#NaN} を返します。
	 * <p>
	 * 型が繰り返しを持つ配列の場合はその先頭の値を返します。長さ 0 の配列の
	 * 場合は {@link Double#NaN} を返します。
	 * <p>
	 * @return フィールドの実数値
	*/
	public double getDouble(){
		double[] d = getDoubles();
		if(d.length == 0){
			return Double.NaN;
		}
		return d[0];
	}

	// ======================================================================
	// 実数値の参照
	// ======================================================================
	/**
	 * 繰り返しを持つフィールドに対して全ての値を実数値配列として参照します。
	 * {@link Type#SFLOAT}, {@link Type#DFLOAT} 型に最適化された値参照用のメ
	 * ソッドです。整数型および有理数型に対しても値を実数配列として参照する
	 * ことが出来ます。非数値型に対しては長さ 0 の配列を返します。
	 * <p>
	 * @return フィールドの実数値
	*/
	public double[] getDoubles(){
		if(number == null || number.length == 0){
			return new double[0];
		}
		double[] num = new double[number.length];
		for(int i=0; i<number.length; i++){
			num[i] = number[i].doubleValue();
		}
		return num;
	}

	// ======================================================================
	// 有理数値の参照
	// ======================================================================
	/**
	 * このフィールドの値を有理数値として参照します。{@link Type#URATIONAL},
	 * {@link Type#RATIONAL} 型に最適化された値参照用のメソッドです。整数型に
	 * 対しても値を有理数として参照することが出来ます。実数または非数値型に対
	 * しては {@link Rational#NaN} を返します。
	 * <p>
	 * 型が繰り返しを持つ配列の場合はその先頭の値を返します。長さ 0 の配列の
	 * 場合は {@link Rational#NaN} を返します。
	 * <p>
	 * @return フィールドの有理数値
	*/
	public Rational getRational(){
		Rational[] f = getRationals();
		if(f.length == 0){
			return Rational.NaN;
		}
		return f[0];
	}

	// ======================================================================
	// 有理数の参照
	// ======================================================================
	/**
	 * 繰り返しを持つフィールドに対して全ての値を有理数値配列として参照します。
	 * {@link Type#URATIONAL}, {@link Type#RATIONAL} 型に最適化された値参照用
	 * のメソッドです。整数型に対しても値を有理数配列として参照することが出来
	 * ます。実数または非数値型に対しては長さ 0 の配列を返します。
	 * <p>
	 * @return フィールドの有理数
	*/
	public Rational[] getRationals(){
		if(number == null || number.length == 0 || number[0] instanceof Float || number[0] instanceof Double){
			return new Rational[0];
		}
		Rational[] num = new Rational[number.length];
		for(int i=0; i<number.length; i++){
			if(number[i] instanceof Rational){
				num[i] = (Rational)number[i];
			} else {
				assert(number[i] instanceof Byte || number[i] instanceof Short || number[i] instanceof Integer || number[i] instanceof Long): number[i].getClass().getSimpleName();
				num[i] = new Rational(number[i].longValue(), 1);
			}
		}
		return num;
	}

	// ======================================================================
	// バイナリデータの参照
	// ======================================================================
	/**
	 * このフィールドの値を生のバイナリとして参照します。フィールドが多バイト
	 * 整数型 ({@link Type#USHORT}, {@link Type#ULONG}, {@link Type#SHORT},
	 * {@link Type#LONG}) あるいは有理数型 ({@link Type#URATIONAL},
	 * {@link Type#RATIONAL}) を示す場合、返値のバイナリはビッグエンディアン
	 * で値を保持しています。
	 * <p>
	 * @return 値のバイナリ値
	*/
	public byte[] getBinary(){
		return (byte[])binary.clone();
	}

	// ======================================================================
	// インスタンスの文字列化
	// ======================================================================
	/**
	 * このインスタンスの内容を文字列化します。
	 * <p>
	 * @return 文字列
	*/
	@Override
	public String toString(){
		return getIFD().getLabel() + ":" + hex(getTag()) + "[" + getString() + "]";
	}

	// ======================================================================
	// スタティックイニシャライザ
	// ======================================================================
	/**
	 * フィールド名テーブルを初期化します。
	 * <p>
	*/
	static{

		define(IFD.I0TH, 0x0100, "ImageWidth", "画像の幅");
		define(IFD.I0TH, 0x0101, "ImageLength", "画像の高さ");
		define(IFD.I0TH, 0x0102, "BitsPerSample", "画像のビットの深さ");
		define(IFD.I0TH, 0x0103, "Compression", "圧縮の種類");
		define(IFD.I0TH, 0x0106, "PhotometricInterpretation", "画素構成");
		define(IFD.I0TH, 0x0112, "Orientation", "画像方向");
		define(IFD.I0TH, 0x0115, "SamplesPerPixel", "コンポーネント数");
		define(IFD.I0TH, 0x011C, "PlanarConfiguration", "画像データの並び");
		define(IFD.I0TH, 0x0212, "YCbCrSubSampling", "YCC の画素構成（C の間引き率）");
		define(IFD.I0TH, 0x0213, "YCbCrPositioning", "YCC の画素構成（Y とC の位置）");
		define(IFD.I0TH, 0x011A, "XResolution", "画像の幅の解像度");
		define(IFD.I0TH, 0x011B, "YResolution", "画像の高さの解像度");
		define(IFD.I0TH, 0x0128, "ResolutionUnit", "画像の幅と高さの解像度の単位");
		define(IFD.I0TH, 0x0111, "StripOffsets", "画像データのロケーション");
		define(IFD.I0TH, 0x0116, "RowsPerStrip", "ストリップ中のライン数");
		define(IFD.I0TH, 0x0117, "StripByteCounts", "ストリップのデータ量");
		define(IFD.I0TH, 0x0201, "JPEGInterchangeFormat", "JPEG のSOI へのオフセット");
		define(IFD.I0TH, 0x0202, "JPEGInterchangeFormatLength", "JPEG データのバイト数");
		define(IFD.I0TH, 0x012D, "TransferFunction", "再生階調カーブ特性");
		define(IFD.I0TH, 0x013E, "WhitePoint", "参照白色点の色度座標値");
		define(IFD.I0TH, 0x013F, "PrimaryChromaticities", "原色の色度座標値");
		define(IFD.I0TH, 0x0211, "YCbCrCoefficients", "色変換マトリクス係数");
		define(IFD.I0TH, 0x0214, "ReferenceBlackWhite", "参照黒色点値と参照白色点値");
		define(IFD.I0TH, 0x0132, "DateTime", "ファイル変更日時");
		define(IFD.I0TH, 0x010E, "ImageDescription", "画像タイトル");
		define(IFD.I0TH, 0x010F, "Make", "画像入力機器のメーカー名");
		define(IFD.I0TH, 0x0110, "Model", "画像入力機器のモデル名");
		define(IFD.I0TH, 0x0131, "Software", "使用ソフトウェア名");
		define(IFD.I0TH, 0x013B, "Artist", "作者名");
		define(IFD.I0TH, 0x8298, "Copyright", "撮影著作権者/編集著作権者");

		define(IFD.I1ST, 0x0100, "ImageWidth", "画像の幅");
		define(IFD.I1ST, 0x0101, "ImageLength", "画像の高さ");
		define(IFD.I1ST, 0x0102, "BitsPerSample", "画像のビットの深さ");
		define(IFD.I1ST, 0x0103, "Compression", "圧縮の種類");
		define(IFD.I1ST, 0x0106, "PhotometricInterpretation", "画素構成");
		define(IFD.I1ST, 0x0112, "Orientation", "画像方向");
		define(IFD.I1ST, 0x0115, "SamplesPerPixel", "コンポーネント数");
		define(IFD.I1ST, 0x011C, "PlanarConfiguration", "画像データの並び");
		define(IFD.I1ST, 0x0212, "YCbCrSubSampling", "YCC の画素構成（C の間引き率）");
		define(IFD.I1ST, 0x0213, "YCbCrPositioning", "YCC の画素構成（Y とC の位置）");
		define(IFD.I1ST, 0x011A, "XResolution", "画像の幅の解像度");
		define(IFD.I1ST, 0x011B, "YResolution", "画像の高さの解像度");
		define(IFD.I1ST, 0x0128, "ResolutionUnit", "画像の幅と高さの解像度の単位");
		define(IFD.I1ST, 0x0111, "StripOffsets", "画像データのロケーション");
		define(IFD.I1ST, 0x0116, "RowsPerStrip", "ストリップ中のライン数");
		define(IFD.I1ST, 0x0117, "StripByteCounts", "ストリップのデータ量");
		define(IFD.I1ST, 0x0201, "JPEGInterchangeFormat", "JPEG のSOI へのオフセット");
		define(IFD.I1ST, 0x0202, "JPEGInterchangeFormatLength", "JPEG データのバイト数");
		define(IFD.I1ST, 0x012D, "TransferFunction", "再生階調カーブ特性");
		define(IFD.I1ST, 0x013E, "WhitePoint", "参照白色点の色度座標値");
		define(IFD.I1ST, 0x013F, "PrimaryChromaticities", "原色の色度座標値");
		define(IFD.I1ST, 0x0211, "YCbCrCoefficients", "色変換マトリクス係数");
		define(IFD.I1ST, 0x0214, "ReferenceBlackWhite", "参照黒色点値と参照白色点値");
		define(IFD.I1ST, 0x0132, "DateTime", "ファイル変更日時");
		define(IFD.I1ST, 0x010E, "ImageDescription", "画像タイトル");
		define(IFD.I1ST, 0x010F, "Make", "画像入力機器のメーカー名");
		define(IFD.I1ST, 0x0110, "Model", "画像入力機器のモデル名");
		define(IFD.I1ST, 0x0131, "Software", "使用ソフトウェア名");
		define(IFD.I1ST, 0x013B, "Artist", "作者名");
		define(IFD.I1ST, 0x8298, "Copyright", "撮影著作権者/編集著作権者");

		define(IFD.EXIF, 0x9000, "ExifVersion", "Exif バージョン");
		define(IFD.EXIF, 0xA000, "FlashPixVersion", "対応フラッシュピックスバージョン");
		define(IFD.EXIF, 0xA001, "ColorSpace", "色空間情報");
		define(IFD.EXIF, 0x9101, "ComponentsConfiguration", "各コンポーネントの意味");
		define(IFD.EXIF, 0x9102, "CompressedBitsPerPixel", "画像圧縮モード");
		define(IFD.EXIF, 0xA002, "PixelXDimension", "実効画像幅");
		define(IFD.EXIF, 0xA003, "PixelYDimension", "実効画像高さ");
		define(IFD.EXIF, 0x927C, "MakerNote", "メーカーノート");
		define(IFD.EXIF, 0x9286, "UserComment", "ユーザーコメント");
		define(IFD.EXIF, 0xA004, "RelatedSoundFile", "関連音声ファイル");
		define(IFD.EXIF, 0x9003, "DateTimeOriginal", "原画像データの生成日時");
		define(IFD.EXIF, 0x9004, "DateTimeDigitized", "ディジタルデータの作成日時");
		define(IFD.EXIF, 0x9290, "SubSecTime", "DateTime のサブセック");
		define(IFD.EXIF, 0x9291, "SubSecTimeOriginal", "DateTimeOriginal のサブセック");
		define(IFD.EXIF, 0x9292, "SubSecTimeDigitized", "DateTimeDigitized のサブセック");
		define(IFD.EXIF, 0xA005, "InteroperabiliyIFDPointer", "互換性IFD へのポインタ");
		define(IFD.EXIF, 0x829A, "ExposureTime", "露出時間");
		define(IFD.EXIF, 0x829D, "FNumber", "F ナンバー");
		define(IFD.EXIF, 0x8822, "ExposureProgram", "露出プログラム");
		define(IFD.EXIF, 0x8824, "SpectralSensitivity", "スペクトル感度");
		define(IFD.EXIF, 0x8827, "ISOSpeedRatings", "ISO スピードレート");
		define(IFD.EXIF, 0x8828, "OECF", "光電変換関数");
		define(IFD.EXIF, 0x9201, "ShutterSpeedValue", "シャッタースピード");
		define(IFD.EXIF, 0x9202, "ApertureValue", "絞り値");
		define(IFD.EXIF, 0x9203, "BrightnessValue", "輝度値");
		define(IFD.EXIF, 0x9204, "ExposureBiasValue", "露光補正値");
		define(IFD.EXIF, 0x9205, "MaxApertureValue", "レンズ最小Ｆ値");
		define(IFD.EXIF, 0x9206, "SubjectDistance", "被写体距離");
		define(IFD.EXIF, 0x9207, "MeteringMode", "測光方式");
		define(IFD.EXIF, 0x9208, "LightSource", "光源");
		define(IFD.EXIF, 0x9209, "Flash", "フラッシュ");
		define(IFD.EXIF, 0x920A, "FocalLength", "レンズ焦点距離");
		define(IFD.EXIF, 0xA20B, "FlashEnergy", "フラッシュ強度");
		define(IFD.EXIF, 0xA20C, "SpatialFrequencyResponse", "空間周波数応答");
		define(IFD.EXIF, 0xA20E, "FocalPlaneXResolution", "焦点面の幅の解像度");
		define(IFD.EXIF, 0xA20F, "FocalPlaneYResolution", "焦点面の高さの解像度");
		define(IFD.EXIF, 0xA210, "FocalPlaneResolutionUnit", "焦点面解像度単位");
		define(IFD.EXIF, 0xA214, "SubjectLocation", "被写体位置");
		define(IFD.EXIF, 0xA215, "ExposureIndex", "露出インデックス");
		define(IFD.EXIF, 0xA217, "SensingMethod", "センサー方式");
		define(IFD.EXIF, 0xA300, "FileSource", "ファイルソース");
		define(IFD.EXIF, 0xA301, "SceneType", "シーンタイプ");
		define(IFD.EXIF, 0xA302, "CFAPattern", "CFA パターン");

		// Exif 2.2
		define(IFD.EXIF, 0xA401, "CustomRendered", "カスタム画像処理");
		define(IFD.EXIF, 0xA402, "ExposureMode", "撮影モード");
		define(IFD.EXIF, 0xA403, "WhiteBalance", "ホワイトバランスモード");
		define(IFD.EXIF, 0xA404, "DigitalZoomRatio", "デジタルズーム");
		define(IFD.EXIF, 0xA405, "FocalLengthIn35mmFilm", "レンズの焦点距離(35mm)");
		define(IFD.EXIF, 0xA406, "SceneCaptureType", "シーン撮影タイプ");
		define(IFD.EXIF, 0xA407, "GainControl", "ゲインコントロール");
		define(IFD.EXIF, 0xA408, "Contrast", "コントラスト");
		define(IFD.EXIF, 0xA409, "Saturation", "彩度");
		define(IFD.EXIF, 0xA40A, "Sharpness", "シャープネス");
		define(IFD.EXIF, 0xA40B, "DeviceSettingDescription", "デバイス設定詳細");
		define(IFD.EXIF, 0xA40C, "SubjectDistanceRange", "被写体の距離範囲");
		define(IFD.EXIF, 0xA420, "ImageUniqueID", "画像識別ID");
		define(IFD.EXIF, 0xA500, "Gamma", "ガンマ値");

		define(IFD.GPS, 0x0000, "GPSVersionID", "GPS タグのバージョン");
		define(IFD.GPS, 0x0001, "GPSLatitudeRef", "北緯(N) or 南緯(S)");
		define(IFD.GPS, 0x0002, "GPSLatitude", "緯度 (数値)");
		define(IFD.GPS, 0x0003, "GPSLongitudeRef", "東経(E) or 西経(W)");
		define(IFD.GPS, 0x0004, "GPSLongitude", "経度 (数値)");
		define(IFD.GPS, 0x0005, "GPSAltitudeRef", "高度の単位");
		define(IFD.GPS, 0x0006, "GPSAltitude", "高度 (数値)");
		define(IFD.GPS, 0x0007, "GPSTimeStamp", "GPS 時間 (原子時計の時間)");
		define(IFD.GPS, 0x0008, "GPSSatellites", "測位に使った衛星信号");
		define(IFD.GPS, 0x0009, "GPSStatus", "GPS 受信機の状態");
		define(IFD.GPS, 0x000A, "GPSMeasureMode", "GPS の測位方法");
		define(IFD.GPS, 0x000B, "GPSDOP", "測位の信頼性");
		define(IFD.GPS, 0x000C, "GPSSpeedRef", "速度の単位");
		define(IFD.GPS, 0x000D, "GPSSpeed", "速度 (数値)");
		define(IFD.GPS, 0x000E, "GPSTrackRef", "進行方向の単位");
		define(IFD.GPS, 0x000F, "GPSTrack", "進行方向 (数値)");
		define(IFD.GPS, 0x0010, "GPSImgDirectionRef", "撮影した画像の方向の単位");
		define(IFD.GPS, 0x0011, "GPSImgDirection", "撮影した画像の方向（数値）");
		define(IFD.GPS, 0x0012, "GPSMapDatum", "測位に用いた地図データ");
		define(IFD.GPS, 0x0013, "GPSDestLatitudeRef", "目的地の北緯(N) or 南緯(S)");
		define(IFD.GPS, 0x0014, "GPSDestLatitude", "目的地の緯度（数値）");
		define(IFD.GPS, 0x0015, "GPSDestLongitudeRef", "目的地の東経(E) or 西経(W)");
		define(IFD.GPS, 0x0016, "GPSDestLongitude", "目的地の経度 (数値)");
		define(IFD.GPS, 0x0017, "GPSDestBearingRef", "目的地の方角の単位");
		define(IFD.GPS, 0x0018, "GPSDestBearing", "目的の方角 (数値)");
		define(IFD.GPS, 0x0019, "GPSDestDistanceRef", "目的地までの距離の単位");
		define(IFD.GPS, 0x001A, "GPSDestDistance", "目的地までの距離 (数値)");
		define(IFD.GPS, 0x001B, "GPSProcessingMethod", "位置確定方法");
		define(IFD.GPS, 0x001C, "GPSAreaInformation", "エリア情報");
		define(IFD.GPS, 0x001D, "GPSDateStamp", "UTC 相対時刻");
		define(IFD.GPS, 0x001E, "GPSDifferential", "誤差");

		define(IFD.INTROP, 0x0001, "InteroperabilityIndex", "互換性識別子");
	}

	// ======================================================================
	// フィールド名/ラベルの設定
	// ======================================================================
	/**
	 * 指定された定義を追加します。
	 * <p>
	 * @param ifd IFD
	 * @param tag タグ
	 * @param name 識別名
	 * @param label 日本語表記名
	*/
	private static void define(IFD ifd, int tag, String name, String label){
		assert(name.matches("[a-zA-Z0-9]+")): name;
		int key = getKey(ifd, tag);
		FIELD_NAME.put(key, name);
		FIELD_LABEL.put(key, label);
		return;
	}

	// ======================================================================
	// キーの参照
	// ======================================================================
	/**
	 * 指定された IFD とタグ値から Map 用のキー値を作成します。
	 * <p>
	 * @param ifd IFD
	 * @param tag タグ
	 * @return キー値
	*/
	private static int getKey(IFD ifd, int tag){
		return ((ifd.getDigit() & 0xFFFF) << 16) | (tag & 0xFFFF);
	}

	// ======================================================================
	// 4 バイト整数 16 進数変換
	// ======================================================================
	/**
	 * 指定された 4 バイト整数を 4 桁の 16 進数変換します。
	 * <p>
	 * @param num 変換する数値
	 * @return 16 進数表記の数値
	*/
	private static String hex(int num){
		return String.format("%04X", num);
	}

}
