/* **************************************************************************
 * Copyright (C) 2008 BJoRFUAN. All Right Reserved
 * **************************************************************************
 * This module, contains source code, binary and documentation, is in the
 * BSD License, and comes with NO WARRANTY.
 * 
 *                                                 torao <torao@bjorfuan.com>
 *                                                       http://www.moyo.biz/
 * $Id: ExifFactory.java,v 1.3 2008/08/11 17:28:21 torao Exp $
*/
package org.koiroha.fixez;

import java.io.*;
import java.net.URLConnection;
import java.nio.*;
import java.nio.channels.FileChannel;
import java.nio.channels.FileChannel.MapMode;
import java.util.*;
import java.util.logging.Logger;



// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// ExifFactory: Exif ファクトリ
// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
/**
 * JPEG のバイトバッファから Exif 情報を読み込むためのファクトリクラスです。
 * <p>
 * @version fixez 1.0 - $Revision: 1.3 $ $Date: 2008/08/11 17:28:21 $
 * @author <a href="mailto:torao@mars.dti.ne.jp">torao</a>
 * @since fixez 1.0 - 2008/08/11
 */
public class ExifFactory {

	// ======================================================================
	// ログ出力先
	// ======================================================================
	/**
	 * このクラスのログ出力先です。
	 * <p>
	 */
	private static final Logger logger = Logger.getLogger(ExifFactory.class.getName());

	// ======================================================================
	// Start of Image
	// ======================================================================
	/**
	 * 圧縮データの先頭を表すマーカーです。
	 * <p>
	*/
	private static final int SOI  = 0xFFD8;

	// ======================================================================
	// APP1
	// ======================================================================
	/**
	 * Exif の付属情報を表すマーカーです。
	 * <p>
	*/
	private static final int APP1 = 0xFFE1;

	// ======================================================================
	// Start of Scan
	// ======================================================================
	/**
	 * コンポーネントに関する各種パラメータを表すマーカーです。
	 * <p>
	*/
	private static final int SOS  = 0xFFDA;

	// ======================================================================
	// End of Image
	// ======================================================================
	/**
	 * 圧縮データの終了を表すマーカーです。
	 * <p>
	*/
	private static final int EOI  = 0xFFD9;

	// ======================================================================
	// コンストラクタ
	// ======================================================================
	/**
	 * コンストラクタは何も行いません。
	 * <p>
	 */
	public ExifFactory() {
		return;
	}

	// ======================================================================
	// インスタンスの作成
	// ======================================================================
	/**
	 * 指定された JPEG ファイルからインスタンスを生成します。ファイル内に Exif
	 * 情報が格納されていない場合には null を返します。
	 * <p>
	 * @param file Exif 情報を読み込む JPEG ファイル
	 * @return Exif 情報
	 * @throws IOException ファイルの読み込みに失敗した場合
	 * @throws ExifFormatException フォーマットが不正な場合
	*/
	public Exif getInstance(File file) throws IOException, ExifFormatException{
		FileChannel channel = null;
		try{
			channel = new FileInputStream(file).getChannel();
			ByteBuffer buffer = channel.map(MapMode.READ_ONLY, 0, channel.size());
			return getInstance(buffer);
		} finally {
			try{
				if(channel != null)	channel.close();
			} catch(IOException ex){/* */}
		}
	}

	// ======================================================================
	// インスタンスの作成
	// ======================================================================
	/**
	 * 指定された JPEG ファイルからインスタンスを生成します。ファイル内に Exif
	 * 情報が格納されていない場合には null を返します。
	 * <p>
	 * Exif の仕様では解析時にファイル内でのシークが必要となるため、ストリーム
	 * から取り出すインターフェースは提供されません。
	 * <p>
	 * @param buffer Exif 情報を読み込む JPEG ファイル
	 * @return Exif 情報
	 * @throws ExifFormatException フォーマットが不正な場合
	*/
	public Exif getInstance(ByteBuffer buffer) throws ExifFormatException{
		assert(buffer != null);

		// バッファの内容から Content-Type の推測
		String contentType = null;
		int pos = buffer.position();
		try{
			InputStream in = new ByteBufferInputStream(buffer);
			contentType = URLConnection.guessContentTypeFromStream(in);
		} catch(IOException ex){/* */}
		buffer.position(pos);
		logger.fine("content-type: " + contentType);

		// 内容から Content-Type を推測できなかった場合
		if(contentType == null){
			logger.fine("content-type unresolved");
			return null;
		}

		// JPEG ファイルの場合
		if(contentType.matches("image/p?jpe?g")){
			logger.fine("parse as jpeg image: " + contentType);
			return readJPEG(buffer);
		}

		// 未対応の形式
		logger.fine("unsupported content-type: " + contentType);
		return null;
	}

	// ======================================================================
	// JPEG ファイルの読み込み
	// ======================================================================
	/**
	 * 指定された JPEG のバイトバッファから Exif 情報を読み込みます。ファイル
	 * 内に Exif 情報が格納されていない場合には null を返します。
	 * <p>
	 * @param buffer Exif 情報を読み込む JPEG ファイル
	 * @return 読み込んだ Exif 情報
	 * @throws ExifFormatException フォーマットが不正な場合
	*/
	private static Exif readJPEG(ByteBuffer buffer) throws ExifFormatException{

		// APP1 Exif 付属情報の位置まで読み飛ばし
		int marker;
		do{
			buffer.order(ByteOrder.BIG_ENDIAN);
			marker = buffer.getShort() & 0xFFFF;
			logger.finer("mark found: " + hex(marker));
			if(marker != SOI){
				int length = (buffer.getShort() & 0xFFFF) - 2;		// ※2=sizeof(short)
				if(marker == APP1){
					logger.finer("APP1 marker found at 0x" + hex(buffer.position() - 4));
					Exif exif = readExif(buffer, length);
					if(exif != null){
						return exif;
					}
					logger.finer("exif information isn't retrieve, continue");
				} else {
					logger.finer("not a APP1 marker, continue");
					buffer.position(buffer.position() + length);
				}
			}
		} while(marker != EOI && marker != SOS);

		logger.fine("APP1 marker not found");
		return null;
	}

	// ======================================================================
	// Exif 情報の読み込み
	// ======================================================================
	/**
	 * 指定されたバイトバッファから Exif 情報を読み出してインスタンスを構築し
	 * ます。バイトバッファは Exif 情報の先頭をポイントしている必要があります。
	 * <p>
	 * @param buffer Exif 情報を読み出すバイトバッファ
	 * @param nextOffset 次の読み出し位置までのオフセット
	 * @return Exif 情報
	 * @throws ExifFormatException フォーマットが不正な場合
	*/
	private static Exif readExif(ByteBuffer buffer, int nextOffset) throws ExifFormatException{

		// Exif ヘッダを取得
		byte[] cmp = {(byte)'E', (byte)'x', (byte)'i', (byte)'f', (byte)0x00, (byte)0x00};
		byte[] sig = new byte[cmp.length];
		for(int i=0; i<sig.length; i++){
			sig[i] = buffer.get();
		}

		// Exif ヘッダでなければ次の読み出し位置まで移動して終了
		if(! Arrays.equals(cmp, sig)){
			buffer.position(buffer.position() + nextOffset - sig.length);
			logger.finer(hex(buffer.position()) + ": not a exif block");
			return null;
		}
		logger.finer("exif signature found");

		// TIFF ヘッダ部の読み込み
		int header = buffer.position();
		logger.finer("header position: 0x" + Integer.toHexString(header));

		// エンディアンの決定
		boolean bigendian = ((buffer.getShort() & 0xFFFF) == (('M' << 8) | 'M'));
		if(bigendian){
			buffer.order(ByteOrder.BIG_ENDIAN);
		} else {
			buffer.order(ByteOrder.LITTLE_ENDIAN);
		}
		logger.finer("byte order: " + buffer.order());

		int marker = buffer.getShort() & 0xFFFF;	// タグマーカー 0x002A
		int offset = buffer.getInt();				// TIFF ヘッダ長 0x00000008
		logger.finer("tiff header: tag marker=" + hex(marker) + ", offset=" + offset);

		if(marker != 0x002A){
			logger.finer("invalid tag marker: 0x" + hex(marker));
			return null;
		}

		// Exif フィールドの読み込み
		List<ExifField> fields = new ArrayList<ExifField>();
		read(fields, buffer, IFD.I0TH, header, offset);
		return new Exif(fields);
	}

	// ======================================================================
	// データの読み込み
	// ======================================================================
	/**
	 * 指定された型のデータを読み込みます。新しい IFD の集合を読み込みます。
	 * <p>
	 * @param fields Exif フィールドの格納先
	 * @param buffer Exif 情報のバイトバッファ
	 * @param marker マーカー
	 * @param header ヘッダの開始位置
	 * @param offset ヘッダ開始位置からデータへのオフセット
	 * @throws ExifFormatException フォーマットが不正な場合
	*/
	private static void read(List<ExifField> fields, ByteBuffer buffer, IFD marker, int header, int offset)
		throws ExifFormatException
	{
		logger.finer("starting ifd: " + marker);
		while(offset != 0){

			// 読み出し開始位置へ移動
			buffer.position(header + offset);

			// エントリ数の読み込み
			int entry = buffer.getShort() & 0xFFFF;
			logger.finer("start " + entry + " entries");

			for(int i=0; i<entry; i++){
				int tag = buffer.getShort() & 0xFFFF;
				int type = buffer.getShort() & 0xFFFF;
				int count = buffer.getInt();
				int value = buffer.getInt();
				logger.finer("tag=" + hex(tag) + ",type=" + hex(type) + ",count=" + count + ",value=" + Integer.toHexString(value));

				// タグ値が既知の IFD である場合は value が示すオフセットのフィールドを読み込み
				if(tag == Tag.IFD_EXIF || tag == Tag.IFD_GPS || tag == Tag.IFD_INTROP){
					int pointer = buffer.position();
					read(fields, buffer, IFD.valueOf(tag), header, value);
					buffer.position(pointer);
					continue;
				}

				// IFD 識別子でなければフィールドとして読み込み

				// 型の特定
				Type t = Type.valueOf(type);
				if(t == null){
					throw new ExifFormatException(
						"unsupported type detected: 0x" + hex(type) + " at position 0x" + Integer.toHexString(buffer.position()));
				}

				// フィールドの構築
				ExifField field = readExifField(buffer, marker, tag, t, count, value, header);
				fields.add(field);
			}

			// 次のオフセットを取得
			offset = buffer.getInt();
			logger.finer("end entries, next offset: 0x" + Integer.toHexString(header + offset));

			// 0th IFD を解析中なら 1st IFD へ移動
			if(marker == IFD.I0TH){
				marker = IFD.I1ST;
				logger.finer("moving ifd: " + marker);
			}
		}
		return;
	}

	// ======================================================================
	// Exif フィールドの読み込み
	// ======================================================================
	/**
	 * 指定されたバイトバッファからタグ、タイプ、数、データを持つフィールドを
	 * 構築します。
	 * <p>
	 * @param buffer バイトバッファ
	 * @param ifd IFD
	 * @param tag タグ
	 * @param type タイプ
	 * @param count 繰り返し回数
	 * @param value 値
	 * @param header ヘッダ
	 * @return Exif フィールド
	*/
	private static ExifField readExifField(ByteBuffer buffer, IFD ifd, int tag, Type type, int count, int value, int header) {
		logger.finest(ifd.toString() + ":" + hex(tag) + ":" + type.toString() + "[" + count + "]");

		byte[] binary = null;
		String text = null;
		Number[] number = null;

		// 型による分岐
		switch(type){

		// unsigned byte / signed byte
		case BYTE:
		case UBYTE:
			binary = readBinary(buffer, 1, count, value, header);
			number = new Number[binary.length];
			for(int i=0; i<binary.length; i++){
				if(type == Type.UBYTE){
					number[i] = new Short((short)(binary[i] & 0xFF));
				} else {
					number[i] = new Byte(binary[i]);
				}
			}
			break;

		// unsigned short / signed short
		case USHORT:
		case SHORT:
			binary = readBinary(buffer, 2, count, value, header);
			number = toShortArray(binary, count, buffer.order(), (type == Type.USHORT));
			break;

		// unsigned long / signed long
		case ULONG:
		case LONG:
			binary = readBinary(buffer, 4, count, value, header);
			number = toIntArray(binary, count, buffer.order(), (type == Type.ULONG));
			break;

		// unsigned rational / signed rational
		case URATIONAL:
		case RATIONAL:
			binary = readBinary(buffer, 4, count * 2, value, header);
			number = toRationalArray(binary, count, buffer.order(), (type == Type.URATIONAL));
			break;

		// single float
		case SFLOAT:
			binary = readBinary(buffer, 4, count, value, header);
			number = toFloatArray(binary, count);
			break;

		// double float
		case DFLOAT:
			binary = readBinary(buffer, 8, count, value, header);
			number = toDoubleArray(binary, count);
			break;

		// ascii strings
		case ASCII:
			binary = readBinary(buffer, 1, count, value, header);
			number = null;
			int len = binary.length;
			while(len > 0 && binary[len-1] == 0x00){	// ※効率化のため後方から評価
				len --;
			}
			text = new String(binary, 0, len);
			break;

		// undefined
		case UNDEFINED:
			// ※UNDEFINED の場合には count バイトのデータとなる
			binary = readBinary(buffer, 1, count, value, header);
			number = null;
			StringBuilder buf = new StringBuilder();
			for(int i=0; i<binary.length; i++){
				char ch = (char)(binary[i] & 0xFF);
				if(Character.isDefined(ch) && ! Character.isISOControl(ch)){
					buf.append(ch);
					if(ch == '\\'){
						buf.append('\\');
					}
				} else {
					buf.append('\\');
					buf.append(Character.forDigit((binary[i] >> 4) & 0x0F, 16));
					buf.append(Character.forDigit((binary[i] >> 0) & 0x0F, 16));
				}
			}
			text = buf.toString();
			break;

		// バグ
		default:
			assert(false): "Bug! unexpected type:" + type;
			break;
		}

		// 数値型でテキストが構築されていない場合
		if(text == null){
			assert(number != null);
			StringBuilder buf = new StringBuilder();
			for(int i=0; i<number.length; i++){
				if(i!=0){
					buf.append(',');
				}
				buf.append(number[i]);
			}
			text = buf.toString();
		}

		// フィールドの構築
		logger.finer(ifd + "." + hex(tag) + ": [" + type + "] " + text);
		return new ExifField(ifd, tag, type, count, text.toString(), binary, number);
	}

	// ======================================================================
	// 整数配列の参照
	// ======================================================================
	/**
	 * 指定されたバイト配列を 2 バイト整数の列とみなして変換して返します。こ
	 * のメソッドは {@code binary} の内容をビッグエンディアンに変換する副作用
	 * を持ちます。
	 * <p>
	 * @param binary バイト配列
	 * @param count 繰り返し回数
	 * @param order バイト順序
	 * @param unsigned 無符号の場合 true
	 * @return 整数配列
	*/
	private static Number[] toShortArray(byte[] binary, int count, ByteOrder order, boolean unsigned){
		ByteBuffer buffer = ByteBuffer.wrap(binary);
		buffer.order(order);
		Number[] num = new Number[count];
		for(int i=0; i<num.length; i++){
			if(unsigned){
				num[i] = new Integer(buffer.getShort() & 0xFFFF);
			} else {
				num[i] = new Short(buffer.getShort());
			}
			if(order == ByteOrder.LITTLE_ENDIAN){
				binary[i * 2 + 0] = (byte)((num[i].intValue() >> 8) & 0xFF);
				binary[i * 2 + 1] = (byte)((num[i].intValue() >> 0) & 0xFF);
			}
		}
		return num;
	}

	// ======================================================================
	// 整数配列の分割
	// ======================================================================
	/**
	 * 指定されたバイト配列を 4 バイト整数の列とみなして変換して返します。こ
	 * のメソッドは {@code binary} の内容をビッグエンディアンに変換する副作用
	 * を持ちます。
	 * <p>
	 * @param binary バイト配列
	 * @param count 繰り返し回数
	 * @param order バイト順序
	 * @param unsigned 無符号の場合 true
	 * @return 整数配列
	*/
	private static Number[] toIntArray(byte[] binary, int count, ByteOrder order, boolean unsigned) {
		ByteBuffer buffer = ByteBuffer.wrap(binary);
		buffer.order(order);
		Number[] num = new Number[count];
		for(int i=0; i<num.length; i++){
			if(unsigned){
				num[i] = new Long(buffer.getInt() & 0xFFFFFFFFL);
			} else {
				num[i] = new Integer(buffer.getInt());
			}
			if(order == ByteOrder.LITTLE_ENDIAN){
				binary[i * 4 + 0] = (byte)((num[i].intValue() >>> 24) & 0xFF);
				binary[i * 4 + 1] = (byte)((num[i].intValue() >>> 16) & 0xFF);
				binary[i * 4 + 2] = (byte)((num[i].intValue() >>>  8) & 0xFF);
				binary[i * 4 + 3] = (byte)((num[i].intValue() >>>  0) & 0xFF);
			}
		}
		return num;
	}

	// ======================================================================
	// 有理数配列の分割
	// ======================================================================
	/**
	 * 指定されたバイト配列を 4 バイト整数×2 の列とみなして変換して返します。
	 * このメソッドは {@code binary} の内容をビッグエンディアンに変換する副作
	 * 用を持ちます。
	 * <p>
	 * @param binary バイト配列
	 * @param count 繰り返し回数
	 * @param order バイト順序
	 * @param unsigned 無符号の場合 true
	 * @return 有理数配列
	*/
	private static Rational[] toRationalArray(byte[] binary, int count, ByteOrder order, boolean unsigned) {
		Number[] num = toIntArray(binary, count * 2, order, unsigned);
		Rational[] r = new Rational[num.length / 2];
		for(int i=0; i<r.length; i++){
			r[i] = new Rational(num[i * 2].longValue(), num[i * 2 + 1].longValue());
		}
		return r;
	}

	// ======================================================================
	// 実数配列の分割
	// ======================================================================
	/**
	 * 指定されたバイト配列を 4 バイト実数の列とみなして変換して返します。
	 * <p>
	 * @param binary バイト配列
	 * @param count 繰り返し回数
	 * @return 実数配列
	*/
	private static Number[] toFloatArray(byte[] binary, int count) {
		ByteBuffer buffer = ByteBuffer.wrap(binary);
		Number[] num = new Number[count];
		for(int i=0; i<num.length; i++){
			num[i] = new Float(buffer.getFloat());
		}
		return num;
	}

	// ======================================================================
	// 実数配列の分割
	// ======================================================================
	/**
	 * 指定されたバイト配列を 4 バイト実数の列とみなして変換して返します。
	 * <p>
	 * @param binary バイト配列
	 * @param count 繰り返し回数
	 * @return 実数配列
	*/
	private static Number[] toDoubleArray(byte[] binary, int count) {
		ByteBuffer buffer = ByteBuffer.wrap(binary);
		Number[] num = new Number[count];
		for(int i=0; i<num.length; i++){
			num[i] = new Double(buffer.getDouble());
		}
		return num;
	}

	// ======================================================================
	// バイナリデータの参照
	// ======================================================================
	/**
	 * 指定された大きさのデータを参照します。データサイズが 4 バイトを超える
	 * 場合には指定されたヘッダからのオフセットから、超えない場合には value
	 * からバイナリを生成します。このメソッドの呼び出しによりファイルのオフ
	 * セットは変わりません。返値のバイナリはファイルのバイト順序です。
	 * <p>
	 * @param buffer バイトバッファ
	 * @param width データの幅
	 * @param count 繰り返し回数
	 * @param value 値
	 * @param header ヘッダ
	 * @return バイナリデータ
	*/
	private static byte[] readBinary(ByteBuffer buffer, int width, int count, int value, int header){
		int offset = buffer.position();
		int size = width * count;
		byte[] binary = new byte[size];
		if(size <= 4){
			if(buffer.order() == ByteOrder.BIG_ENDIAN){
				if(binary.length > 0)	binary[0] = (byte)((value >>> 24) & 0xFF);
				if(binary.length > 1)	binary[1] = (byte)((value >>> 16) & 0xFF);
				if(binary.length > 2)	binary[2] = (byte)((value >>>  8) & 0xFF);
				if(binary.length > 3)	binary[3] = (byte)((value >>>  0) & 0xFF);
			} else {
				if(binary.length > 0)	binary[0] = (byte)((value >>>  0) & 0xFF);
				if(binary.length > 1)	binary[1] = (byte)((value >>>  8) & 0xFF);
				if(binary.length > 2)	binary[2] = (byte)((value >>> 16) & 0xFF);
				if(binary.length > 3)	binary[3] = (byte)((value >>> 24) & 0xFF);
			}
		} else {
			buffer.position(header + value);
			buffer.get(binary);
		}
		if(buffer.position() != offset){
			buffer.position(offset);
		}
		return binary;
	}

	// ======================================================================
	// 4バイト整数 16 進数変換
	// ======================================================================
	/**
	 * 指定された4バイト整数を 16 進数変換します。
	 * <p>
	 * @param num 変換する数値
	 * @return 16 進数表記の数値
	*/
	private static String hex(int num){
		return String.format("%04X", num);
	}

}
