/* **************************************************************************
 * Copyright (C) 2004 BJORFUAN. All Right Reserved
 * **************************************************************************
 *                                               ToRA. <torao@@mars.dti.ne.jp>
 *                                                       http://www.moyo.biz/
 * $Id: Exif.java,v 1.3 2008/08/11 17:27:40 torao Exp $
*/
package org.koiroha.fixez;

import java.io.*;
import java.util.*;



// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// Exif: Exif 情報クラス
// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
/**
 * JPEG に保存されている Exif 情報を表すクラスです。
 * <p>
 * @version fixez 1.0 - $Revision: 1.3 $ $Date: 2008/08/11 17:27:40 $
 * @author <a href="mailto:torao@mars.dti.ne.jp">torao</a>
 * @since fixez 1.0 - 2005/06/02
 * @see <a href="http://park2.wakwak.com/~tsuruzoh/Computer/Digicams/exif.html">
 * 		Exif形式の画像ファイル解説</a>
*/
public class Exif implements Serializable{

	// ======================================================================
	// シリアルバージョン
	// ======================================================================
	/**
	 * このクラスのシリアルバージョンです。
	 * <p>
	*/
	private static final long serialVersionUID = 1L;

	// ======================================================================
	// Exif フィールド
	// ======================================================================
	/**
	 * このインスタンスが持つ Exif フィールドのコレクションです。
	 * <p>
	*/
	private final Collection<ExifField> field = new ArrayList<ExifField>();


	// ======================================================================
	// IFD+タグインデックス
	// ======================================================================
	/**
	 * Exif フィールドを IFD とタグで参照するためのインデックスです。
	 * <p>
	*/
	private final Map<Integer, ExifField> ifdTagIndex = new HashMap<Integer, ExifField>();

	// ======================================================================
	// コンストラクタ
	// ======================================================================
	/**
	 * コンストラクタはクラス内に隠蔽されています。
	 * <p>
	 * @param fields Exif フィールド
	*/
	Exif(Collection<ExifField> fields){
		for(ExifField field: fields){
			setField(field);
		}
		return;
	}

	// ======================================================================
	// フィールドの参照
	// ======================================================================
	/**
	 * この Exif 情報に格納されているすべてのフィールドを参照します。返値の
	 * コレクションは不変です。
	 * <p>
	 * @return {@link ExifField} のコレクション
	*/
	public Collection<ExifField> getFields(){
		return Collections.unmodifiableCollection(field);
	}

	// ======================================================================
	// フィールドの参照
	// ======================================================================
	/**
	 * 指定された IFD 識別子、タグを持つフィールドを参照します。該当するフィー
	 * ルドが存在しない場合には null を返します。
	 * <p>
	 * @param ifd IFD 識別子
	 * @param tag タグ
	 * @return フィールド
	*/
	public ExifField getField(IFD ifd, int tag){
		return ifdTagIndex.get(getKey(ifd, tag));
	}

	// ======================================================================
	// フィールドの設定
	// ======================================================================
	/**
	 * 指定された IFD 識別子、タグを持つフィールドを作成します。
	 * <p>
	 * @param ifd IFD 識別子
	 * @param tag タグ
	 * @param value フィールドの値
	*/
//	public void setField(IFD ifd, int tag, String value){
//		byte[] temp = value.getBytes();
//		byte[] binary = new byte[temp.length + 1];
//		System.arraycopy(temp, 0, binary, 0, temp.length);
//		binary[binary.length - 1] = 0x00;
//		setField(new ExifField(ifd, tag, Type.ASCII, binary.length, value, binary, null));
//		return;
//	}

	// ======================================================================
	// Exif 情報の設定
	// ======================================================================
	/**
	 * このインスタンスに Exif 情報を設定します。
	 * <p>
	 * @return このインスタンスの文字列表現
	*/
	@Override
	public String toString(){
		return field.toString();
	}

	// ======================================================================
	// フィールドの設定
	// ======================================================================
	/**
	 * 指定されたフィールドを設定します。
	 * <p>
	 * @param field 設定するフィールド
	*/
	private void setField(ExifField field){
		this.field.add(field);
		this.ifdTagIndex.put(getKey(field.getIFD(), field.getTag()), field);
		return;
	}

	// ======================================================================
	// キーの参照
	// ======================================================================
	/**
	 * 指定された IFD とタグ値から Map 用のキー値を作成します。
	 * 指定された定義を追加します。
	 * <p>
	 * @param ifd IFD
	 * @param tag タグ
	 * @return キー値
	*/
	private static int getKey(IFD ifd, int tag){
		return ((ifd.getDigit() & 0xFFFF) << 16) | (tag & 0xFFFF);
	}

}
