package org.maachang.shm;

import java.io.IOException;

import org.maachang.shm.core.CoreShm;

/**
 * 共有メモリサーバオブジェクト.
 * <BR>
 * サーバ側共有メモリオブジェクト.
 *
 * @version 2008/03/09
 * @author  masahito suzuki
 * @since  ShareMemory 1.03
 */
public class ShmServer {
    
    /**
     * 基本アクセスオブジェクト.
     */
    private BaseSharedIO base = null ;
    
    /**
     * サーバ側管理スレッド.
     */
    private ShmServerThread thread = null ;
    
    /**
     * サーバタイムアウト値.
     */
    private int timeout = -1 ;
    
    /**
     * コンストラクタ.
     */
    private ShmServer() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR>
     * @param name 共有名を設定してオブジェクトを生成します.
     * @param timeout 受信タイムアウト値を設定します.
     * @exception Exception 例外.
     */
    public ShmServer( String name,int timeout ) throws Exception {
        if( name == null || ( name = name.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        name = name.trim() ;
        this.base = new BaseSharedIO( true,name,name ) ;
        this.thread = new ShmServerThread( base ) ;
        this.timeout = timeout ;
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     */
    public void destroy() {
        if( thread != null ) {
            thread.destroy() ;
        }
        thread = null ;
        if( base != null ) {
            base.destroy() ;
        }
        base = null ;
    }
    
    /**
     * データ取得.
     * <BR>
     * @return ShmIoData 受信データが返されます.
     * @exception Exception 例外.
     */
    public ShmIoData read() throws Exception {
        return this.read( timeout ) ;
    }
    
    /**
     * データ取得.
     * <BR>
     * @param timeout 受信タイムアウト値を設定します.
     * @return ShmIoData 受信データが返されます.
     * @exception Exception 例外.
     */
    public ShmIoData read( long timeout ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        long time = System.currentTimeMillis() ;
        for( ;; ) {
            if( isUse() == false ) {
                throw new IOException( "オブジェクトは既に破棄されています" ) ;
            }
            ShmData d = thread.getRecvData() ;
            if( d == null ) {
                if( timeout > 0 && time + timeout <= System.currentTimeMillis() ) {
                    throw new InterruptedException( "timeout" ) ;
                }
                Thread.sleep( 15L ) ;
            }
            else {
                return new ShmIoData( d.getSequenceId(),d.getNowTime(),d.getData() ) ;
            }
        }
    }
    
    /**
     * データ書き込み.
     * <BR>
     * @param info 送信対象データを設定します.<BR>
     *             受信処理で受け取ったオブジェクトに、
     *             データを付加した内容を設定します.
     * @exception Exception 例外.
     */
    public void write( ShmIoData info ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( info == null || info.getData() == null || info.getData().length <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        ShmData data = new ShmData( info.getSequenceId(),info.getNowTime() ) ;
        data.setData( info.getData() ) ;
        thread.setSendData( data ) ;
    }
    
    /**
     * 共有名を取得.
     * <BR>
     * @return name 共有名が返されます.
     */
    public String getName() {
        if( isUse() == false ) {
            return null ;
        }
        return base.getCoreShm().getShareName() ;
    }
    
    /**
     * オブジェクトが有効かチェック.
     * <BR>
     * @return boolean [true]の場合、有効です.
     */
    public boolean isUse() {
        return ( base != null && base.getCoreShm() != null &&
            base.getCoreShm().isUse() == true &&
            thread != null && thread.isStop() == false ) ;
    }
    
    /**
     * コアオブジェクトを取得.
     * <BR>
     * @return CoreShm コアオブジェクトが返されます.
     */
    public CoreShm getCoreShm() {
        if( isUse() == false ) {
            return null ;
        }
        return base.getCoreShm() ;
    }
}
