package org.maachang.shm;

import java.io.IOException;

import org.maachang.shm.core.CoreShm;

/**
 * 共有メモリクライアントオブジェクト.
 * <BR>
 * クライアント側共有メモリオブジェクト.
 *
 * @version 2008/03/09
 * @author  masahito suzuki
 * @since  ShareMemory 1.03
 */
public class ShmClient {
    
    /**
     * 基本アクセスオブジェクト.
     */
    private BaseSharedIO base = null ;
    
    /**
     * クライアント側管理スレッド.
     */
    private ShmClientThread thread = null ;
    
    /**
     * タイムアウト値.
     */
    private int timeout = -1 ;
    
    /**
     * コンストラクタ.
     */
    private ShmClient() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR>
     * @param name 共有名を設定してオブジェクトを生成します.
     * @param timeout 受信タイムアウト値を設定します.
     * @exception Exception 例外.
     */
    public ShmClient( String name,int timeout ) throws Exception {
        if( name == null || ( name = name.trim() ).length() <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        name = name.trim() ;
        this.base = new BaseSharedIO( false,name,name ) ;
        this.thread = new ShmClientThread( base ) ;
        this.timeout = timeout ;
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     */
    public void destroy() {
        if( thread != null ) {
            thread.destroy() ;
        }
        thread = null ;
        if( base != null ) {
            base.destroy() ;
        }
        base = null ;
    }
    
    /**
     * データ送信して取得.
     * <BR>
     * @param sendData 送信データを設定します.
     * @return byte[] 受信データが返されます.
     * @exception Exception 例外.
     */
    public byte[] io( byte[] sendData ) throws Exception {
        return io( sendData,timeout ) ;
    }
    
    /**
     * データ送信して取得.
     * <BR>
     * @param sendData 送信データを設定します.
     * @param timeout 受信タイムアウト値を設定します.
     * @return byte[] 受信データが返されます.
     * @exception Exception 例外.
     */
    public byte[] io( byte[] sendData,int timeout ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        if( sendData == null || sendData.length <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        long time = System.currentTimeMillis() ;
        int seqId = thread.getSequenceId() ;
        ShmData d = new ShmData( seqId ) ;
        long nowTime = d.getNowTime() ;
        d.setData( sendData ) ;
        thread.setSendData( d ) ;
        for( ;; ) {
            if( isUse() == false ) {
                throw new IOException( "オブジェクトは既に破棄されています" ) ;
            }
            d = thread.getReceiveData( seqId,nowTime ) ;
            if( d == null ) {
                if( timeout > 0 && time + timeout <= System.currentTimeMillis() ) {
                    throw new InterruptedException( "timeout" ) ;
                }
                Thread.sleep( 15L ) ;
            }
            else {
                return d.getData() ;
            }
        }
    }
    
    /**
     * 共有名を取得.
     * <BR>
     * @return name 共有名が返されます.
     */
    public String getName() {
        if( isUse() == false ) {
            return null ;
        }
        return base.getCoreShm().getShareName() ;
    }
    
    /**
     * オブジェクトが有効かチェック.
     * <BR>
     * @return boolean [true]の場合、有効です.
     */
    public boolean isUse() {
        return ( base != null && base.getCoreShm() != null &&
            base.getCoreShm().isUse() == true &&
            thread != null && thread.isStop() == false ) ;
    }
    
    /**
     * コアオブジェクトを取得.
     * <BR>
     * @return CoreShm コアオブジェクトが返されます.
     */
    public CoreShm getCoreShm() {
        if( isUse() == false ) {
            return null ;
        }
        return base.getCoreShm() ;
    }
    
    
}
