package org.maachang.shm;

import org.maachang.shm.core.CoreShm;

class BaseSharedIO {
    private boolean mode = false ;
    private CoreShm shm = null ;
    private byte[] wData = null ;
    private byte[] rData = null ;
    
    /**
     * コンストラクタ.
     * <BR>
     * @param mode [true]の場合、サーバモードで生成します.
     * @param semName セマフォー名を設定します.
     * @param shmName 共有メモリ名を設定します.
     * @exception Exception 例外.
     */
    public BaseSharedIO( boolean mode,String semName,String shmName )
        throws Exception {
        this.wData = new byte[ ShmCommon.IO_LENGTH ] ;
        this.rData = new byte[ ShmCommon.IO_LENGTH ] ;
        this.mode = mode ;
        this.shm = new CoreShm( mode,semName,shmName,ShmCommon.SHARED_LENGTH ) ;
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        destroy() ;
    }
    
    /**
     * オブジェクトを破棄.
     */
    public void destroy() {
        if( shm != null ) {
            try {
                shm.destroy() ;
            } catch( Exception e ) {
            }
        }
        wData = null ;
        rData = null ;
        shm = null ;
    }
    
    /**
     * コアオブジェクトを取得.
     * <BR>
     * @return CoreShm コアオブジェクトが返されます.
     */
    public CoreShm getCoreShm() {
        return shm ;
    }
    
    /**
     * 生成モードを取得.
     * <BR>
     * @return [true]の場合、サーバモードで生成されています.
     */
    public boolean isMode() {
        return mode ;
    }
    
    /**
     * ロック処理.
     * <BR>
     * @exception Exception 例外.
     */
    public void lock() throws Exception {
        shm.lock() ;
    }
    
    /**
     * アンロック処理.
     * <BR>
     * @exception Exception 例外.
     */
    public void unLock() throws Exception {
        shm.unLock() ;
    }
    
    /**
     * シーケンスIDを取得
     * <BR>
     * @return int シーケンスIDが返されます.
     * @exception Exception 例外.
     */
    public int getSequenceId() throws Exception {
        return ShmCommon.getSequenceId( shm ) ;
    }
    
    /**
     * 書き込みヘッダ内容を読み込む.
     * <BR>
     * @param sync [true]の場合同期します.
     * @return ShmHeader ヘッダ内容が返されます.
     * @exception Exception 例外.
     */
    public ShmHeader getWriteHeader( boolean sync )
        throws Exception {
        return ShmCommon.getHeader( ( mode == false ),sync,shm ) ;
    }
    
    /**
     * 読み込みヘッダ内容を読み込む.
     * <BR>
     * @param sync [true]の場合同期します.
     * @return ShmHeader ヘッダ内容が返されます.
     * @exception Exception 例外.
     */
    public ShmHeader getReadHeader( boolean sync )
        throws Exception {
        return ShmCommon.getHeader( ( mode == true ),sync,shm ) ;
    }
    
    /**
     * 書き込みデータを読み込む.
     * <BR>
     * @param sync [true]の場合同期します.
     * @param length データ長を設定します.
     * @return byte[] データ領域が返されます.
     * @exception Exception 例外.
     */
    public byte[] getWriteData( boolean sync,int length )
        throws Exception {
        ShmCommon.getData( ( mode == false ),sync,wData,shm,length ) ;
        return wData ;
    }
    
    /**
     * 読み込みデータを読み込む.
     * <BR>
     * @param sync [true]の場合同期します.
     * @param length データ長を設定します.
     * @return byte[] データ領域が返されます.
     * @exception Exception 例外.
     */
    public byte[] getReadData( boolean sync,int length )
        throws Exception {
        ShmCommon.getData( ( mode == true ),sync,rData,shm,length ) ;
        return rData ;
    }
    
    /**
     * 書き込み情報を設定.
     * <BR>
     * @param header 対象のヘッダ情報を設定します.
     * @param data 対象のデータ情報を設定します.
     * @param offset オフセット値を設定します.
     * @return int 次開始領域を設定します.<BR>
     *             [-1]の場合、書き込み可能領域は存在しません.
     * @exception Exception 例外.
     */
    public int setWrite( ShmHeader header,byte[] data,int offset )
        throws Exception {
        if( data == null ) {
            header.setDataLength( 0 ) ;
            ShmCommon.setHeader( ( mode == false ),true,shm,header.getSequenceId(),header.getNowTime(),header.getDataLength() ) ;
            return -1 ;
        }
        else {
            if( offset <= 0 ) {
                offset = 0 ;
            }
            int ret = -1 ;
            int dataLen = data.length - offset ;
            if( dataLen > ShmCommon.IO_LENGTH ) {
                dataLen = ShmCommon.IO_LENGTH ;
                header.setDataLength( ShmCommon.NEXT_LENGTH ) ;
                ret = offset + ShmCommon.IO_LENGTH ;
            }
            else {
                header.setDataLength( dataLen ) ;
                ret = -1 ;
            }
            int len = header.getDataLength() ;
            if( len >= ShmCommon.IO_LENGTH ) {
                len = ShmCommon.IO_LENGTH ;
            }
            ShmCommon.outputHeaderData( ( mode == false ),true,shm,data,len,header.getSequenceId(),header.getNowTime(),header.getDataLength() ) ;
            return ret ;
        }
    }
    
    /**
     * 読み込み情報を設定.
     * <BR>
     * @param header 対象のヘッダ情報を設定します.
     * @param data 対象のデータ情報を設定します.
     * @param offset オフセット値を設定します.
     * @return int 次開始領域を設定します.<BR>
     *             [-1]の場合、書き込み可能領域は存在しません.
     * @exception Exception 例外.
     */
    public int setRead( ShmHeader header,byte[] data,int offset )
        throws Exception {
        if( data == null ) {
            header.setDataLength( 0 ) ;
            ShmCommon.setHeader( ( mode == true ),true,shm,header.getSequenceId(),header.getNowTime(),header.getDataLength() ) ;
            return -1 ;
        }
        else {
            if( offset <= 0 ) {
                offset = 0 ;
            }
            int ret = -1 ;
            int dataLen = data.length - offset ;
            if( dataLen > ShmCommon.IO_LENGTH ) {
                dataLen = ShmCommon.IO_LENGTH ;
                header.setDataLength( ShmCommon.NEXT_LENGTH ) ;
                ret = offset + ShmCommon.IO_LENGTH ;
            }
            else {
                header.setDataLength( dataLen ) ;
                ret = -1 ;
            }
            int len = header.getDataLength() ;
            if( len >= ShmCommon.IO_LENGTH ) {
                len = ShmCommon.IO_LENGTH ;
            }
            ShmCommon.outputHeaderData( ( mode == true ),true,shm,data,len,header.getSequenceId(),header.getNowTime(),header.getDataLength() ) ;
            return ret ;
        }
    }

}
