//
// Linux用のSharedMemoyには、IPCを利用する。
//
#include <unistd.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/shm.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "../../include/org_maachang_shm_core_NativeSharedMemory.h"

///////////////////////////////////////////////////////////////////////////////
// ShareMemory内部処理.
///////////////////////////////////////////////////////////////////////////////

#define MODE_BY_SERVER			(0)
#define MODE_NOT_SYNC			(0)

#define SEQ_LENGTH				(4)
#define HEADER_LENGTH			(16)
#define NOT_DATA_LENGTH			(36)
#define WRITE_POSITION			(0)

// 各データヘッダ.
typedef struct {
	int seq ;
	int timeD ;
	int timeU ;
	int length ;
} SHM_HD,*LPSHM_HD ;

union semun {
  int val;					/* SETVAL の値 */
  struct semid_ds *buf; 	/* IPC_STAT, IPC_SET 用のバッファ */
  unsigned short *array;	/* GETALL, SETALL 用の配列 */
  /* Linux 特有の部分: */
  struct seminfo *__buf;	/* IPC_INFO 用のバッファ */
};

static int semId ;
static int shmId ;
static int shmMode ;
static int shareSize ;
static char* shareData ;
static int ioLength ;
static union semun semon ;
static int semMode ;

// プロセスIDを取得.
pid_t getProcessId() {
	return getpid() ;
}

// 文字列から、数値を求める[fnv-hash32a].
key_t ConvertCharArrayByInt( char* b ) {
	key_t c = 0x811C9DC5 ;
	int len = strlen( b ) ;
	for( int i = 0 ; i < len ; i ++ ) {
		c ^= b[i];
		c += (c << 1L) + (c << 4L) + (c << 7L) + (c << 8L) + (c << 24L);
	}
	return c ;
}

// セマフォ初期化.
int InitSemaphoreHandle(int mode,char* name) {
	key_t key ;
	char semName[ 2048 ] ;
	sprintf( semName,"sem_%s\0",name ) ;
	key = ConvertCharArrayByInt( semName ) ;
	int code ;
	if( mode == MODE_BY_SERVER ) {
		code = IPC_CREAT | 0666 ;
	}
	else {
		code = 0666 ;
	}
	if( ( semId = semget(key, 1, code) ) == -1 ) {
		return -1 ;
	}
	if( mode == MODE_BY_SERVER ) {
		if(semctl(semId,0,GETVAL,semon) == 0){
			semon.val=1;
			if(semctl(semId,0,SETVAL,semon) == -1){
				return -1 ;
			}
		}
	}
	semMode = 0 ;
	return 0 ;
}

// セマフォ終了化.
void CloseSemaphoreHandle() {
	if( shmMode == MODE_BY_SERVER && semId > 0 ) {
		semctl(semId, 0, IPC_RMID, semon) ;
		semId = -1 ;
	}
}

// セマフォロック.
int LockSharedMemory() {
	if( semMode == 0 ) {
		struct sembuf sb;
		sb.sem_num=0;
		sb.sem_op=-1;
		sb.sem_flg=SEM_UNDO;
		if(semop(semId,&sb,1)== -1){
			return -1 ;
		}
		semMode = 1 ;
	}
	return 0 ;
}

// セマフォアンロック.
int UnLockSharedMemory() {
	if( semMode == 1 ) {
		struct sembuf sb;
		sb.sem_num=0;
		sb.sem_op=1;
		sb.sem_flg=SEM_UNDO;
		if(semop(semId,&sb,1)== -1){
			return -1 ;
		}
		semMode = 0 ;
	}
	return 0 ;
}

// 共有メモリオープン.
int OpenShare(int mode,char* shareName,int size) {
	key_t key ;
	key = ConvertCharArrayByInt( shareName ) ;
	int code ;
	if( mode == MODE_BY_SERVER ) {
		code = IPC_CREAT | 0666 ;
	}
	else {
		code = 0666 ;
	}
	if ((shmId = shmget(key, size, code)) < 0) {
		return -1 ;
	}
	shareData = (char *)shmat(shmId, (void *)0, 0);
	if (shareData == (char *)-1) {
		shareData = NULL ;
		return -1 ;
	}
	if( mode == MODE_BY_SERVER ) {
		memset( shareData,'\0',size ) ;
	}
	ioLength = ( size - NOT_DATA_LENGTH ) / 2 ;
	return 0 ;
}

// 共有メモリクローズ.
void CloseShare() {
	if( shareData != NULL ) {
		shmdt(shareData) ;
		shareData = NULL ;
	}
	if( shmMode == MODE_BY_SERVER && shmId > 0 ) {
		shmctl(shmId, IPC_RMID, 0) ;
		shmId = -1 ;
	}
}

// 共有メモリ処理全般初期化処理.
int InitShareMemory( int mode,char* semName,char* shareName,int size ) {
	if( InitSemaphoreHandle( mode,semName ) <= -1 ) {
		return -1 ;
	}
	 ;
	if( LockSharedMemory() >= 0 ) {
		if( OpenShare( mode,shareName,size ) <= -1 ) {
			UnLockSharedMemory() ;
			return -1 ;
		}
		UnLockSharedMemory() ;
	}
	shmMode = mode ;
	shareSize = size ;
	return 0 ;
}

// 共有メモリ処理全般終了化処理.
int DestroyShareMemory() {
	if( LockSharedMemory() >= 0 ) {
		CloseShare() ;
		UnLockSharedMemory() ;
	}
	else {
		return -1 ;
	}
	CloseSemaphoreHandle() ;
	return 0 ;
}

// 共有メモリに情報追加.
int PutShareData( int md,unsigned char* data,int pos,int off,int len ) {
	if( md == MODE_NOT_SYNC ) {
		memcpy( shareData+pos,data+off,len ) ;
	}
	else {
		if( LockSharedMemory() >= 0 ) {
			memcpy( shareData+pos,data+off,len ) ;
			UnLockSharedMemory() ;
		}
		else {
			return -1 ;
		}
	}
	return 0 ;
}

// 共有メモリから情報取得.
int GetShareData( int md,unsigned char* data,int pos,int off,int len ) {
	if( md == MODE_NOT_SYNC ) {
		memcpy( data+off,shareData+pos,len ) ;
	}
	else {
		if( LockSharedMemory() >= 0 ) {
			memcpy( data+off,shareData+pos,len ) ;
			UnLockSharedMemory() ;
		}
		else {
			return -1 ;
		}
	}
	return 0 ;
}

// 共有メモリ全体長を取得.
int MaxShreSize() {
	return shareSize ;
}

///////////////////////////////////////////////////////////////////////////////
// 共有メモリ拡張処理.
///////////////////////////////////////////////////////////////////////////////

// シーケンスIDを取得.
inline int GetSequenceId( int md ) {
	int seq = -1 ;
	if( md != MODE_NOT_SYNC ) {
		LockSharedMemory() ;
	}
	int len = sizeof( seq ) ;
	memcpy( &seq,shareData,len ) ;
	if( seq <= 0 || seq >= 0x7fffffff ) {
		seq = 1 ;
	}
	else {
		seq ++ ;
	}
	memcpy( shareData,&seq,len ) ;
	if( md != MODE_NOT_SYNC ) {
		UnLockSharedMemory() ;
	}
	return seq ;
}

// ヘッダ領域を取得.
inline void GetSmHeader( int md,int t,LPSHM_HD hd ) {
	int p ;
	if( md != MODE_NOT_SYNC ) {
		LockSharedMemory() ;
	}
	if( t == WRITE_POSITION ) {
		p = SEQ_LENGTH ;
	}
	else {
		p = SEQ_LENGTH + HEADER_LENGTH + ioLength ;
	}
	memcpy( hd,shareData+p,sizeof( SHM_HD ) ) ;
	if( md != MODE_NOT_SYNC ) {
		UnLockSharedMemory() ;
	}
}

// データ情報を取得.
inline void GetShmValue( int md,int t,unsigned char* data,int len ) {
	int p ;
	if( md != MODE_NOT_SYNC ) {
		LockSharedMemory() ;
	}
	if( t == WRITE_POSITION ) {
		p = SEQ_LENGTH + HEADER_LENGTH ;
	}
	else {
		p = SEQ_LENGTH + HEADER_LENGTH + ioLength + HEADER_LENGTH ;
	}
	memcpy( data,shareData+p,len ) ;
	if( md != MODE_NOT_SYNC ) {
		UnLockSharedMemory() ;
	}
}

// ヘッダ情報を書き込む.
inline void PutSmHeader( int md,int t,LPSHM_HD hd ) {
	int p ;
	if( md != MODE_NOT_SYNC ) {
		LockSharedMemory() ;
	}
	if( t == WRITE_POSITION ) {
		p = SEQ_LENGTH ;
	}
	else {
		p = SEQ_LENGTH + HEADER_LENGTH + ioLength ;
	}
	memcpy( shareData+p,hd,sizeof( SHM_HD ) ) ;
	if( md != MODE_NOT_SYNC ) {
		UnLockSharedMemory() ;
	}
}

// データ情報を書き込む.
inline void PutShmValue( int md,int t,unsigned char* data,int len ) {
	int p ;
	if( md != MODE_NOT_SYNC ) {
		LockSharedMemory() ;
	}
	if( t == WRITE_POSITION ) {
		p = SEQ_LENGTH + HEADER_LENGTH ;
	}
	else {
		p = SEQ_LENGTH + HEADER_LENGTH + ioLength + HEADER_LENGTH ;
	}
	memcpy( shareData+p,data,len ) ;
	if( md != MODE_NOT_SYNC ) {
		UnLockSharedMemory() ;
	}
}

// ヘッダ＋データ情報を書き込む.
inline void PutHdShmValue( int md,int t,LPSHM_HD hd,unsigned char* data,int len ) {
	int p ;
	if( md != MODE_NOT_SYNC ) {
		LockSharedMemory() ;
	}
	if( t == WRITE_POSITION ) {
		p = SEQ_LENGTH ;
	}
	else {
		p = SEQ_LENGTH + HEADER_LENGTH + ioLength ;
	}
	memcpy( shareData+p,hd,sizeof( SHM_HD ) ) ;
	memcpy( shareData+(p+HEADER_LENGTH),data,len ) ;
	if( md != MODE_NOT_SYNC ) {
		UnLockSharedMemory() ;
	}
}

///////////////////////////////////////////////////////////////////////////////
// JNI処理.
///////////////////////////////////////////////////////////////////////////////

// NativeShareMemory.initShareMemory呼び出し.
JNIEXPORT jint JNICALL Java_org_maachang_shm_core_NativeSharedMemory_initShareMemory
  (JNIEnv* env, jobject o, jint mode,jbyteArray n1, jbyteArray n2, jint sz ) {
	jboolean b ;
	jbyte* semName=(*env).GetByteArrayElements(n1,&b);
	jbyte* shareName=(*env).GetByteArrayElements(n2,&b);
	jint ret = InitShareMemory( mode,( char* )semName,( char* )shareName,sz ) ;
	(*env).ReleaseByteArrayElements(n1, semName, 0);
	(*env).ReleaseByteArrayElements(n2, shareName, 0);
	return ret ;
}

// NativeShareMemory.destroyShareMemory呼び出し.
JNIEXPORT jint JNICALL Java_org_maachang_shm_core_NativeSharedMemory_destroyShareMemory
  (JNIEnv* env, jobject o) {
	shareSize = -1 ;
	return DestroyShareMemory() ;
}

// NativeShareMemory.writeShareMemory呼び出し.
JNIEXPORT jint JNICALL Java_org_maachang_shm_core_NativeSharedMemory_writeShareMemory
  (JNIEnv* env, jobject o, jint md, jbyteArray in, jint ps, jint of, jint sz) {
	jboolean b ;
	jbyte* input=(*env).GetByteArrayElements(in,&b);
	jint ret = PutShareData( md,( unsigned char* )input,ps,of,sz ) ;
	(*env).ReleaseByteArrayElements(in, input, 0);
	return ret ;
}

// NativeShareMemory.readShareMemory呼び出し.
JNIEXPORT jint JNICALL Java_org_maachang_shm_core_NativeSharedMemory_readShareMemory
  (JNIEnv* env, jobject o, jint md, jbyteArray ot, jint ps, jint of, jint sz) {
	jboolean b ;
	jbyte* output=(*env).GetByteArrayElements(ot,&b);
	jint ret = GetShareData( md,( unsigned char* )output,ps,of,sz ) ;
	(*env).ReleaseByteArrayElements(ot, output, 0);
	return ret ;
}

// NativeShareMemory.sizeShareMemory呼び出し.
JNIEXPORT jint JNICALL Java_org_maachang_shm_core_NativeSharedMemory_sizeShareMemory
  (JNIEnv* env, jobject o) {
	return MaxShreSize() ;
}

// NativeShareMemory.startLock呼び出し.
JNIEXPORT jint JNICALL Java_org_maachang_shm_core_NativeSharedMemory_startLock
  (JNIEnv* env, jobject o) {
	return LockSharedMemory() ;
}

// NativeShareMemory.endLock呼び出し.
JNIEXPORT jint JNICALL Java_org_maachang_shm_core_NativeSharedMemory_endLock
  (JNIEnv* env, jobject o) {
	return UnLockSharedMemory() ;
}

// NativeShareMemory.processId呼び出し.
JNIEXPORT jint JNICALL Java_org_maachang_shm_core_NativeSharedMemory_processId
  (JNIEnv* env, jobject o) {
	return ( int )getProcessId() ;
}

// NativeShareMemory.getSequenceId呼び出し.
JNIEXPORT jint JNICALL Java_org_maachang_shm_core_NativeSharedMemory_getSequenceId
  (JNIEnv* env, jobject o, jint md) {
	return ( int )::GetSequenceId( md ) ;
}

// NativeShareMemory.getHeader呼び出し.
JNIEXPORT void JNICALL Java_org_maachang_shm_core_NativeSharedMemory_getHeader
  (JNIEnv* env, jobject o, jint md, jint tp, jintArray seq, jintArray tu, jintArray td, jintArray dLen) {
	SHM_HD hd ;
	GetSmHeader( md,tp,&hd ) ;
	jboolean b ;
	jint* iseq = (*env).GetIntArrayElements(seq,&b) ;
	iseq[ 0 ] = hd.seq ;
	(*env).ReleaseIntArrayElements(seq,iseq,0);

	jint* itd = (*env).GetIntArrayElements(td,&b) ;
	itd[ 0 ] = hd.timeD ;
	(*env).ReleaseIntArrayElements(td,itd,0);

	jint* itu = (*env).GetIntArrayElements(tu,&b) ;
	itu[ 0 ] = hd.timeU ;
	(*env).ReleaseIntArrayElements(tu,itu,0);

	jint* idLen = (*env).GetIntArrayElements(dLen,&b) ;
	idLen[ 0 ] = hd.length ;
	(*env).ReleaseIntArrayElements(dLen,idLen,0);
}

// NativeShareMemory.putHeader呼び出し.
JNIEXPORT void JNICALL Java_org_maachang_shm_core_NativeSharedMemory_putHeader
  (JNIEnv* env, jobject o, jint md, jint tp, jint seq, jint tu, jint td, jint dLen) {
	SHM_HD hd ;
	hd.seq = seq ;
	hd.timeD = td ;
	hd.timeU = tu ;
	hd.length = dLen ;
	PutSmHeader( md,tp,&hd ) ;
}

// NativeShareMemory.getValue呼び出し.
JNIEXPORT void JNICALL Java_org_maachang_shm_core_NativeSharedMemory_getValue
  (JNIEnv* env, jobject o, jint md, jint tp, jbyteArray data, jint len) {
	jboolean b ;
	jbyte* dt=(*env).GetByteArrayElements(data,&b);
	GetShmValue( md,tp,( unsigned char* )dt,len ) ;
	(*env).ReleaseByteArrayElements(data, dt, 0);
}

// NativeShareMemory.putValue呼び出し.
JNIEXPORT void JNICALL Java_org_maachang_shm_core_NativeSharedMemory_putValue
  (JNIEnv* env, jobject o, jint md, jint tp, jbyteArray data, jint len) {
	jboolean b ;
	jbyte* dt=(*env).GetByteArrayElements(data,&b);
	PutShmValue( md,tp,( unsigned char* )dt,len ) ;
	(*env).ReleaseByteArrayElements(data, dt, 0);
}

// NativeShareMemory.putHdValue呼び出し.
JNIEXPORT void JNICALL Java_org_maachang_shm_core_NativeSharedMemory_putHdValue
  (JNIEnv* env, jobject o, jint md, jint tp, jint seq, jint tu, jint td, jint dLen, jbyteArray data, jint len) {
	SHM_HD hd ;
	hd.seq = seq ;
	hd.timeD = td ;
	hd.timeU = tu ;
	hd.length = dLen ;
	jboolean b ;
	jbyte* dt=(*env).GetByteArrayElements(data,&b);
	PutHdShmValue( md,tp,&hd,( unsigned char* )dt,len ) ;
	(*env).ReleaseByteArrayElements(data, dt, 0);
}

