package org.maachang.shm.core;

import java.io.BufferedOutputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;

/**
 * コア部分となる共有メモリオブジェクト.
 * <BR>
 * このオブジェクトは、共有メモリを作成し、その共有メモリI/Oを
 * 処理するための、コアオブジェクトです.
 *  
 * @version 2008/02/11
 * @author  masahito suzuki
 * @since  SharedMemory 1.00
 */
public class CoreShm {
    /**
     * 共有メモリーオブジェクト.
     */
    private NativeSharedMemory mem = null ;
    
    /**
     * セマフォー名.
     */
    private String semName = null ;
    
    /**
     * 共有メモリー名.
     */
    private String shareName = null ;
    
    /**
     * コンストラクタ.
     */
    private CoreShm() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR>
     * @param mode [true]の場合サーバモードで生成します.
     * @param semName 対象のセマフォー名を設定します.
     * @param shareName 対象の共有名を設定します.
     * @param length 対象の共有メモリー幅を設定します.
     * @exception Exception 例外.
     */
    public CoreShm( boolean mode,String semName,String shareName,int length )
        throws Exception {
        try {
            this.mem = new NativeSharedMemory( ( mode == true ) ? 0 : 1,semName,shareName,length ) ;
            this.semName = semName ;
            this.shareName = shareName ;
            if( mode == true ) {
                this.write( new byte[ length ],0,0,length ) ;
            }
        } catch( Exception e ) {
            this.destroy() ;
            throw e ;
        }
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     */
    public void destroy() {
        if( getMem() != null ) {
            getMem().destroy() ;
        }
        semName = null ;
        shareName = null ;
    }
    
    /**
     * セマフォ名を取得.
     * <BR>
     * @return String セマフォ名が返されます.
     */
    public String getSemaphoreName() {
        if( isUse() == false ) {
            return null ;
        }
        return this.semName ;
    }
    
    /**
     * 共有メモリ名を取得.
     * <BR>
     * @return String 共有メモリ名が返されます.
     */
    public String getShareName() {
        if( isUse() == false ) {
            return null ;
        }
        return this.shareName ;
    }
    
    /**
     * 共有メモリー最大幅を取得.
     * <BR>
     * @return int 共有メモリ最大幅を取得します.
     */
    public int getShareLength() {
        if( isUse() == false ) {
            return -1 ;
        }
        return getMem().size() ;
    }
    
    /**
     * 共有メモリに情報セット.
     * <BR>
     * @param binary 対象のバイナリを設定します.
     * @param pos 対象のポジションを設定します.
     * @param off 対象のオフセット値を設定します.
     * @param length 対象のデータ長を設定します.
     * @return int 書き込まれたサイズが返されます.
     * @exception Exception 例外.
     */
    public int write( byte[] binary,int pos,int off,int length )
        throws Exception {
        return write( true,binary,pos,off,length ) ;
    }
    
    /**
     * 共有メモリに情報セット.
     * <BR>
     * @param mode [true]の場合、同期を取ります.
     * @param binary 対象のバイナリを設定します.
     * @param pos 対象のポジションを設定します.
     * @param off 対象のオフセット値を設定します.
     * @param length 対象のデータ長を設定します.
     * @return int 書き込まれたサイズが返されます.
     * @exception Exception 例外.
     */
    public int write( boolean mode,byte[] binary,int pos,int off,int length )
        throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        int ln = getMem().size() ;
        if( binary == null || binary.length <= 0 || pos <= -1 || pos >= ln ||
            off <= -1 || off >= binary.length || length <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( off + length >= binary.length ) {
            length = binary.length - off ;
        }
        if( pos + length >= ln ) {
            length = ln - pos ;
        }
        getMem().write( ( mode == false ) ? 0 : 1,binary,pos,off,length ) ;
        return length ;
    }
    
    /**
     * 共有メモリから情報取得.
     * <BR>
     * @param binary 対象のバイナリを設定します.
     * @param pos 対象のポジションを設定します.
     * @param off 対象のオフセット値を設定します.
     * @param length 対象のデータ長を設定します.
     * @return int 読み込まれたサイズが返されます.
     * @exception Exception 例外.
     */
    public int read( byte[] binary,int pos,int off,int length )
        throws Exception {
        return read( true,binary,pos,off,length ) ;
    }
    
    /**
     * 共有メモリから情報取得.
     * <BR>
     * @param mode [true]の場合、同期を取ります.
     * @param binary 対象のバイナリを設定します.
     * @param pos 対象のポジションを設定します.
     * @param off 対象のオフセット値を設定します.
     * @param length 対象のデータ長を設定します.
     * @return int 読み込まれたサイズが返されます.
     * @exception Exception 例外.
     */
    public int read( boolean mode,byte[] binary,int pos,int off,int length )
        throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        int ln = getMem().size() ;
        if( binary == null || binary.length <= 0 || pos <= -1 || pos >= ln ||
            off <= -1 || off >= binary.length || length <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( off + length >= binary.length ) {
            length = binary.length - off ;
        }
        if( pos + length >= ln ) {
            length = ln - pos ;
        }
        getMem().read( ( mode == false ) ? 0 : 1,binary,pos,off,length ) ;
        return length ;
    }
    
    /**
     * 現在の共有メモリ内容をファイル出力.
     * <BR>
     * @param name 出力先のファイル名を設定します.
     * @exception Exception 例外.
     */
    public void outputFile( String name ) throws Exception {
        int len = getShareLength() ;
        byte[] b = new byte[ len ] ;
        getMem().read( 1,b,0,0,len ) ;
        BufferedOutputStream bo = new BufferedOutputStream( new FileOutputStream( name ) ) ;
        bo.write( b ) ;
        bo.flush() ;
        bo.close() ;
    }
    
    /**
     * 現在の共有メモリ内容を出力.
     * <BR>
     * @param output 出力先のオブジェクトを設定します.
     * @exception Exception 例外.
     */
    public void outputFile( OutputStream output ) throws Exception {
        int len = getShareLength() ;
        byte[] b = new byte[ len ] ;
        getMem().read( 1,b,0,0,len ) ;
        output.write( b ) ;
        output.flush() ;
    }
    
    /**
     * ロック処理を実行.
     * <BR>
     * @exception Exception 例外.
     */
    public void lock() throws Exception {
        getMem().semLock() ;
    }
    
    /**
     * ロック処理を解除.
     * <BR>
     * @exception Exception 例外.
     */
    public void unLock() throws Exception {
        getMem().semUnLock() ;
    }
    
    /**
     * オブジェクト有効チェック.
     * <BR>
     * @return boolean [true]の場合、有効です.
     */
    public boolean isUse() {
        return ( getMem() != null ) ;
    }
    
    private synchronized NativeSharedMemory getMem() {
        if( mem == null ) {
            return null ;
        }
        if( mem.size() <= -1 ) {
            mem = null ;
            return null ;
        }
        return mem ;
    }
}

