package org.maachang.shm;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.io.OutputStream;

import org.maachang.shm.core.CoreShm;

/**
 * SharedMemory用OutputStream.
 *  
 * @version 2008/02/17
 * @author  masahito suzuki
 * @since  SharedMemory 1.01
 */
class ShmOutputStream extends OutputStream {
    private CoreShm shm = null ;
    private AbstractShmConnector conn = null ;
    private boolean mode = false ;
    private int no = -1 ;
    private int sequenceId = -1 ;
    private int length = -1 ;
    private int timeout = 0 ;
    private int pos = 0 ;
    private int beforeState = -1 ;
    private long beforeTimeout = -1L ;
    private byte[] ioBinary = null ;
    
    private ShmOutputStream() {
        
    }
    
    public ShmOutputStream( CoreShm shm,AbstractShmConnector conn,boolean mode,int timeout,int no,int sequenceId,int length )
        throws Exception {
        if( shm == null || shm.isUse() == false ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( timeout <= -1 ) {
            timeout = 0 ;
        }
        this.shm = shm ;
        this.conn = conn ;
        this.mode = mode ;
        this.no = no ;
        this.sequenceId = sequenceId ;
        this.length = length ;
        this.timeout = timeout ;
        this.ioBinary = new byte[ ShmDefine.ONE_DATA_LENGTH ] ;
        beforeTimeout() ;
    }
    
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    public void destroy() {
        if( shm != null ) {
            try {
                close() ;
            } catch( Exception e ) {
            }
        }
        shm = null ;
        conn = null ;
        mode = false ;
        no = -1 ;
        sequenceId = -1 ;
        length = -1 ;
        timeout = 0 ;
        pos = 0 ;
        beforeState = -1 ;
        ioBinary = null ;
    }
    
    public void close() throws IOException {
        if( shm != null && pos > 0 ) {
            this.flush( false ) ;
        }
    }
    
    public void flush() throws IOException {
        flush( false ) ;
    }
    
    public void flush( boolean overMode ) throws IOException {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        beforeTimeout() ;
        try {
            long t = System.currentTimeMillis() + ( long )timeout ;
            for( ;; ) {
                if( writeBody( overMode ) == true ) {
                    break ;
                }
                if( timeout > 0 && t <= System.currentTimeMillis() ) {
                    throw new InterruptedIOException( "timeout" ) ;
                }
                Thread.sleep( 15L ) ;
            }
        } catch( NotTargetAccessException na ) {
            conn.notTargetByDestroy() ;
            throw na ;
        } catch( IOException io ) {
            throw io ;
        } catch( Exception e ) {
            throw new IOException( e.getMessage() ) ;
        }
    }
    
    public void write(int b) throws IOException {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        beforeTimeout() ;
        if( pos >= ShmDefine.ONE_DATA_LENGTH ) {
            flush( true ) ;
        }
        ioBinary[ pos ] = ( byte )( b & 0x000000ff ) ;
        pos ++ ;
    }
    
    public boolean getMode() {
        return mode ;
    }
    
    public int getNo() {
        return no ;
    }
    
    public int getSequenceId() {
        return sequenceId ;
    }
    
    public int getLength() {
        return length ;
    }
    
    public int getTimeout() {
        return timeout ;
    }
    
    public int getBeforeState() {
        return beforeState ;
    }
    
    public boolean isTimeout() {
        long time = getBeforeTimeout() ;
        return ( timeout > 0 && time + ( long )timeout <= System.currentTimeMillis() ) ;
    }
    
    public boolean isUse() {
        return ( shm != null && shm.isUse() == true ) ;
    }
    
    public String toString() {
        if( isUse() == false ) {
            return "ShmOutputStream is closed." ;
        }
        return new StringBuilder().
            append( "ShmOutputStream" ).
            append( "[" ).
            append( shm.getSemaphoreName() ).
            append( "," ).
            append( shm.getShareName() ).
            append( "," ).
            append( shm.getShareLength() ).
            append( "]" ).
            append( " mode:" ).
            append( mode ).
            append( " no:" ).
            append( no ).
            append( " sequenceId:" ).
            append( sequenceId ).
            append( " length:" ).
            append( length ).
            append( " timeout:" ).
            append( timeout ).
            append( " pos:" ).
            append( pos ).
            append( " beforeState:" ).
            append( beforeState ).
            toString() ;
    }
    
    private boolean writeBody( boolean overMode ) throws Exception {
        int[] state = new int[ 1 ] ;
        if( ShmCommon.writeBody( mode,state,overMode,ioBinary,pos,shm,sequenceId,no,length ) == true ) {
            beforeState = state[ 0 ] ;
            pos = 0 ;
            return true ;
        }
        else if( state[ 0 ] == -1 ) {
            throw new IOException( "範囲外の位置にI/O処理を行っています" ) ;
        }
        beforeState = state[ 0 ] ;
        return false ;
    }
    
    private synchronized long getBeforeTimeout() {
        return beforeTimeout ;
    }
    
    private synchronized void beforeTimeout() {
        beforeTimeout = System.currentTimeMillis() ;
    }
}
