package org.maachang.shm;

import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;

import org.maachang.shm.core.CoreShm;

/**
 * SharedMemory用InputStream.
 *  
 * @version 2008/02/16
 * @author  masahito suzuki
 * @since  SharedMemory 1.01
 */
class ShmInputStream extends InputStream {
    private static final int MAX_RETRY_LOOP = 64 ;
    private CoreShm shm = null ;
    private AbstractShmConnector conn = null ;
    private boolean mode = false ;
    private int no = -1 ;
    private int sequenceId = -1 ;
    private int length = -1 ;
    private int timeout = 0 ;
    private int pos = 0 ;
    private int bodyLen = -1 ;
    private int beforeState = -1 ;
    private long beforeTimeout = -1L ;
    private int retryNextData = 0 ;
    private byte[] ioBinary = null ;
    
    private ShmInputStream() {
        
    }
    
    protected ShmInputStream( CoreShm shm,AbstractShmConnector conn,boolean mode,int timeout,int no,int sequenceId,int length )
        throws Exception {
        if( shm == null || shm.isUse() == false ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( timeout <= -1 ) {
            timeout = 0 ;
        }
        this.shm = shm ;
        this.conn = conn ;
        this.mode = mode ;
        this.no = no ;
        this.sequenceId = sequenceId ;
        this.length = length ;
        this.timeout = timeout ;
        this.ioBinary = null ;
        beforeTimeout() ;
    }
    
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    public void destroy() {
        shm = null ;
        conn = null ;
        mode = false ;
        no = -1 ;
        sequenceId = -1 ;
        length = -1 ;
        timeout = 0 ;
        pos = 0 ;
        bodyLen = -1 ;
        beforeState = -1 ;
        ioBinary = null ;
    }
    
    public void close() {
        
    }
    
    public int read() throws IOException {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既にクローズしています" ) ;
        }
        long t = System.currentTimeMillis() + ( long )timeout ;
        try {
            for( ;; ) {
                readBody() ;
                if( bodyLen <= -1 || bodyLen <= pos ) {
                    if( bodyLen >= 0 && bodyLen < ShmDefine.ONE_DATA_LENGTH ) {
                        return -1 ;
                    }
                    if( timeout > 0 && t <= System.currentTimeMillis() ) {
                        throw new InterruptedIOException( "timeout" ) ;
                    }
                    Thread.sleep( 5L ) ;
                }
                else {
                    int p = ( int )( ioBinary[ pos ] & 0x000000ff ) ;
                    pos ++ ;
                    beforeTimeout() ;
                    return p ;
                }
            }
        } catch( NotTargetAccessException na ) {
            conn.notTargetByDestroy() ;
            throw na ;
        } catch( IOException io ) {
            throw io ;
        } catch( Exception e ) {
            throw new IOException( e.getMessage() ) ;
        }
    }
    
    public int available() throws IOException {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既にクローズしています" ) ;
        }
        beforeTimeout() ;
        try {
            readBody() ;
            if( bodyLen <= -1 ) {
                return 0 ;
            }
            return bodyLen - pos ;
        } catch( IOException io ) {
            throw io ;
        } catch( Exception e ) {
            throw new IOException( e.getMessage() ) ;
        }
    }
    
    public boolean getMode() {
        return mode ;
    }
    
    public int getNo() {
        return no ;
    }
    
    public int getSequenceId() {
        return sequenceId ;
    }
    
    public int getLength() {
        return length ;
    }
    
    public int getTimeout() {
        return timeout ;
    }
    
    public int getBeforeState() {
        return beforeState ;
    }
    
    public boolean isTimeout() {
        long time = getBeforeTimeout() ;
        return ( timeout > 0 && time + ( long )timeout <= System.currentTimeMillis() ) ;
    }
    
    public boolean isUse() {
        return ( shm != null && shm.isUse() == true ) ;
    }
    
    public String toString() {
        if( isUse() == false ) {
            return "ShmInputStream is closed." ;
        }
        return new StringBuilder().
            append( "ShmInputStream" ).
            append( "[" ).
            append( shm.getSemaphoreName() ).
            append( "," ).
            append( shm.getShareName() ).
            append( "," ).
            append( shm.getShareLength() ).
            append( "]" ).
            append( " mode:" ).
            append( mode ).
            append( " no:" ).
            append( no ).
            append( " sequenceId:" ).
            append( sequenceId ).
            append( " length:" ).
            append( length ).
            append( " timeout:" ).
            append( timeout ).
            append( " pos:" ).
            append( pos ).
            append( " bodyLength:" ).
            append( bodyLen ).
            append( " beforeState:" ).
            append( beforeState ).
            toString() ;
    }
    
    private boolean readBody()
        throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        // bodyが読まれていない、またはバッファ数を越した読み込みの場合.
        if( bodyLen <= -1 || ( bodyLen >= ShmDefine.ONE_DATA_LENGTH && bodyLen <= pos ) ) {
            if( ioBinary == null ) {
                ioBinary = new byte[ ShmDefine.ONE_DATA_LENGTH ] ;
            }
            int[] state = new int[ 1 ] ;
            int len = -1 ;
            for( ;; ) {
                len = ShmCommon.readBody( mode,state,ioBinary,shm,sequenceId,no,length ) ;
                if( state[ 0 ] == -1 ) {
                    throw new IOException( "範囲外の位置にI/O処理を行っています" ) ;
                }
                else if( len <= -1 ) {
                    if( state[ 0 ] == ShmDefine.STATE_FREE ) {
                        throw new IOException( "接続は既に閉じられています" ) ;
                    }
                    if( retryNextData <= 0 ) {
                        retryNextData = 0 ;
                        return false ;
                    }
                    retryNextData -- ;
                    Thread.sleep( 5L ) ;
                    continue ;
                }
                if( retryNextData > 0 ) {
                    retryNextData = 0 ;
                }
                break ;
            }
            if( state[ 0 ] == ShmDefine.STATE_SET ) {
                ShmCommon.setState( mode,ShmDefine.STATE_NEXT,shm,no,length ) ;
            }
            if( len > ShmDefine.ONE_DATA_LENGTH ) {
                bodyLen = ShmDefine.ONE_DATA_LENGTH ;
                retryNextData = MAX_RETRY_LOOP ;
            }
            else {
                bodyLen = len ;
                retryNextData = 0 ;
            }
            pos = 0 ;
            return true ;
        }
        // データ取得数が、バッファを越さない状態で、データ終了の場合.
        else if( bodyLen >= 0 && bodyLen < ShmDefine.ONE_DATA_LENGTH && bodyLen <= pos ) {
            int nowState = ShmCommon.getState( mode,shm,no,length ) ;
            if( nowState == ShmDefine.STATE_FREE ) {
                throw new IOException( "接続は既に閉じられています" ) ;
            }
            else if( nowState == ShmDefine.STATE_SET ) {
                ShmCommon.setState( mode,ShmDefine.STATE_NEXT,shm,no,length ) ;
            }
            bodyLen = -1 ;
            pos = 0 ;
        }
        return false ;
    }
    
    private synchronized long getBeforeTimeout() {
        return beforeTimeout ;
    }
    
    private synchronized void beforeTimeout() {
        beforeTimeout = System.currentTimeMillis() ;
    }
}
