package org.maachang.shm;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import org.maachang.shm.core.CoreShm;

/**
 * SharedMemory用接続実装オブジェクト.
 *  
 * @version 2008/02/16
 * @author  masahito suzuki
 * @since  SharedMemory 1.01
 */
class ShmConnectorImpl
    extends AbstractShmConnector implements ShmConnector {
    
    /**
     * coreSharedMemory.
     */
    private CoreShm shm = null ;
    
    /**
     * InputStream.
     */
    private ShmInputStream input = null ;
    
    /**
     * outputStream.
     */
    private ShmOutputStream output = null ;
    
    /**
     * 実行モード.
     */
    private boolean mode = false ;
    
    /**
     * シーケンスID.
     */
    private int sequenceId = -1 ;
    
    /**
     * データポジション.
     */
    private int position = -1 ;
    
    /**
     * I/O個数.
     */
    private int length = -1 ;
    
    /**
     * I/Oタイムアウト値.
     */
    private int timeout = -1 ;
    
    /**
     * コンストラクタ.
     */
    private ShmConnectorImpl() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * クライアントモードで接続します.
     * <BR>
     * @param shm CoreSharedMemoryオブジェクトを設定します.
     * @param length I/O個数を設定します.
     * @param timeout タイムアウト値を設定します.<BR>
     *                [0]以下の場合、無限待ちとなります.
     * @exception Exception 例外.
     */
    protected ShmConnectorImpl( CoreShm shm,int length,int timeout )
        throws Exception {
        if( shm == null || shm.isUse() == false || length <= 0 || length > ShmDefine.MAX_IO_LENGTH ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( timeout <= 0 ) {
            timeout = 0 ;
        }
        // 空き条件を検索.
        int[] res = ShmCommon.reservationSpace( shm,length ) ;
        if( res == null ) {
            throw new IOException( "接続空きが存在しません" ) ;
        }
        try {
            this.shm = shm ;
            this.mode = false ;
            this.sequenceId = res[ 0 ] ;
            this.position = res[ 1 ] ;
            this.length = length ;
            this.timeout = timeout ;
            this.input = new ShmInputStream( this.shm,this,false,this.timeout,
                this.position,this.sequenceId,this.length ) ;
            this.output = new ShmOutputStream( this.shm,this,true,this.timeout,
                this.position,this.sequenceId,this.length ) ;
        } catch( Exception e ) {
            this.close() ;
            throw e ;
        }
    }
    
    /**
     * コンストラクタ.
     * <BR><BR.
     * サーバモードで接続します.
     * <BR>
     * @parma shm CoreSharedMemoryオブジェクトを設定します.
     * @param position 接続位置を設定します.
     * @param length I/O個数を設定します.
     * @param timeout タイムアウト値を設定します.<BR>
     *                [0]以下の場合、無限待ちとなります.
     * @exception Exception 例外.
     */
    protected ShmConnectorImpl( CoreShm shm,int position,int length,int timeout )
        throws Exception {
        if( shm == null || shm.isUse() == false || position < 0 || position >= ShmDefine.MAX_IO_LENGTH ||
            length <= 0 || length > ShmDefine.MAX_IO_LENGTH ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        if( timeout <= 0 ) {
            timeout = 0 ;
        }
        try {
            int sequenceId = ShmCommon.getBodySequenceId( false,shm,position,length ) ;
            if( sequenceId <= ShmDefine.NOT_SEQUENCE ) {
                throw new IOException( "指定位置("+position+")には、接続条件が存在しません" ) ;
            }
            this.shm = shm ;
            this.mode = true ;
            this.sequenceId = sequenceId ;
            this.position = position ;
            this.length = length ;
            this.timeout = timeout ;
            this.input = new ShmInputStream( this.shm,this,true,this.timeout,
                this.position,this.sequenceId,this.length ) ;
            this.output = new ShmOutputStream( this.shm,this,false,this.timeout,
                this.position,this.sequenceId,this.length ) ;
        } catch( Exception e ) {
            this.close() ;
            throw e ;
        }
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        try {
            this.close() ;
        } catch( Exception e ) {
        }
    }
    
    /**
     * 他接続検知によるコネクション破棄.
     */
    protected synchronized void notTargetByDestroy() {
        if( input != null ) {
            input.destroy() ;
        }
        //if( output != null ) {
        //    output.destroy() ;
        //}
        shm = null ;
        input = null ;
        output = null ;
        sequenceId = -1 ;
        position = -1 ;
        length = -1 ;
        timeout = -1 ;
    }
    
    /**
     * コネクションクローズ.
     * <BR>
     * @exception Exception 例外.
     */
    public synchronized void close() throws Exception {
        if( shm != null ) {
            // コネクション破棄.
            ShmCommon.freeSpace( shm,sequenceId,position,length ) ;
        }
        notTargetByDestroy() ;
    }
    
    /**
     * InputStreamを取得.
     * <BR>
     * @return InputStream InputStreamが返されます.
     */
    public synchronized InputStream getInputStream() {
        if( isConnect() == false ) {
            return null ;
        }
        return input ;
    }
    
    /**
     * OutputStreamを取得.
     * <BR>
     * @return OutputStream OutputStreamが返されます.
     */
    public synchronized OutputStream getOutputStream() {
        if( isConnect() == false ) {
            return null ;
        }
        return output ;
    }
    
    /**
     * 接続IDを取得.
     * <BR>
     * @return int 接続IDが返されます.
     */
    public synchronized int getId() {
        return sequenceId ;
    }
    
    /**
     * 接続タイムアウトを取得.
     * <BR>
     * @return int 接続タイムアウトが返されます.
     */
    public synchronized int getTimeout() {
        return timeout ;
    }
    
    /**
     * オブジェクトモードを取得.
     * <BR>
     * @return boolean [true]の場合、サーバモードです.
     */
    public synchronized boolean isMode() {
        return mode ;
    }
    
    /**
     * タイムアウトであるかチェック.
     * <BR>
     * @return boolean [true]の場合、タイムアウトしています.
     */
    public synchronized boolean isTimeout() {
        return ( input.isTimeout() == true || output.isTimeout() == true ) ;
    }
    
    /**
     * 接続状態を確認.
     * <BR>
     * @return boolean [true]の場合、接続中です.
     */
    public synchronized boolean isConnect() {
        try {
            if( shm == null || shm.isUse() == false ) {
                return false ;
            }
            int seqId = ShmCommon.getBodySequenceId( false,shm,position,length ) ;
            int seqWId = ShmCommon.getBodySequenceId( true,shm,position,length ) ;
            if( seqId <= -1 || seqId != sequenceId || seqWId <= -1 || seqWId != sequenceId ) {
                return false ;
            }
            return true ;
        } catch( Exception e ) {
        }
        return false ;
    }
    
    public synchronized String toString() {
        return new StringBuilder().append( "mode:" ).append( mode ).
            append( " sequenceId:" ).append( sequenceId ).
            append( " position:" ).append( position ).
            append( " length:" ).append( length ).
            append( " timeout:" ).append( timeout ).toString() ;
    }
    
    public synchronized void debugOutFile( String name ) throws Exception {
        shm.outputFile( name ) ;
    }
    
    public void debugOutputStream( OutputStream output ) throws Exception {
        shm.outputFile( output ) ;
    }
}
