package org.maachang.shm;

import java.util.ArrayList;

import org.maachang.shm.core.CoreShm;

/**
 * SharedMemory用共通定義.
 *  
 * @version 2008/02/16
 * @author  masahito suzuki
 * @since  SharedMemory 1.01
 */
class ShmCommon {
    /**
     * 共有メモリサイズを計算.
     * <BR>
     * @param length 対象の共有I/O数を設定します.
     * @return int 共有メモリサイズが返されます.
     * @exception Exception 例外.
     */
    public static final int maxIOLength( int length )
        throws Exception {
        if( length <= 0 || length > ShmDefine.MAX_IO_LENGTH ) {
            throw new IllegalArgumentException( "共有I/Oサイズ("+length+")が不正です" ) ;
        }
        length = trimLength( length ) ;
        int ret = ShmDefine.SEQUENCE_LENGTH + ShmDefine.USE_CONNECT_NO + length +
            ( ( ShmDefine.ONE_HEADER_LENGTH + ShmDefine.ONE_DATA_LENGTH ) * ( ( length * 8 ) * 2 ) ) ;
        return ret ;
    }
    
    /**
     * データ長さを整形.
     * <BR>
     * @param length データの長さを整形します.
     * @return int 整形された長さが返されます.
     */
    public static final int trimLength( int length ) {
        return ( ( length & 0x00000007 ) == 0 ? 0 : 1 ) +
            ( ( length & 0x7ffffff8 ) >> 3 ) ;
    }
    
    /**
     * データ開始ポジションを取得.
     * <BR>
     * @param mode [true]の場合、書き込み情報位置です.
     * @param shm SharedMemoryを設定します.
     * @param no 対象の項番を設定します.
     * @param length データ個数を設定します.
     * @return int 開始位置が返されます.
     * @exception Exception 例外.
     */
    public static final int startPoint( boolean mode,CoreShm shm,int no,int length )
        throws Exception {
        int pkt = ShmDefine.ONE_HEADER_LENGTH + ShmDefine.ONE_DATA_LENGTH ;
        int ret = ( ( no * 2 ) * pkt ) + trimLength( length ) + ShmDefine.SEQUENCE_LENGTH + ShmDefine.USE_CONNECT_NO ;
        if( mode == true ) {
            ret += pkt ;
        }
        if( shm.getShareLength() <= ret ) {
            return -1 ;
        }
        return ret ;
    }
    
    /**
     * 現在のシーケンスIDを取得.
     * <BR>
     * @param shm SharedMemoryを設定します.
     * @return int シーケンスIDが返されます.
     * @exception Exception 例外.
     */
    public static final int nowSequenceId( CoreShm shm )
        throws Exception {
        byte[] b = new byte[ 4 ] ;
        shm.read( b,0,0,ShmDefine.SEQUENCE_LENGTH ) ;
        return ( ( int )b[ 0 ] & 0x000000ff ) |
            ( ( ( int )b[ 1 ] & 0x000000ff ) << 8 ) |
            ( ( ( int )b[ 2 ] & 0x000000ff ) << 16 ) |
            ( ( ( int )b[ 3 ] & 0x000000ff ) << 24 ) ;
    }
    
    /**
     * 更新されたシーケンスIDを取得.
     * <BR>
     * @param sync 同期を取る場合は[true]を設定します.
     * @param shm SharedMemoryを設定します.
     * @return int シーケンスIDが返されます.
     * @exception Exception 例外.
     */
    public static final int updateSequenceId( boolean sync,CoreShm shm )
        throws Exception {
        try {
            if( sync == true ) {
                shm.lock() ;
            }
            byte[] b = new byte[ 4 ] ;
            shm.read( false,b,0,0,ShmDefine.SEQUENCE_LENGTH ) ;
            int ret = ( ( int )b[ 0 ] & 0x000000ff ) |
                ( ( ( int )b[ 1 ] & 0x000000ff ) << 8 ) |
                ( ( ( int )b[ 2 ] & 0x000000ff ) << 16 ) |
                ( ( ( int )b[ 3 ] & 0x000000ff ) << 24 ) ;
            if( ret >= Integer.MAX_VALUE ) {
                ret = 0 ;
            }
            else {
                ret ++ ;
            }
            b[ 0 ] = ( byte )( ret & 0x000000ff ) ;
            b[ 1 ] = ( byte )( ( ret & 0x0000ff00 ) >> 8 ) ;
            b[ 2 ] = ( byte )( ( ret & 0x00ff0000 ) >> 16 ) ;
            b[ 3 ] = ( byte )( ( ret & 0xff000000 ) >> 24 ) ;
            shm.write( false,b,0,0,ShmDefine.SEQUENCE_LENGTH ) ;
            return ret ;
        } finally {
            if( sync == true ) {
                shm.unLock() ;
            }
        }
    }
    
    /**
     * ステータスを取得.
     * <BR>
     * @param mode [true]の場合、書き込み情報位置です.
     * @param shm SharedMemoryを設定します.
     * @param no 対象の項番を設定します.
     * @param length データ個数を設定します.
     * @return int ステータスが返されます.
     * @exception Exception 例外.
     */
    public static final int getState( boolean mode,CoreShm shm,int no,int length )
        throws Exception {
        int p = startPoint( mode,shm,no,length ) ;
        if( p <= -1 ) {
            return -1 ;
        }
        byte[] b = new byte[ 1 ] ;
        shm.read( b,p,0,1 ) ;
        return ( ( int )b[ 0 ] & 0x000000ff ) ;
    }
    
    /**
     * ステータスを設定.
     * <BR>
     * @param mode [true]の場合、書き込み情報位置です.
     * @param state 対象のステータスを設定します.
     * @param shm SharedMemoryを設定します.
     * @param no 対象の項番を設定します.
     * @param length データ個数を設定します.
     * @return boolean [true]の場合、正常に書き込めました.
     * @exception Exception 例外.
     */
    public static final boolean setState( boolean mode,int state,CoreShm shm,int no,int length )
        throws Exception {
        int p = startPoint( mode,shm,no,length ) ;
        if( p <= -1 ) {
            return false ;
        }
        byte[] b = new byte[ 1 ] ;
        b[ 0 ] = ( byte )( state & 0x000000ff ) ;
        shm.write( b,p,0,1 ) ;
        return true ;
    }
    
    /**
     * BodyシーケンスIDを取得.
     * <BR>
     * @param mode [true]の場合、書き込み情報位置です.
     * @param shm SharedMemoryを設定します.
     * @param no 対象の項番を設定します.
     * @param length データ個数を設定します.
     * @return int BodyシーケンスIDが返されます.
     * @exception Exception 例外.
     */
    public static final int getBodySequenceId( boolean mode,CoreShm shm,int no,int length )
        throws Exception {
        int p = startPoint( mode,shm,no,length ) ;
        if( p <= -1 ) {
            return -1 ;
        }
        p += 1 ;
        byte[] b = new byte[ 4 ] ;
        shm.read( b,p,0,4 ) ;
        return ( ( int )b[ 0 ] & 0x000000ff ) |
            ( ( ( int )b[ 1 ] & 0x000000ff ) << 8 ) |
            ( ( ( int )b[ 2 ] & 0x000000ff ) << 16 ) |
            ( ( ( int )b[ 3 ] & 0x000000ff ) << 24 ) ;
    }
    
    /**
     * BodyシーケンスIdを設定.
     * <BR>
     * @param mode [true]の場合、書き込み情報位置です.
     * @param seq 対象のBodyシーケンスIDを設定します.
     * @param shm SharedMemoryを設定します.
     * @param no 対象の項番を設定します.
     * @param length データ個数を設定します.
     * @return boolean [true]の場合、正常に書き込めました.
     * @exception Exception 例外.
     */
    public static final boolean setBodySequenceId( boolean mode,int seq,CoreShm shm,int no,int length )
        throws Exception {
        int p = startPoint( mode,shm,no,length ) ;
        if( p <= -1 ) {
            return false ;
        }
        p += 1 ;
        byte[] b = new byte[ 4 ] ;
        b[ 0 ] = ( byte )( seq & 0x000000ff ) ;
        b[ 1 ] = ( byte )( ( seq & 0x0000ff00 ) >> 8 ) ;
        b[ 2 ] = ( byte )( ( seq & 0x00ff0000 ) >> 16 ) ;
        b[ 3 ] = ( byte )( ( seq & 0xff000000 ) >> 24 ) ;
        shm.write( b,p,0,4 ) ;
        return true ;
    }
    
    /**
     * Body長を取得.
     * <BR>
     * @param mode [true]の場合、書き込み情報位置です.
     * @param shm SharedMemoryを設定します.
     * @param no 対象の項番を設定します.
     * @param length データ個数を設定します.
     * @return int Body長が返されます.
     * @exception Exception 例外.
     */
    public static final int getBodyLength( boolean mode,CoreShm shm,int no,int length )
        throws Exception {
        int p = startPoint( mode,shm,no,length ) ;
        if( p <= -1 ) {
            return -1 ;
        }
        p += 5 ;
        byte[] b = new byte[ 2 ] ;
        shm.read( b,p,0,2 ) ;
        return ( ( int )b[ 0 ] & 0x000000ff ) |
            ( ( ( int )b[ 1 ] & 0x000000ff ) << 8 ) ;
    }
    
    /**
     * Body長を設定.
     * <BR>
     * @param mode [true]の場合、書き込み情報位置です.
     * @param bodyLength 対象のBody長を設定します.
     * @param shm SharedMemoryを設定します.
     * @param no 対象の項番を設定します.
     * @param length データ個数を設定します.
     * @return boolean [true]の場合、正常に書き込めました.
     * @exception Exception 例外.
     */
    public static final boolean setBodyLength( boolean mode,int bodyLength,CoreShm shm,int no,int length )
        throws Exception {
        int p = startPoint( mode,shm,no,length ) ;
        if( p <= -1 ) {
            return false ;
        }
        p += 5 ;
        byte[] b = new byte[ 2 ] ;
        b[ 0 ] = ( byte )( bodyLength & 0x000000ff ) ;
        b[ 1 ] = ( byte )( ( bodyLength & 0x0000ff00 ) >> 8 ) ;
        shm.write( b,p,0,2 ) ;
        return true ;
    }
    
    /**
     * Bodyデータを取得.
     * <BR>
     * @param mode [true]の場合、書き込み情報位置です.
     * @param outState 現在のステータスが返されます.
     * @param binary 読み込み対象バイナリを設定します.
     * @param shm SharedMemoryを設定します.
     * @param sequenceId 対象のシーケンスIDを設定します.
     * @param no 対象の項番を設定します.
     * @param length データ個数を設定します.
     * @return int 取得された長さが返されます.<BR>
     *             [-1]の場合、情報取得に失敗しました.
     * @exception Exception 例外.
     */
    public static final int readBody( boolean mode,int[] outState,byte[] binary,CoreShm shm,int sequenceId,int no,int length )
        throws Exception {
        try {
            shm.lock() ;
            byte[] b = null ;
            int p = startPoint( mode,shm,no,length ) ;
            if( p <= -1 ) {
                if( outState != null ) {
                    outState[ 0 ] = -1 ;
                }
                return -1 ;
            }
            b = new byte[ 4 ] ;
            shm.read( false,b,p+1,0,4 ) ;
            int seqId = ( ( int )b[ 0 ] & 0x000000ff ) |
                ( ( ( int )b[ 1 ] & 0x000000ff ) << 8 ) |
                ( ( ( int )b[ 2 ] & 0x000000ff ) << 16 ) |
                ( ( ( int )b[ 3 ] & 0x000000ff ) << 24 ) ;
            b = null ;
            if( seqId <= -1 || seqId != sequenceId ) {
                throw new NotTargetAccessException( "現在の接続位置("+no+")は、別の接続条件(this:" + sequenceId +
                    " target:" + seqId + ")が利用しています" ) ;
            }
            b = new byte[ 1 ] ;
            shm.read( false,b,p,0,1 ) ;
            int state = ( ( int )b[ 0 ] & 0x000000ff ) ;
            if( outState != null ) {
                outState[ 0 ] = state ;
            }
            b = null ;
            if( state != ShmDefine.STATE_SET ) {
                return -1 ;
            }
            b = new byte[ 2 ] ;
            shm.read( false,b,p+5,0,2 ) ;
            int ret = ( ( int )b[ 0 ] & 0x000000ff ) |
                ( ( ( int )b[ 1 ] & 0x000000ff ) << 8 ) ;
            int readLen = ret ;
            if( readLen >= ShmDefine.ONE_DATA_LENGTH ) {
                readLen = ShmDefine.ONE_DATA_LENGTH ;
            }
            shm.read( false,binary,p+ShmDefine.ONE_HEADER_LENGTH,0,readLen ) ;
            return ret ;
        } finally {
            shm.unLock() ;
        }
    }
    
    /**
     * Bodyデータを書き込む.
     * <BR>
     * @param mode [true]の場合、書き込み情報位置です.
     * @param outState 書き込み前のステータスが返されます.
     * @param overMode 最大書き込み長を越す書き込み領域が存在する場合は[true].
     * @param binary 読み込み対象バイナリを設定します.
     * @param bodyLen Body長を設定します.
     * @param shm SharedMemoryを設定します.
     * @param sequenceId 対象のシーケンスIDを設定します.
     * @param no 対象の項番を設定します.
     * @param length データ個数を設定します.
     * @return boolean [true]の場合、書き込み成功です.
     * @exception Exception 例外.
     */
    public static final boolean writeBody( boolean mode,int[] outState,boolean overMode,
        byte[] binary,int bodyLen,CoreShm shm,int sequenceId,int no,int length )
        throws Exception {
        if( bodyLen <= 0 ) {
            return true ;
        }
        try {
            shm.lock() ;
            byte[] b = null ;
            int p = startPoint( mode,shm,no,length ) ;
            if( p <= -1 ) {
                if( outState != null ) {
                    outState[ 0 ] = -1 ;
                }
                return false ;
            }
            b = new byte[ 4 ] ;
            shm.read( false,b,p+1,0,4 ) ;
            int seqId = ( ( int )b[ 0 ] & 0x000000ff ) |
                ( ( ( int )b[ 1 ] & 0x000000ff ) << 8 ) |
                ( ( ( int )b[ 2 ] & 0x000000ff ) << 16 ) |
                ( ( ( int )b[ 3 ] & 0x000000ff ) << 24 ) ;
            b = null ;
            if( seqId <= -1 || seqId != sequenceId ) {
                throw new NotTargetAccessException( "現在の接続位置("+no+")は、別の接続条件(this:" + sequenceId +
                    " target:" + seqId + ")が利用しています" ) ;
            }
            b = new byte[ 1 ] ;
            shm.read( false,b,p,0,1 ) ;
            int state = ( ( int )b[ 0 ] & 0x000000ff ) ;
            if( outState != null ) {
                outState[ 0 ] = state ;
            }
            b = null ;
            if( state != ShmDefine.STATE_RES && state != ShmDefine.STATE_NEXT ) {
                return false ;
            }
            b = new byte[ 2 ] ;
            if( overMode == true ) {
                b[ 0 ] = ( byte )( ( bodyLen + 1 ) & 0x000000ff ) ;
                b[ 1 ] = ( byte )( ( ( bodyLen + 1 ) & 0x0000ff00 ) >> 8 ) ;
            }
            else {
                b[ 0 ] = ( byte )( bodyLen & 0x000000ff ) ;
                b[ 1 ] = ( byte )( ( bodyLen & 0x0000ff00 ) >> 8 ) ;
            }
            shm.write( false,b,p+5,0,2 ) ;
            b = null ;
            shm.write( false,binary,p+ShmDefine.ONE_HEADER_LENGTH,0,bodyLen ) ;
            b = new byte[ 1 ] ;
            b[ 0 ] = ( byte )( ShmDefine.STATE_SET & 0x000000ff ) ;
            shm.write( false,b,p,0,1 ) ;
            return true ;
        } finally {
            shm.unLock() ;
        }
    }
    
    /**
     * 指定位置を空き領域に変更.
     * <BR>
     * @param shm SharedMemoryを設定します.
     * @param sequenceId 対象のシーケンスIDを設定します.
     * @param no 空き領域位置を設定します.
     * @param length データ個数を設定します.
     * @exception Exception 例外.
     */
    public static final void freeSpace( CoreShm shm,int sequenceId,int no,int length )
        throws Exception {
        try {
            shm.lock() ;
            int p = startPoint( false,shm,no,length ) ;
            if( p <= -1 ) {
                return ;
            }
            byte[] b = new byte[ 4 ] ;
            shm.read( false,b,p+1,0,4 ) ;
            int seqId = ( ( int )b[ 0 ] & 0x000000ff ) |
                ( ( ( int )b[ 1 ] & 0x000000ff ) << 8 ) |
                ( ( ( int )b[ 2 ] & 0x000000ff ) << 16 ) |
                ( ( ( int )b[ 3 ] & 0x000000ff ) << 24 ) ;
            b = null ;
            if( seqId <= -1 || seqId != sequenceId ) {
                return ;
            }
            int len = trimLength( length ) ;
            int n = trimLength( no+1 ) - 1 ;
            b = new byte[ len ] ;
            shm.read( false,b,ShmDefine.SEQUENCE_LENGTH+ShmDefine.USE_CONNECT_NO,0,len ) ;
            b[ n ] = ( byte )( ( ( int )b[ n ] & ( ~( 1 << ( no - ( n * 8 ) ) ) ) ) & 0x000000ff ) ;
            shm.write( false,b,ShmDefine.SEQUENCE_LENGTH+ShmDefine.USE_CONNECT_NO,0,len ) ;
            b = null ;
            b = new byte[ 7 ] ;
            b[ 0 ] = ( byte )( ShmDefine.STATE_FREE & 0x000000ff ) ;
            b[ 1 ] = ( byte )( ShmDefine.NOT_SEQUENCE & 0x000000ff ) ;
            b[ 2 ] = ( byte )( ( ShmDefine.NOT_SEQUENCE & 0x0000ff00 ) >> 8 ) ;
            b[ 3 ] = ( byte )( ( ShmDefine.NOT_SEQUENCE & 0x00ff0000 ) >> 16 ) ;
            b[ 4 ] = ( byte )( ( ShmDefine.NOT_SEQUENCE & 0xff000000 ) >> 24 ) ;
            b[ 5 ] = ( byte )0 ;
            b[ 6 ] = ( byte )0 ;
            shm.write( false,b,p,0,7 ) ;
            p = startPoint( true,shm,no,length ) ;
            shm.write( false,b,p,0,7 ) ;
        } finally {
            shm.unLock() ;
        }
    }
    
    /**
     * 空き位置を予約.
     * <BR>
     * @param shm SharedMemoryを設定します.
     * @param length データ個数を設定します.
     * @return int 予約された内容が返されます.<BR>
     *             int[ 0 ]は、シーケンスIDです.<BR>
     *             int[ 1 ]は、開始ポジションです.<BR>
     *             [null]が返された場合は、空き情報はありません.
     * @exception Exception 例外.
     */
    public static final int[] reservationSpace( CoreShm shm,int length )
        throws Exception {
        try {
            shm.lock() ;
            int len = trimLength( length ) ;
            byte[] b = new byte[ len ] ;
            shm.read( false,b,ShmDefine.SEQUENCE_LENGTH+ShmDefine.USE_CONNECT_NO,0,len ) ;
            int pos = getFreeSpacePos( b,len,length ) ;
            if( pos == -1 ) {
                return null ;
            }
            shm.write( false,b,ShmDefine.SEQUENCE_LENGTH+ShmDefine.USE_CONNECT_NO,0,len ) ;
            b = null ;
            int seqId = updateSequenceId( false,shm ) ;
            b = new byte[ 5 ] ;
            b[ 0 ] = ( byte )( ShmDefine.STATE_RES & 0x000000ff ) ;
            b[ 1 ] = ( byte )( seqId & 0x000000ff ) ;
            b[ 2 ] = ( byte )( ( seqId & 0x0000ff00 ) >> 8 ) ;
            b[ 3 ] = ( byte )( ( seqId & 0x00ff0000 ) >> 16 ) ;
            b[ 4 ] = ( byte )( ( seqId & 0xff000000 ) >> 24 ) ;
            int p = startPoint( false,shm,pos,length ) ;
            shm.write( false,b,p,0,5 ) ;
            p = startPoint( true,shm,pos,length ) ;
            shm.write( false,b,p,0,5 ) ;
            // 予約位置を空間にセット.
            useAcceptPos( shm,pos ) ;
            int[] ret = new int[ 2 ] ;
            ret[ 0 ] = seqId ;
            ret[ 1 ] = pos ;
            return ret ;
        } finally {
            shm.unLock() ;
        }
    }
    
    /**
     * 空き管理情報を取得.
     * <BR>
     * @param mode [true]の場合、書き込み情報位置です.
     * @param out 受け取りバイナリを設定します.
     * @param shm SharedMemoryを設定します.
     * @exception Exception 例外.
     */
    public static final void readFreeSpace( boolean sync,byte[] out,CoreShm shm )
        throws Exception {
        try {
            if( sync == true ) {
                shm.lock() ;
            }
            shm.read( false,out,ShmDefine.SEQUENCE_LENGTH+ShmDefine.USE_CONNECT_NO,0,out.length ) ;
        } finally {
            if( sync == true ) {
                shm.unLock() ;
            }
        }
    }
    
    /**
     * 空き管理情報から、空き位置を取得.
     */
    private static final int getFreeSpacePos( byte[] b,int trimLen,int length ) {
        int ln = trimLen - 1 ;
        int ret = -1 ;
        for( int i = 0 ; i < ln ; i ++ ) {
            int cd = ( ( int )b[ i ] & 0x000000ff ) ;
            if( cd != 0x000000ff ) {
                int lenJ = 8 ;
                int sft = 1 ;
                for( int j = 0 ; j < lenJ ; j ++ ) {
                    if( ( cd & ( sft << j ) ) == 0 ) {
                        b[ i ] = ( byte )( ( ( cd ) | ( sft << j ) ) & 0x000000ff ) ;
                        ret = ( i * 8 ) + j ;
                        break ;
                    }
                }
                if( ret != -1 ) {
                    break ;
                }
            }
        }
        if( ret == -1 ) {
            int lenJ = 8 ;
            int sft = 1 ;
            int cd = b[ ln ] ;
            for( int j = 0 ; j < lenJ ; j ++ ) {
                if( ( cd & ( sft << j ) ) == 0 ) {
                    b[ ln ] = ( byte )( ( ( cd ) | ( sft << j ) ) & 0x000000ff ) ;
                    ret = ( ln * 8 ) + j ;
                    break ;
                }
            }
        }
        return ret ;
    }
    
    /**
     * 予約位置条件を付加.
     */
    private static final void useAcceptPos( CoreShm shm,int position )
        throws Exception {
        byte[] bin = new byte[ ShmDefine.USE_CONNECT_NO ] ;
        for( ;; ) {
            boolean flg = false ;
            shm.read( false,bin,ShmDefine.SEQUENCE_LENGTH,0,bin.length ) ;
            for( int i = 0 ; i < ShmDefine.USE_CONNECT_SIZE ; i ++ ) {
                int p = i * 4 ;
                int x = ( bin[ p ] & 0x000000ff ) |
                    ( ( bin[ p+1 ] & 0x000000ff ) << 8 ) |
                    ( ( bin[ p+2 ] & 0x000000ff ) << 16 ) |
                    ( ( bin[ p+3 ] & 0x000000ff ) << 24 ) ;
                if( x <= 0 ) {
                    byte[] b = new byte[ 4 ] ;
                    position ++ ;
                    b[ 0 ] = ( byte )( position & 0x000000ff ) ;
                    b[ 1 ] = ( byte )( ( position & 0x0000ff00 ) >> 8 ) ;
                    b[ 2 ] = ( byte )( ( position & 0x00ff0000 ) >> 16 ) ;
                    b[ 3 ] = ( byte )( ( position & 0xff000000 ) >> 24 ) ;
                    shm.write( false,b,ShmDefine.SEQUENCE_LENGTH+p,0,4 ) ;
                    flg = true ;
                    break ;
                }
            }
            if( flg == false ) {
                // 同期中の場合でも、強引に同期解除.
                shm.unLock() ;
                try {Thread.sleep( 60L ) ;}catch( Exception e ){}
                shm.lock() ;
            }
            else {
                break ;
            }
        }
    }
    
    /**
     * 予約位置を取得.
     * <BR>
     * @param out 予約位置を格納するオブジェクトを設定します.
     * @param shm SharedMemoryを設定します.
     * @exception Exception 例外.
     */
    public static final void getAcceptPos( ArrayList<Integer> out,CoreShm shm )
        throws Exception {
        boolean lockFlag = true ;
        try {
            shm.lock() ;
            byte[] bin = new byte[ ShmDefine.USE_CONNECT_NO ] ;
            shm.read( false,bin,ShmDefine.SEQUENCE_LENGTH,0,bin.length ) ;
            shm.write( false,new byte[ ShmDefine.USE_CONNECT_NO ],ShmDefine.SEQUENCE_LENGTH,0,ShmDefine.USE_CONNECT_NO ) ;
            shm.unLock() ;
            lockFlag = false ;
            for( int i = 0 ; i < ShmDefine.USE_CONNECT_SIZE ; i ++ ) {
                int p = i * 4 ;
                int x = ( bin[ p ] & 0x000000ff ) |
                    ( ( bin[ p+1 ] & 0x000000ff ) << 8 ) |
                    ( ( bin[ p+2 ] & 0x000000ff ) << 16 ) |
                    ( ( bin[ p+3 ] & 0x000000ff ) << 24 ) ;
                if( x > 0 ) {
                    out.add( new Integer( x-1 ) ) ;
                }
            }
        } finally {
            if( lockFlag == true ) {
                shm.unLock() ;
            }
        }
    }
}
