package org.maachang.shm;

import java.io.IOException;
import java.io.OutputStream;

import org.maachang.shm.core.CoreShm;

/**
 * 共有メモリオブジェクト.
 * <BR>
 * 共有メモリ全体を扱う場合は、このオブジェクトを利用します.
 *
 * @version 2008/02/16
 * @author  masahito suzuki
 * @since  ShareMemory 1.00
 */
public class SharedMemory {
    /**
     * 共有メモリーオブジェクト.
     */
    private CoreShm mem = null ;
    
    /**
     * コンストラクタ.
     */
    private SharedMemory() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR>
     * @param mem 対象のコア共有メモリオブジェクトを設定します.
     * @exception Exception 例外.
     */
    public SharedMemory( CoreShm mem ) throws Exception {
        if( mem == null || mem.isUse() == false ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        this.mem = mem ;
    }
    
    /**
     * コンストラクタ.
     * <BR>
     * @param mode [true]の場合サーバモードで生成します.
     * @param semName 対象のセマフォー名を設定します.
     * @param shareName 対象の共有名を設定します.
     * @param length 対象の共有メモリー幅を設定します.
     * @exception Exception 例外.
     */
    public SharedMemory( boolean mode,String semName,String shareName,int length )
        throws Exception {
        try {
            this.mem = new CoreShm( mode,semName,shareName,length ) ;
        } catch( Exception e ) {
            this.destroy() ;
            throw e ;
        }
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     */
    public void destroy() {
        mem = null ;
    }
    
    /**
     * セマフォ名を取得.
     * <BR>
     * @return String セマフォ名が返されます.
     */
    public String getSemaphoreName() {
        if( isUse() == false ) {
            return null ;
        }
        return mem.getSemaphoreName() ;
    }
    
    /**
     * 共有メモリ名を取得.
     * <BR>
     * @return String 共有メモリ名が返されます.
     */
    public String getShareName() {
        if( isUse() == false ) {
            return null ;
        }
        return mem.getShareName() ;
    }
    
    /**
     * 共有メモリー最大幅を取得.
     * <BR>
     * @return int 共有メモリ最大幅を取得します.
     */
    public int getShareLength() {
        if( isUse() == false ) {
            return -1 ;
        }
        return mem.getShareLength() ;
    }
    
    /**
     * コア共有メモリオブジェクトを取得.
     * <BR>
     * @return CoreShm コア共有メモリオブジェクトが返されます.
     */
    public CoreShm getCore() {
        if( isUse() == false ) {
            return null ;
        }
        return mem ;
    }
    
    /**
     * 共有メモリに情報セット.
     * <BR>
     * @param binary 対象のバイナリを設定します.
     * @param pos 対象のポジションを設定します.
     * @param off 対象のオフセット値を設定します.
     * @param length 対象のデータ長を設定します.
     * @return int 書き込まれたサイズが返されます.
     * @exception Exception 例外.
     */
    public int write( byte[] binary,int pos,int off,int length )
        throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        return mem.write( binary,pos,off,length ) ;
    }
    
    /**
     * 共有メモリから情報取得.
     * <BR>
     * @param binary 対象のバイナリを設定します.
     * @param pos 対象のポジションを設定します.
     * @param off 対象のオフセット値を設定します.
     * @param length 対象のデータ長を設定します.
     * @return int 読み込まれたサイズが返されます.
     * @exception Exception 例外.
     */
    public int read( byte[] binary,int pos,int off,int length )
        throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        return mem.read( binary,pos,off,length ) ;
    }
    
    /**
     * オブジェクト有効チェック.
     * <BR>
     * @return boolean [true]の場合、有効です.
     */
    public boolean isUse() {
        return ( mem != null && mem.isUse() ) ;
    }
    
    /**
     * デバッグ出力.
     * <BR>
     * @param name 出力先ファイル名を設定します.
     * @exception Exception 例外.
     */
    public void debugOutFile( String name ) throws Exception {
        mem.outputFile( name ) ;
    }
    
    /**
     * デバッグ出力.
     * <BR>
     * @param output 出力先オブジェクトを設定します.
     * @exception Exception 例外.
     */
    public void debugOutputStream( OutputStream output ) throws Exception {
        mem.outputFile( output ) ;
    }
}

