package org.maachang.shm;

import java.io.IOException;
import java.util.ArrayList;

import org.maachang.shm.core.CoreShm;

/**
 * SharedMemory用サーバ処理.
 * <BR>
 * 通常のSharedMemoryでは、１つのメモリー空間として提供された形のものとして利用しますが、この場合、
 * 複数のクライアントと、１つのサーバでの通信の場合は、接続数単位で、共有メモリを作るか、１つの
 * 共有メモリーを自前で管理し、複数対１の通信を実現する必要があります。<BR>
 * また、複数対１のアクセスを行いたい場合、使い方のイメージとしては、java.net.ServerSocketのような
 * 感じが、もっとも単純で、利用しやすい形であると思います。<BR>
 * このオブジェクトは、java.net.ServerSocketのようにacceptで接続を待ち、そこから取得された通信用オブジェクト
 * を使って、相互に、通信を行うことができます。<BR>
 * <div style="border:1px solid gray; padding:5px;">
 * <pre>
 * ＜例:サーバ側処理＞
 * public static final void main( String[] args ) throws Exception {
 *   ArrayList<ConnThread> threadMan = new ArrayList<ConnThread>() ;
 *   ServerShm server = new ServerShm( "test","test",10,-1 ) ;
 *   for( ;; ) {
 *     ShmConnector c = server.accept() ;
 *     if( c != null ) {
 *       threadMan.add( new ConnThread( c ) ) ;
 *     }
 *   }
 * }
 * class ConnThread extends Thread {
 *   private ShmConnector con = null ;
 *   public ConnThread( ShmConnector con ) {
 *     this.con = con ;
 *     this.setDaemon( true ) ;
 *     this.start() ;
 *   }
 *   public void run() {
 *     for( ;; ) {
 *       if( con == null || con.isConnect() == false ) {
 *         break ;
 *       }
 *       try {
 *          Thread.sleep( 30L ) ;
 *          int len = in.available() ;
 *          if( len > 0 ) {
 *            ByteArrayOutputStream bo = new ByteArrayOutputStream() ;
 *            for( ;; ) {
 *              if( in.available() <= 0 ) {
 *                 break ;
 *              }
 *              int n = in.read() ;
 *              if( n <= -1 ) {
 *                break ;
 *              }
 *              bo.write( n ) ;
 *            }
 *            byte[] bin = bo.toByteArray() ;
 *            bo.close() ;
 *            System.out.println( "read("+con.getId()+"):len:" + bin.length + "=" + new String( bin,"UTF8" ) ) ;
 *            out.write( "server:".getBytes( "UTF8" ) ) ;
 *            out.write( bin ) ;
 *            out.flush() ;
 *          }
 *        } catch( Exception e ) {
 *          e.printStackTrace() ;
 *          try {
 *            con.close() ;
 *          } catch( Exception ee ) {
 *          }
 *          con = null ;
 *        }
 *      }
 *    }
 *  }
 * </pre></div>
 * <BR>
 * <div style="border:1px solid gray; padding:5px;">
 * <pre>
 * ＜例：クライアント側処理＞
 *  public static final void main( String[] args ) throws Exception {
 *    BufferedReader be = new BufferedReader( new InputStreamReader( System.in ) ) ;
 *    ClientShm shm = new ClientShm( "test","test",10,-1 ) ;
 *    ShmConnector conn = shm.connect() ;
 *    for( ;; ) {
 *      OutputStream output = conn.getOutputStream() ;
 *      InputStream input = conn.getInputStream() ;
 *      System.out.print( "send:" ) ;
 *      String in = be.readLine() ;
 *      if( in == null || in.length() <= 0 ) {
 *        continue ;
 *      }
 *      output.write( in.getBytes( "UTF8" ) ) ;
 *      output.flush() ;
 *      for( ;; ) {
 *        int len = input.available() ;
 *        if( len > 0 ) {
 *          ByteArrayOutputStream bo = new ByteArrayOutputStream() ;
 *          for( ;; ) {
 *            if( input.available() <= 0 ) {
 *              break ;
 *            }
 *            int n = input.read() ;
 *            if( n <= -1 ) {
 *              break ;
 *            }
 *            bo.write( n ) ;
 *          }
 *          byte[] bin = bo.toByteArray() ;
 *          System.out.println( "read("+conn.getId()+"):" + new String( bin,"UTF8" ) ) ;
 *          bo.close() ;
 *          break ;
 *        }
 *        Thread.sleep( 30L ) ;
 *      }
 *    }
 *  }
 * </pre></div>
 * <BR>
 * このような感じで利用ができます。<BR>
 * また、通常の共有メモリとして利用する場合は、org.maachang.shm.SharedMemoryを利用します。
 *
 * @version 2008/02/18
 * @author  masahito suzuki
 * @since  SharedMemory 1.02
 * @see org.maachang.shm.ClientShm
 * @see org.maachang.shm.SharedMemory
 */
public class ServerShm {
    
    /**
     * デフォルトタイムアウト値.
     */
    private static final int DEF_TIMEOUT = 30000 ;
    
    /**
     * CoreSharedMemory.
     */
    private CoreShm shm = null ;
    
    /**
     * Accept管理.
     */
    private ShmServerAcceptThread thread = null ;
    
    /**
     * タイムアウトコネクション管理.
     */
    private ShmTimeoutConnectThread timeoutThread = null ;
    
    /**
     * 接続管理.
     */
    private ArrayList<ShmConnectorImpl> conns = null ;
    
    /**
     * タイムアウト値.
     */
    private int timeout = -1 ;
    
    /**
     * コネクション数.
     */
    private int length = -1 ;
    
    /**
     * 同期オブジェクト.
     */
    private final Object sync = new Object() ;
    
    /**
     * コンストラクタ.
     */
    private ServerShm() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR>
     * @param semName 対象のセマフォー名を設定します.
     * @param shareName 対象の共有名を設定します.
     * @param length 最大接続数を設定します.
     * @param timeout タイムアウト値を設定します.
     * @exception Exception 例外.
     */
    public ServerShm( String semName,String shareName,int length,int timeout )
        throws Exception {
        if( timeout <= 0 ) {
            timeout = DEF_TIMEOUT ;
        }
        this.shm = new CoreShm( true,semName,shareName,ShmCommon.maxIOLength( length ) ) ;
        this.length = ShmCommon.trimLength( length ) * 8 ;
        this.thread = new ShmServerAcceptThread( shm,sync,this.length,timeout ) ;
        this.timeout = timeout ;
        this.conns = new ArrayList<ShmConnectorImpl>() ;
        this.timeoutThread = new ShmTimeoutConnectThread( this.conns,sync ) ;
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        this.close() ;
    }
    
    /**
     * オブジェクトクローズ.
     */
    public void close() {
        synchronized( sync ) {
            if( thread != null ) {
                thread.destroy() ;
            }
            if( timeoutThread != null ) {
                timeoutThread.destroy() ;
            }
            if( shm != null ) {
                shm.destroy() ;
            }
            if( conns != null ) {
                conns.clear() ;
            }
            thread = null ;
            shm = null ;
            conns = null ;
        }
    }
    
    /**
     * 新しい接続を取得.
     * <BR>
     * @return ShmConnector 接続コネクターが返されます.
     * @exception Exception 例外.
     */
    public ShmConnector accept() throws Exception {
        return accept( -1 ) ;
    }
    
    /**
     * 新しい接続を取得.
     * <BR>
     * @param timeout タイムアウト値を設定します.
     * @return ShmConnector 接続コネクターが返されます.
     * @exception Exception 例外.
     */
    public ShmConnector accept( int timeout ) throws Exception {
        if( isUse() == false ) {
            throw new IOException( "オブジェクトは既に破棄されています" ) ;
        }
        long tm = System.currentTimeMillis() ;
        ShmConnectorImpl ret = null ;
        for( ;; ) {
            ret = thread.getQueue() ;
            if( ret != null ) {
                synchronized( sync ) {
                    conns.add( ret ) ;
                }
                return ret ;
            }
            if( timeout > 0 && tm + ( long )timeout <= System.currentTimeMillis() ) {
                return null ;
            }
            Thread.sleep( 5 ) ;
        }
    }
    
    /**
     * 最大コネクション数を取得.
     * <BR>
     * @return int 最大コネクション数が返されます.
     */
    public int maxLength() {
        synchronized( sync ) {
            if( isUse() == false ) {
                return -1 ;
            }
            return shm.getShareLength() ;
        }
    }
    
    /**
     * タイムアウト値を取得.
     * <BR>
     * @return int タイムアウト値が返されます.
     */
    public int getTimeout() {
        synchronized( sync ) {
            if( isUse() == false ) {
                return -1 ;
            }
            return timeout ;
        }
    }
    
    /**
     * セマフォ名を取得.
     * <BR>
     * @return String セマフォ名が返されます.
     */
    public String getSemaphoreName() {
        synchronized( sync ) {
            if( isUse() == false ) {
                return null ;
            }
            return shm.getSemaphoreName() ;
        }
    }
    
    /**
     * 共有メモリ名を取得.
     * <BR>
     * @return String 共有メモリ名が返されます.
     */
    public String getShareName() {
        synchronized( sync ) {
            if( isUse() == false ) {
                return null ;
            }
            return shm.getShareName() ;
        }
    }
    
    /**
     * 現在の接続数を取得します.
     * <BR>
     * @return int 接続数が返されます.
     */
    public int size() {
        synchronized( sync ) {
            if( isUse() == false ) {
                return -1 ;
            }
            return conns.size() ;
        }
    }
    
    /**
     * オブジェクトが利用可能かチェック.
     * <BR>
     * @return boolean [true]の場合、利用可能です.
     */
    public boolean isUse() {
        synchronized( sync ) {
            return ( shm != null && shm.isUse() &&
                thread != null && thread.isStop() == false &&
                conns != null) ;
        }
    }
}

class ShmServerAcceptThread extends Thread {
    
    private volatile boolean stopFlag = true ;
    private int useSeqId = 0 ;
    private CoreShm shm = null ;
    private int length = -1 ;
    private int timeout = -1 ;
    private ArrayList<ShmConnectorImpl> queue = null ;
    private ArrayList<Integer> accepts = null ;
    private Object sync = null ;
    
    public ShmServerAcceptThread( CoreShm shm,Object sync,int length,int timeout ) throws Exception {
        this.useSeqId = ShmCommon.nowSequenceId( shm ) ;
        this.shm = shm ;
        this.length = length ;
        this.timeout = timeout ;
        this.queue = new ArrayList<ShmConnectorImpl>() ;
        this.accepts = new ArrayList<Integer>() ;
        this.stopFlag = false ;
        this.sync = sync ;
        this.setDaemon( true ) ;
        this.start() ;
    }
    
    public void destroy() {
        setStop( true ) ;
    }
    
    public synchronized boolean isStop() {
        return stopFlag ;
    }
    
    private synchronized void setStop( boolean mode ) {
        this.stopFlag = mode ;
    }
    
    protected ShmConnectorImpl getQueue() {
        ShmConnectorImpl ret = null ;
        synchronized( sync ) {
            if( queue.size() > 0 ) {
                ret = queue.remove( 0 ) ;
            }
        }
        return ret ;
    }
    
    private static final long WAIT = 60L ;
    
    public void run() {
        boolean endFlag = false ;
        ThreadDeath threadDeach = null ;
        for( ;; ) {
            if( endFlag == true || isStop() == true ) {
                break ;
            }
            try {
                Thread.sleep( WAIT ) ;
                int seq = ShmCommon.nowSequenceId( shm ) ;
                if( seq != useSeqId ) {
                    useConnect( seq ) ;
                }
            } catch( OutOfMemoryError mem ) {
            } catch( Exception e ) {
            } catch( ThreadDeath td ) {
                endFlag = true ;
                threadDeach = td ;
            }
        }
        if( queue != null ) {
            queue.clear() ;
        }
        this.shm = null ;
        this.queue = null ;
        setStop( true ) ;
        if( threadDeach != null ) {
            throw threadDeach ;
        }
    }
    
    private void useConnect( int seq ) throws Exception {
        ShmCommon.getAcceptPos( accepts,shm ) ;
        int len = accepts.size() ;
        for( int i = 0 ; i < len ; i ++ ) {
            int no = accepts.remove( 0 ).intValue() ;
            synchronized( sync ) {
                ShmConnectorImpl impl = new ShmConnectorImpl( shm,no,length,timeout ) ;
                queue.add( impl ) ;
            }
        }
        useSeqId = seq ;
    }
}

class ShmTimeoutConnectThread extends Thread {
    private volatile boolean stopFlag = true ;
    private ArrayList<ShmConnectorImpl> conns = null ;
    private Object sync = null ;
    
    public ShmTimeoutConnectThread( ArrayList<ShmConnectorImpl> conns,Object sync ) throws Exception {
        this.conns = conns ;
        this.stopFlag = false ;
        this.sync = sync ;
        this.setDaemon( true ) ;
        this.start() ;
    }
    
    public void destroy() {
        setStop( true ) ;
    }
    
    public synchronized boolean isStop() {
        return stopFlag ;
    }
    
    private synchronized void setStop( boolean mode ) {
        this.stopFlag = mode ;
    }
    
    private static final long WAIT = 30L ;
    
    public void run() {
        boolean endFlag = false ;
        ThreadDeath threadDeach = null ;
        for( ;; ) {
            if( endFlag == true || isStop() == true ) {
                break ;
            }
            try {
                int len = -1 ;
                synchronized( sync ) {
                    len = conns.size() ;
                }
                if( len > 0 ) {
                    for( int i = len-1 ; i >= 0 ; i -- ) {
                        Thread.sleep( WAIT ) ;
                        ShmConnectorImpl impl = null ;
                        synchronized( sync ) {
                            impl = conns.get( i ) ;
                            if( impl != null && impl.isTimeout() ) {
                                impl.close() ;
                                conns.remove( i ) ;
                            }
                            else if( impl.isConnect() == false ) {
                                conns.remove( i ) ;
                            }
                        }
                    }
                }
                else {
                    Thread.sleep( WAIT ) ;
                }
            } catch( OutOfMemoryError mem ) {
            } catch( Exception e ) {
            } catch( ThreadDeath td ) {
                endFlag = true ;
                threadDeach = td ;
            }
        }
        conns = null ;
        setStop( true ) ;
        if( threadDeach != null ) {
            throw threadDeach ;
        }
    }
}
