package org.maachang.shm;

import org.maachang.shm.core.CoreShm;

/**
 * SharedMemory用クライアント処理.
 *  
 * @version 2008/02/18
 * @author  masahito suzuki
 * @since  SharedMemory 1.02
 */
public class ClientShm {
    
    /**
     * デフォルトタイムアウト値.
     */
    private static final int DEF_TIMEOUT = 30000 ;
    
    /**
     * CoreSharedMemory.
     */
    private CoreShm shm = null ;
    
    /**
     * タイムアウト値.
     */
    private int timeout = -1 ;
    
    /**
     * コネクション数.
     */
    private int length = -1 ;
    
    /**
     * 同期オブジェクト.
     */
    private final Object sync = new Object() ;
    
    /**
     * コンストラクタ.
     */
    private ClientShm() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR>
     * @param semName 対象のセマフォー名を設定します.
     * @param shareName 対象の共有名を設定します.
     * @param length 最大接続数を設定します.
     * @param timeout タイムアウト値を設定します.
     * @exception Exception 例外.
     */
    public ClientShm( String semName,String shareName,int length,int timeout )
        throws Exception {
        this.open( semName,shareName,length,timeout ) ;
    }
    
    /**
     * デストラクタ.
     */
    protected void finalize() throws Exception {
        this.close() ;
    }
    
    /**
     * オープン処理.
     * <BR>
     * @param semName 対象のセマフォー名を設定します.
     * @param shareName 対象の共有名を設定します.
     * @param length 最大接続数を設定します.
     * @param timeout タイムアウト値を設定します.
     * @exception Exception 例外.
     */
    public void open( String semName,String shareName,int length,int timeout )
        throws Exception {
        if( timeout <= 0 ) {
            timeout = DEF_TIMEOUT ;
        }
        this.shm = new CoreShm( false,semName,shareName,ShmCommon.maxIOLength( length ) ) ;
        this.length = ShmCommon.trimLength( length ) * 8 ;
        this.timeout = timeout ;
    }
    
    /**
     * オブジェクトクローズ.
     */
    public void close() {
        if( shm != null ) {
            shm.destroy() ;
        }
        shm = null ;
    }
    
    /**
     * 新しい接続を取得.
     * <BR>
     * @return ShmConnector 接続コネクターが返されます.
     * @exception Exception 例外.
     */
    public ShmConnector connect() throws Exception {
        return new ShmConnectorImpl( shm,length,timeout ) ;
    }
    
    /**
     * 最大コネクション数を取得.
     * <BR>
     * @return int 最大コネクション数が返されます.
     */
    public int maxLength() {
        synchronized( sync ) {
            if( isUse() == false ) {
                return -1 ;
            }
            return length ;
        }
    }
    
    /**
     * タイムアウト値を取得.
     * <BR>
     * @return int タイムアウト値が返されます.
     */
    public int getTimeout() {
        synchronized( sync ) {
            if( isUse() == false ) {
                return -1 ;
            }
            return timeout ;
        }
    }
    
    /**
     * セマフォ名を取得.
     * <BR>
     * @return String セマフォ名が返されます.
     */
    public String getSemaphoreName() {
        synchronized( sync ) {
            if( isUse() == false ) {
                return null ;
            }
            return shm.getSemaphoreName() ;
        }
    }
    
    /**
     * 共有メモリ名を取得.
     * <BR>
     * @return String 共有メモリ名が返されます.
     */
    public String getShareName() {
        synchronized( sync ) {
            if( isUse() == false ) {
                return null ;
            }
            return shm.getShareName() ;
        }
    }
    
    /**
     * オブジェクトが利用可能かチェック.
     * <BR>
     * @return boolean [true]の場合、利用可能です.
     */
    public boolean isUse() {
        synchronized( sync ) {
            return ( shm != null && shm.isUse() ) ;
        }
    }
}
