//
// Linux用のSharedMemoyには、IPCを利用する。
//
#include <unistd.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/shm.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "../../include/org_maachang_shm_core_NativeSharedMemory.h"

///////////////////////////////////////////////////////////////////////////////
// ShareMemory内部処理.
///////////////////////////////////////////////////////////////////////////////

#define MODE_BY_SERVER			(0)
#define MODE_NOT_SYNC			(0)


static int semId ;
static int shmId ;
static int shmMode ;
static int shareSize ;
static char* shareData ;

union semun {
  int val;					/* SETVAL の値 */
  struct semid_ds *buf; 	/* IPC_STAT, IPC_SET 用のバッファ */
  unsigned short *array;	/* GETALL, SETALL 用の配列 */
  /* Linux 特有の部分: */
  struct seminfo *__buf;	/* IPC_INFO 用のバッファ */
};

// 文字列から、数値を求める[fnv-hash32a].
key_t ConvertCharArrayByInt( char* b ) {
	key_t c = 0x811C9DC5 ;
	int len = strlen( b ) ;
	for( int i = 0 ; i < len ; i ++ ) {
		c ^= b[i];
		c += (c << 1L) + (c << 4L) + (c << 7L) + (c << 8L) + (c << 24L);
	}
	return c ;
}

// セマフォ初期化.
int InitSemaphoreHandle(int mode,char* name) {
	key_t key ;
	//if ((key = ftok(name, 'R')) == -1) {
	//	return -1 ;
	//}
	key = ConvertCharArrayByInt( name ) ;
	int code ;
	if( mode == MODE_BY_SERVER ) {
		code = IPC_CREAT | 0666 ;
	}
	else {
		code = 0666 ;
	}
	semId = semget(key, 1, code) ;
	if( semId <= -1 ) {
		return -1 ;
	}
	union semun semunion;
	if(semctl(semId,0,GETVAL,semunion)==0){
		semunion.val = 1;
		if(semctl(semId,0,SETVAL,semunion) == -1){
			return -1 ;
		}
	}
	return 0 ;
}

// セマフォ終了化.
void CloseSemaphoreHandle() {
	if( shmMode == MODE_BY_SERVER && semId > 0 ) {
		semctl(semId, 0, IPC_RMID, NULL) ;
		semId = -1 ;
	}
}

// セマフォロック.
int LockSharedMemory() {
	struct sembuf sb;
	sb.sem_num=0;
	sb.sem_op=-1;
	sb.sem_flg=0;
	if(semop(semId,&sb,1)== -1){
		return -1 ;
	}
	return 0 ;
}

// セマフォアンロック.
int UnLockSharedMemory() {
	struct sembuf sb;
	sb.sem_num=0;
	sb.sem_op=1;
	sb.sem_flg=0;
	if(semop(semId,&sb,1)== -1){
		return -1 ;
	}
	return 0 ;
}

// 共有メモリオープン.
int OpenShare(int mode,char* shareName,int size) {
	key_t key ;
	//if ((key = ftok(shareName, 'X')) == -1) {
	//	return -1 ;
	//}
	key = ConvertCharArrayByInt( shareName ) ;
	int code ;
	if( mode == MODE_BY_SERVER ) {
		code = IPC_CREAT | 0666 ;
	}
	else {
		code = 0666 ;
	}
	if ((shmId = shmget(key, size, code)) < 0) {
		return -1 ;
	}
	shareData = (char *)shmat(shmId, (void *)0, 0);
	if (shareData == (char *)-1) {
		shareData = NULL ;
		return -1 ;
	}
	if( mode == MODE_BY_SERVER ) {
		memset( shareData,'\0',size ) ;
	}
	return 0 ;
}

// 共有メモリクローズ.
void CloseShare() {
	if( shareData != NULL ) {
		shmdt(shareData) ;
		shareData = NULL ;
	}
	if( shmMode == MODE_BY_SERVER && shmId > 0 ) {
		shmctl(shmId, IPC_RMID, 0) ;
		shmId = -1 ;
	}
}

// 共有メモリ処理全般初期化処理.
int InitShareMemory( int mode,char* semName,char* shareName,int size ) {
	if( InitSemaphoreHandle( mode,semName ) <= -1 ) {
		return -1 ;
	}
	 ;
	if( LockSharedMemory() >= 0 ) {
		if( OpenShare( mode,shareName,size ) <= -1 ) {
			UnLockSharedMemory() ;
			return -1 ;
		}
		UnLockSharedMemory() ;
	}
	shmMode = mode ;
	shareSize = size ;
	return 0 ;
}

// 共有メモリ処理全般終了化処理.
int DestroyShareMemory() {
	if( LockSharedMemory() >= 0 ) {
		CloseShare() ;
		UnLockSharedMemory() ;
	}
	else {
		return -1 ;
	}
	CloseSemaphoreHandle() ;
	return 0 ;
}

// 共有メモリに情報追加.
int PutShareData( int md,unsigned char* data,int pos,int off,int len ) {
	if( md == MODE_NOT_SYNC ) {
		memcpy( shareData+pos,data+off,len ) ;
	}
	else {
		if( LockSharedMemory() >= 0 ) {
			memcpy( shareData+pos,data+off,len ) ;
			UnLockSharedMemory() ;
		}
		else {
			return -1 ;
		}
	}
	return 0 ;
}

// 共有メモリから情報取得.
int GetShareData( int md,unsigned char* data,int pos,int off,int len ) {
	if( md == MODE_NOT_SYNC ) {
		memcpy( data+off,shareData+pos,len ) ;
	}
	else {
		if( LockSharedMemory() >= 0 ) {
			memcpy( data+off,shareData+pos,len ) ;
			UnLockSharedMemory() ;
		}
		else {
			return -1 ;
		}
	}
	return 0 ;
}

// 共有メモリ全体長を取得.
int MaxShreSize() {
	return shareSize ;
}

///////////////////////////////////////////////////////////////////////////////
// JNI処理.
///////////////////////////////////////////////////////////////////////////////

// NativeShareMemory.initShareMemory呼び出し.
JNIEXPORT jint JNICALL Java_org_maachang_shm_core_NativeSharedMemory_initShareMemory
  (JNIEnv* env, jobject o, jint mode,jbyteArray n1, jbyteArray n2, jint sz ) {
	jboolean b ;
	jbyte* semName=(*env).GetByteArrayElements(n1,&b);
	jbyte* shareName=(*env).GetByteArrayElements(n2,&b);
	jint ret = InitShareMemory( mode,( char* )semName,( char* )shareName,sz ) ;
	(*env).ReleaseByteArrayElements(n1, semName, 0);
	(*env).ReleaseByteArrayElements(n2, shareName, 0);
	return ret ;
}

// NativeShareMemory.destroyShareMemory呼び出し.
JNIEXPORT jint JNICALL Java_org_maachang_shm_core_NativeSharedMemory_destroyShareMemory
  (JNIEnv* env, jobject o) {
	shareSize = -1 ;
	return DestroyShareMemory() ;
}

// NativeShareMemory.writeShareMemory呼び出し.
JNIEXPORT jint JNICALL Java_org_maachang_shm_core_NativeSharedMemory_writeShareMemory
  (JNIEnv* env, jobject o, jint md, jbyteArray in, jint ps, jint of, jint sz) {
	jboolean b ;
	jbyte* input=(*env).GetByteArrayElements(in,&b);
	jint ret = PutShareData( md,( unsigned char* )input,ps,of,sz ) ;
	(*env).ReleaseByteArrayElements(in, input, 0);
	return ret ;
}

// NativeShareMemory.readShareMemory呼び出し.
JNIEXPORT jint JNICALL Java_org_maachang_shm_core_NativeSharedMemory_readShareMemory
  (JNIEnv* env, jobject o, jint md, jbyteArray ot, jint ps, jint of, jint sz) {
	jboolean b ;
	jbyte* output=(*env).GetByteArrayElements(ot,&b);
	jint ret = GetShareData( md,( unsigned char* )output,ps,of,sz ) ;
	(*env).ReleaseByteArrayElements(ot, output, 0);
	return ret ;
}

// NativeShareMemory.sizeShareMemory呼び出し.
JNIEXPORT jint JNICALL Java_org_maachang_shm_core_NativeSharedMemory_sizeShareMemory
  (JNIEnv* env, jobject o) {
	return MaxShreSize() ;
}

// NativeShareMemory.startLock呼び出し.
JNIEXPORT jint JNICALL Java_org_maachang_shm_core_NativeSharedMemory_startLock
  (JNIEnv* env, jobject o) {
	return LockSharedMemory() ;
}

// NativeShareMemory.endLock呼び出し.
JNIEXPORT jint JNICALL Java_org_maachang_shm_core_NativeSharedMemory_endLock
  (JNIEnv* env, jobject o) {
	return UnLockSharedMemory() ;
}

