/**
 * JPicosheet: Spreadsheet engine for Java Applications
 * Copyright (C) 2011 yusuke nishikawa
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
/**
 *
 */
package jp.co.nissy.jpicosheet.core;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * セルの行列を表すオブジェクトです。<br>
 * 1つ1つのセルとして、セルの集合として、また行および列方向に順序を持った行列として扱うことができます。<br>
 *
 * テーブル名には末尾に<code>"#"</code>が付きます。テーブルの名前として<code>hoge</code>を使う場合、<code>"hoge#"</code>と指定します。<br>
 * テーブル名だけを指定した場合、それはテーブル全体を指定したことを意味します。たとえば大きさが3*3のテーブル<code>hoge#</code>を
 * 作成した場合、<code>hoge#</code>は9つのセルを含んでいます。<br>
 * テーブル内の特定のセルを指定する場合、"R1C1形式"のセルアドレス指定を行います。インデックスは0から始まるため、
 * <code>hoge#</code>の左上のセルを指定するには<code>hoge#R0C0</code>と指定します。この方法でテーブル内の単一のセルを指定した場合、
 * その扱いは通常のセルと同じです。
 *
 * @author yusuke nishikawa
 *
 */
public class Table implements CellGroupReference {

	/**
	 * テーブル名
	 */
	private String _name;

	/**
	 * このテーブルの行
	 */
	private List<List<Cell>> _rows;

	/**
	 * このテーブルを持っているSheetオブジェクトへの参照
	 */
	private Sheet _sheet;

	/**
	 * このテーブルのカラムサイズ
	 */
	private int _colSize;

	/**
	 * テーブル名のパターン文字列
	 */
	static final String TABLE_NAME_PATTERN = "[a-zA-Z_][a-zA-Z0-9_]*#";

	/**
	 * テーブル名の正規表現パターン
	 */
	private static Pattern _tableNamePattern = Pattern.compile(TABLE_NAME_PATTERN);

	/**
	 * 完全修飾テーブル名のパターン文字列
	 */
	static final String FULLY_QUALIFIED_TABLE_NAME_PATTERN = Sheet.SHEET_NAME_PATTERN + "!" + TABLE_NAME_PATTERN;

	/**
	 * 完全修飾テーブル名の正規表現パターン
	 */
	private static Pattern _fullyQualifiedTableNamePattern = Pattern.compile(FULLY_QUALIFIED_TABLE_NAME_PATTERN);

	/**
	 * アドレスのパターン文字列
	 */
	static final String ADDRESS_PATTERN = "(R[0-9]+C[0-9]+)";

	/**
	 * アドレスの正規表現パターン
	 */
	private static Pattern _AddressPattern = Pattern.compile(ADDRESS_PATTERN);

	/**
	 * アドレス付きテーブル名のパターン文字列
	 */
	static final String TABLE_NAME_WITH_ADDRESS_PATTERN = TABLE_NAME_PATTERN + ADDRESS_PATTERN;

	/**
	 * アドレス付きテーブル名の正規表現パターン
	 */
	private static Pattern _tableNameWithAddressPattern = Pattern.compile(TABLE_NAME_PATTERN + ADDRESS_PATTERN);

	/**
	 * 完全修飾アドレス付きテーブル名のパターン文字列
	 */
	static final String FULLY_QUALIFIED_TABLE_NAME_WITH_ADDRESS_PATTERN =Sheet.SHEET_NAME_PATTERN + "!" +  TABLE_NAME_PATTERN + ADDRESS_PATTERN;

	/**
	 * 完全修飾アドレス付きテーブル名の正規表現パターン
	 */
	private static Pattern _fullyQualifiedtableNameWithAddressPattern = Pattern.compile(FULLY_QUALIFIED_TABLE_NAME_WITH_ADDRESS_PATTERN);




	/**
	 * 範囲のパターン文字列
	 */
	static final String RANGE_PARSE_PATTERN =
		"^(?:" +
		"(?:(R[0-9]+)(C[0-9]+)(:)(R[0-9]+)(C[0-9]+))|(?:(R[0-9]+)(C[0-9]+)(:)(R[0-9]+)(Cx))|(?:(R[0-9]+)(C[0-9]+)(:)(Rx)(C[0-9]+))|(?:(R[0-9]+)(C[0-9]+)(:)(Rx)(Cx))" + "|" +
		"(?:(R[0-9]+)(Cx))|(?:(Rx)(C[0-9]+))" + "|" +
		")$";

	static final String RANGE_PATTERN_BASE =
		"R[0-9]+C[0-9]+:R[0-9]+C[0-9]+|R[0-9]+C[0-9]+:R[0-9]+Cx|R[0-9]+C[0-9]+:RxC[0-9]+|R[0-9]+C[0-9]+:RxCx" + "|" +
		"R[0-9]+Cx|RxC[0-9]+";

	static final String RANGE_PATTERN =
		"^(?:" +
		RANGE_PATTERN_BASE +
		")$";

	/**
	 * 範囲のパース用の正規表現パターン
	 */
	private static Pattern _RangeParsePattern = Pattern.compile(RANGE_PARSE_PATTERN);

	/**
	 * 範囲の正規表現パターン
	 */
	private static Pattern _RangePattern = Pattern.compile(RANGE_PATTERN);

	/**
	 * 範囲付きテーブル名のパターン文字列
	 */
	static final String TABLE_NAME_WITH_RANGE_PATTERN =
		TABLE_NAME_PATTERN + "(?:" + RANGE_PATTERN_BASE + ")";

	/**
	 * 範囲付きテーブル名の正規表現パターン
	 */
	private static Pattern _tableNameWithRangePattern = Pattern.compile(TABLE_NAME_WITH_RANGE_PATTERN);

	/**
	 * 完全修飾範囲付きテーブル名のパターン文字列
	 */
	static final String FULLY_QUALIFIED_TABLE_NAME_WITH_RANGE_PATTERN =Sheet.SHEET_NAME_PATTERN + "!" +  TABLE_NAME_WITH_RANGE_PATTERN;

	/**
	 * 完全修飾範囲付きテーブル名の正規表現パターン
	 */
	private static Pattern _fullyQualifiedtableNameWithRangePattern = Pattern.compile(FULLY_QUALIFIED_TABLE_NAME_WITH_RANGE_PATTERN);

	/**
	 * テーブル名とシートオブジェクトを指定してオブジェクトを作成します
	 * @param tableName テーブル名
	 * @param rowSize テーブルの行数
	 * @param colSize テーブルの列数
	 * @param sheet このテーブルを持っているSheetオブジェクトへの参照
	 */
	Table(String tableName, int rowSize, int colSize, Sheet sheet) throws IllegalArgumentException {

		validateTableName(tableName);

		// rowSize, colSizeは1以上でなければならない
		if (rowSize <= 0 || colSize <= 0) {
			throw new IllegalArgumentException("invalid row/col size. " +
					"row=" + Integer.toString(rowSize) +" col=" + Integer.toString(colSize));
		}

		_name = tableName;
		_colSize = colSize;
		_sheet = sheet;

		// 引数で指定されたサイズのテーブルを作成する。セルも同時に作成する
		_rows = new ArrayList<List<Cell>>(rowSize);
		for (int row = 0; row < rowSize; row++) {
			_rows.add(createRow(row));
		}
	}


	/**
	 * 渡された文字列がテーブル名として正しいかチェックします。<br>
	 * 正しくない場合、例外がスローされます。
	 * @param tableName チェックするテーブル名
	 * @throws IllegalArgumentException テーブル名として正しくない場合
	 */
	private void validateTableName(String tableName) throws IllegalArgumentException {

		if (!isValidTableName(tableName)) {
			throw new IllegalArgumentException("invalid table name \"" + tableName + "\"");
		}
	}

	/**
	 * 渡された文字列がテーブル名として正しいかチェックします。
	 * @param tableName チェックするテーブル名
	 * @return テーブル名として正しい場合true、そうでない場合false
	 */
	static boolean isValidTableName(String tableName) {

		return _tableNamePattern.matcher(tableName).matches();

	}

	/**
	 * 渡された文字列が完全修飾テーブル名として正しいかチェックします。
	 * @param cellName チェックするテーブル名
	 * @return テーブル名として正しい場合true、そうでない場合false
	 */
	static boolean isValidFullyQualifiedTableName(String fullyQualifiedTableName) {

		return _fullyQualifiedTableNamePattern.matcher(fullyQualifiedTableName).matches();

	}

	/** 渡された文字列がアドレス付きテーブル名として正しいかチェックします。<
	 * @param tableNameWithAddress チェックするアドレス付きテーブル名
	 * @return アドレス付きテーブル名として正しい場合true、そうでない場合false
	 */
	static boolean isValidTableNameWithAddress(String tableNameWithAddress) {

		return _tableNameWithAddressPattern.matcher(tableNameWithAddress).matches();

	}

	/** 渡された文字列が完全修飾アドレス付きテーブル名として正しいかチェックします。
	 * @param tableNameWithAddress チェックするアドレス付きテーブル名
	 * @return アドレス付きテーブル名として正しい場合true、そうでない場合false
	 */
	static boolean isValidFullyQualifiedTableNameWithAddress(String fullyQualifiedTableNameWithAddress) {

		return _fullyQualifiedtableNameWithAddressPattern.matcher(fullyQualifiedTableNameWithAddress).matches();

	}

	/**
	 * 渡された文字列が範囲名として正しいかチェックします
	 * @param rangeName チェックする範囲名
	 * @return 範囲名として正しい場合true、そうでない場合false
	 */
	static boolean isValidRangeName(String rangeName) {
		return _RangePattern.matcher(rangeName).matches();
	}

	/**
	 * 渡された文字列が範囲付きテーブル名として正しいかチェックします。
	 * @param tableNameWithRange チェックする範囲付きテーブル名
	 * @return 範囲付きテーブル名として正しい場合true、そうでない場合false
	 */
	static boolean isValidTableNameWithRange(String tableNameWithRange) {

		return _tableNameWithRangePattern.matcher(tableNameWithRange).matches();

	}

	/**
	 * 渡された文字列が完全修飾範囲付きテーブル名として正しいかチェックします。
	 * @param tableNameWithRange チェックする完全修飾範囲付きテーブル名
	 * @return 完全修飾範囲付きテーブル名として正しい場合true、そうでない場合false
	 */
	static boolean isValidFullyQualifiedTableNameWithRange(String fullyQualifiedTableNameWithRange) {

		return _fullyQualifiedtableNameWithRangePattern.matcher(fullyQualifiedTableNameWithRange).matches();
	}


	/**
	 * 渡された文字列が範囲名として正しいかチェックします。<br>
	 * 正しくない場合、例外がスローされます。
	 * @param rangeName チェックする範囲名
	 * @throws IllegalArgumentException テーブル名として正しくない場合
	 */
	private void validateRangeName(String rangeName) throws IllegalArgumentException {

		if (!isValidRangeName(rangeName)) {
			throw new IllegalArgumentException("invalid range name \"" + rangeName + "\"");
		}
	}

	/**
	 * このテーブルの名前を返します
	 * @return テーブルの名前
	 */
	public String getName() {
		return _name;
	}

	/**
	 * このテーブルの完全修飾名を返します
	 * @return このテーブルの完全修飾名
	 */
	public String getFullyQualifiedName() {
		return _sheet.getName() + "!" + _name;
	}

	/**
	 * 引数に渡されたrowNoの、カラムサイズcolSizeの行データを作成します
	 * @param rowNo 行番号
	 * @return 行データ
	 */
	private List<Cell> createRow(int rowNo) {

		List<Cell> rowData = new ArrayList<Cell>(_colSize);
		String rowNoString = Integer.toString(rowNo);

		for (int col = 0; col < _colSize; col++) {
			rowData.add(_sheet.addCell(_name + "R" + rowNoString + "C" + Integer.toString(col), true));
		}
		return rowData;
	}

	/**
	 * このテーブルの行方向の末尾に新しい行を1行追加します。
	 * @return このテーブル
	 */
	public Table addRow() {
		return addRow(1);
	}

	/**
	 * このテーブルの行方向の末尾に新しい行を指定した行数だけ追加します。
	 * @param addRowCount 追加する行数
	 * @return このテーブル
	 * @throws IllegalArgumentException 行数の指定が0未満の場合
	 */
	public Table addRow(int addRowCount) {

		if (addRowCount < 0) {
			throw new IllegalArgumentException("invalid addRowCount: " + Integer.toString(addRowCount));
		}

		for (int i = 0; i < addRowCount; i++) {
			_rows.add(createRow(_rows.size()));
		}
		return this;
	}

	/**
	 * このテーブルの列方向の末尾に新しい列を1列追加します。
	 * @return このテーブル
	 */
	public Table addColumn() {
		return addColumn(1);
	}

	/**
	 * このテーブルの列方向の末尾に新しい列を指定した列数だけ追加します。
	 * @param addColumnCount 追加する列数
	 * @return このテーブル
	 * @throws IllegalArgumentException 列数の指定が0未満の場合
	 */
	public Table addColumn(int addColumnCount) {

		if (addColumnCount < 0) {
			throw new IllegalArgumentException("invalid addColumnCount: " + Integer.toString(addColumnCount));
		}

		for (int i = 0; i < _rows.size(); i++) {
			for (int o = 0; o < addColumnCount; o++) {
				String newCellName = _name + "R" + Integer.toString(i) + "C" + Integer.toString(_rows.get(i).size());
				Cell cell = _sheet.addCell(newCellName, true);
				_rows.get(i).add(cell);
			}
		}
		return this;
	}

	/**
	 * このテーブルの指定した位置に1行挿入します。
	 * @param rowPos 挿入する位置
	 * @return このテーブル
	 */
	public Table insertRow(int rowPos) {

		return insertRow(rowPos, 1);
	}

	/**このテーブルの指定した位置に、指定した行数の行を挿入します。
	 * @param rowPos 挿入する位置
	 * @param insertRowCount 挿入する行数
	 * @return このテーブル
	 */
	public Table insertRow(int rowPos, int insertRowCount) {
		// rowPosがマイナスであってはならない
		if (rowPos < 0) {
			throw new IllegalArgumentException("ingalid rowPos: " + Integer.toString(rowPos));
		}
		// rowPosがテーブルの最大行以上であってはならない
		if (_rows.size() <= rowPos) {
			throw new IllegalArgumentException("ingalid rowPos: " + Integer.toString(rowPos) + " table rowsize: " + Integer.toString(_rows.size()));
		}
		// insertRowCountが1以下であってはならない
		if (insertRowCount < 1) {
			throw new IllegalArgumentException("ingalid insertRowCount: " + Integer.toString(insertRowCount));
		}

		// rowPosより後ろの行のセルの、セル名をリネームする 名前が重複しないよう行末からリネーム
		StringBuilder cellName = new StringBuilder();
		StringBuilder newCellName = new StringBuilder();
		for (int rowIndex = _rows.size() - 1; rowPos <= rowIndex ; rowIndex--) {
			for (int colIndex = _rows.get(0).size() - 1; 0 <= colIndex; colIndex--) {
				cellName.delete(0, cellName.length());
				newCellName.delete(0, newCellName.length());
				cellName.append(_name).append("R").append(rowIndex).append("C").append(colIndex);
				newCellName.append(_name).append("R").append(rowIndex + insertRowCount).append("C").append(colIndex);
				try {
					_sheet.renameCell(cellName.toString(), newCellName.toString());
				} catch (ReferenceNotFoundException e) {
					// これが発生するということは、この処理での名前の指定が悪いということ
					assert false: "Cell not found: " + cellName.toString();
				}
			}
		}

		// rowPosで指定された位置にinsertRowCount行だけ行を挿入する
		for (int rowIndex = rowPos; rowIndex < rowPos + insertRowCount; rowIndex++) {
			_rows.add(rowIndex, createRow(rowIndex));
		}

		return this;
	}


	/**このテーブルの指定した位置に1列挿入します。
	 * @param rowPos 挿入する位置
	 * @return このテーブル
	 */
	public Table insertColumn(int columnPos) {

		return insertColumn(columnPos, 1);
	}

	/**このテーブルの指定した位置に、指定した列数の列を挿入します。
	 * @param columnPos 挿入する位置
	 * @param insertColumnCount 挿入する列数
	 * @return このテーブル
	 */
	public Table insertColumn(int columnPos, int insertColumnCount) {
		// columnPosがマイナスであってはならない
		if (columnPos < 0) {
			throw new IllegalArgumentException("ingalid columnPos: " + Integer.toString(columnPos));
		}
		// columnPosがテーブルの最大列以上であってはならない
		if (_rows.get(0).size() <= columnPos) {
			throw new IllegalArgumentException("ingalid columnPos: " + Integer.toString(columnPos) + " table columnsize: " + Integer.toString(_rows.get(0).size()));
		}
		// insertColumnCountが1以下であってはならない
		if (insertColumnCount < 1) {
			throw new IllegalArgumentException("ingalid insertColumnCount: " + Integer.toString(insertColumnCount));
		}

		// すべての行に対し、指定位置に列を挿入する
		StringBuilder cellName = new StringBuilder();
		StringBuilder newCellName = new StringBuilder();
		// 行ぶんループ
		for (int rowIndex = 0; rowIndex < _rows.size(); rowIndex++) {
			// 挿入位置より後ろの列をリネーム。リネームを行うため、末尾から操作する
			for (int colIndex = _rows.get(rowIndex).size() - 1; columnPos <= colIndex; colIndex--) {
				cellName.delete(0, cellName.length());
				newCellName.delete(0, newCellName.length());
				cellName.append(_name).append("R").append(rowIndex).append("C").append(colIndex);
				newCellName.append(_name).append("R").append(rowIndex).append("C").append(colIndex + insertColumnCount);
				try {
					_sheet.renameCell(cellName.toString(), newCellName.toString());
				} catch (ReferenceNotFoundException e) {
					// これが発生するということは、この処理での名前の指定が悪いということ
					assert false: "Cell not found: " + cellName.toString();
				}
			}
			// 指定された列位置にセルを挿入する
			for (int i = 0; i < insertColumnCount; i++) {
				String addCellName = _name + "R" + Integer.toString(rowIndex) + "C" + Integer.toString(columnPos + i);
				Cell cell = _sheet.addCell(addCellName, true);
				_rows.get(rowIndex).add(columnPos + i, cell);
			}
		}

		return this;
	}


	/**
	 * このテーブルの行数を返します
	 * @return このテーブルの行数
	 */
	public int rowSize() {
		return _rows.size();
	}

	/**
	 * このテーブルの列数を返します
	 * @return このテーブルの列数
	 */
	public int colSize() {
		return _colSize;
	}

	/**
	 * このテーブル全体を現すRangeオブジェクトを返します。<br>
	 * これは、<code>getRange("R0C0:RxCx")</code>を指定したのと同じです。
	 * @return Rangeオブジェクト
	 */
	public Range getRange() {
		return getRange("R0C0:RxCx");
	}

	/**
	 * このテーブルの指定した範囲を表すRangeオブジェクトを返します。<br>
	 * テーブルの範囲指定として以下のような指定が可能です。<br>
	 * R0C0:RxCx このテーブルすべて
	 * R2Cx 3行目(行インデックス2)のすべての列
	 * RxC3 4列目(列インデックス3)のすべての行
	 * R2C0:R5Cx 3行目から6行目のすべての列
	 * R0C2:RxC5 3列目から6列目のすべての行
	 * R0C0:R0C0 1行1列から1行1列の1x1の範囲、つまりtableの左上のセル
	 * R3C2:R5C4 4行3列から6行4列の3x3の範囲
	 * R3C2:R5Cx 4行3列から5行最大列まで
	 * R3C2:RxC6 4行3列から最大行4列まで
	 * R3C2:RxCx 4行3列から最大行最大列まで
	 *
	 * @param range このテーブルの範囲を表す文字列
	 * @return Rangeオブジェクト
	 * @throws IllegalArgumentException rangeに指定されたテーブル範囲指定文字列が正しくない場合
	 */
	public Range getRange(String range) throws IllegalArgumentException {

		validateRangeName(range);

		String rowFromStr = "";
		String colFromStr = "";
		String rowToStr = "";
		String colToStr = "";

		int rowFrom = 0;
		int colFrom = 0;
		int rowTo = 0;
		int colTo = 0;

		// range からセル範囲を得る
		Matcher matcher = _RangeParsePattern.matcher(range);
		List<String> tokens = new ArrayList<String>();

		while (matcher.find()) {
			for (int i = 1; i <= matcher.groupCount(); i++) {
				if (matcher.group(i) != null) {
					tokens.add(matcher.group(i));
				}
			}
		}

		switch (tokens.size()) {
		case 2:
			// テーブル全体、あるいは一行、あるいは1列
			rowFromStr = tokens.get(0).substring(1);
			colFromStr = tokens.get(1).substring(1);

//			// "RxCx"の場合
//			if (rowFromStr.equals("x") && colFromStr.equals("x")) {
//				rowFrom = _rows.size() - 1;
//				rowTo = rowFrom;
//				colFrom = _colSize - 1;
//				colTo = colFrom;
//				return new Range(this, rowFrom, colFrom, rowTo, colTo);
//			}

			// RnCxあるいはRxCnの場合、1行もしくは1列選択
			if (rowFromStr.equals("x")) {
				rowFrom = 0;
				rowTo = _rows.size() - 1;
			} else {
				rowFrom = Integer.parseInt(rowFromStr);
				rowTo = rowFrom;
			}
			if (colFromStr.equals("x")) {
				colFrom = 0;
				colTo = _colSize - 1;
			} else {
				colFrom = Integer.parseInt(colFromStr);
				colTo = colFrom;
			}
			break;
		case 5:
			// それ以外の範囲
			rowFromStr = tokens.get(0).substring(1);
			colFromStr = tokens.get(1).substring(1);
			// 3番目の要素は":"なので無視
			rowToStr = tokens.get(3).substring(1);
			colToStr = tokens.get(4).substring(1);

			// FromとToの指定がある場合、From側には必ず数字が入っている
			rowFrom = Integer.parseInt(rowFromStr);
			colFrom = Integer.parseInt(colFromStr);
			// To側は空の指定の場合がある
			if (rowToStr.equals("x")) {
				rowTo = _rows.size() - 1;
			} else {
				rowTo = Integer.parseInt(rowToStr);
			}
			if (colToStr.equals("x")) {
				colTo = _colSize - 1;
			} else {
				colTo = Integer.parseInt(colToStr);
			}
			break;
		default:
			// それ以外の場合、validateRangeName()でチェックできていない。RangePatternの正規表現がおかしい。
			assert false: "unknown range " + range;
		}
		// レンジを作成して返す
		return new Range(this, rowFrom, colFrom, rowTo, colTo);
	}

	/**
	 * このテーブルの指定された位置のCellオブジェクトを返します
	 * @param rowIndex 行インデックス
	 * @param colIndex 列インデックス
	 * @return 指定された位置のCellオブジェクト
	 */
	public Cell getCell(int rowIndex, int colIndex) {
		return _rows.get(rowIndex).get(colIndex);
	}


	/* (非 Javadoc)
	 * @see jp.co.nissy.jpicosheet.core.CellGroupReference#getCells()
	 */
	public Collection<Cell> getCells() {
		List<Cell> cells = new ArrayList<Cell>(_rows.size() * _rows.get(0).size());
		for (int i = 0; i < _rows.size(); i++) {
			for (int o = 0; o < _rows.get(0).size(); o++) {
				cells.add(_rows.get(i).get(o));
			}
		}
		return cells;
	}

}
