/**
 * JPicosheet: Spreadsheet engine for Java Applications
 * Copyright (C) 2011 yusuke nishikawa
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
/**
 *
 */
package jp.co.nissy.jpicosheet.core;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * テーブルの一部または全部の範囲を表すオブジェクトです。<br>
 * オブジェクトの作成時に元とするテーブルオブジェクトの範囲を指定します。指定した範囲の左上のセル位置が
 * 新しく作成するこのオブジェクトの左上位置となります。<br>
 * このオブジェクトが示す範囲は元となるTableオブジェクトに対し固定的です。例えばこのオブジェクトが示す
 * 範囲にかかるようにTableオブジェクトから行を挿入した場合、このオブジェクトからアクセスできるセルは
 * 挿入された行を含み、それまで範囲の最下行にあった行は範囲から外れ、アクセスできなくなります。
 *
 * @author yusuke nishikawa
 *
 */
public class Range implements CellGroupReference {

	/**
	 * この範囲の元となるTableオブジェクト
	 */
	private Table _table;
	/**
	 * 行の開始位置の、Tableに対するオフセット
	 */
	private int _offsetRowFrom;
	/**
	 * 行の終了位置の、Tableに対するオフセット
	 */
	private int _offsetRowTo;
	/**
	 * 列の開始位置の、Tableに対するオフセット
	 */
	private int _offsetColFrom;
	/**
	 * 列の終了位置の、Tableに対するオフセット
	 */
	private int _offsetColTo;


	/**
	 * 指定されたテーブルの指定範囲を表すRangeオブジェクトを作ります。<br>
	 * @param table 元とするTableオブジェクト
	 * @param rowNumFrom tableの行の開始位置
	 * @param colNumFrom tableの列の開始位置
	 * @param rowNumTo tableの行の終了位置
	 * @param colNumTo tableの列の終了位置
	 */
	Range(Table table, int rowNumFrom, int colNumFrom, int rowNumTo, int colNumTo) {

		// tableはnullであってはならない
		if (table == null) {
			throw new NullPointerException("table is null.");
		}

		_table = table;

		// rowNumFrom, colNumFromは0以上でなければならず、 rowNumTo, colNumToはそれぞれのFrom以上でなければならない
		if ((rowNumFrom < 0) ||
				(colNumFrom < 0) ||
				(rowNumTo < rowNumFrom) ||
				(colNumTo < colNumFrom) ) {
			throw new IllegalArgumentException("ingalid row/col parameter." +
					" rowNumFrom=" + Integer.toString(rowNumFrom) +
					" colNumFrom=" + Integer.toString(colNumFrom) +
					" rowNumTo=" + Integer.toString(rowNumTo) +
					" colNumTo=" + Integer.toString(colNumTo));
		}

		_offsetRowFrom = rowNumFrom;
		_offsetColFrom = colNumFrom;
		_offsetRowTo = rowNumTo;
		_offsetColTo = colNumTo;

	}


	/**
	 * 指定した位置のセルを返します
	 * @param rowPos このレンジに対する行位置
	 * @param colPos このレンジに対する列位置
	 * @return 指定した位置のセルオブジェクト
	 */
	public Cell getCell(int rowPos, int colPos) {
		return _table.getCell(_offsetRowFrom + rowPos, _offsetColFrom + colPos);
	}


	/**
	 * 指定した行を返します
	 * @param rowPos このレンジに対する行位置
	 * @return 指定した行のセルオブジェクト配列
	 */
	public Cell[] getRow(int rowPos) {
		Cell[] cells = new Cell[_offsetColTo - _offsetColFrom + 1];
		for (int i = 0; i < cells.length; i++) {
			cells[i] = _table.getCell(_offsetRowFrom + rowPos, _offsetColFrom + i);
		}
		return cells;
	}


	/**
	 * 指定した列を返します
	 * @param colPos このレンジに対する列位置
	 * @return 指定した列のセルオブジェクト配列
	 */
	public Cell[] getCol(int colPos) {
		Cell[] cols = new Cell[_offsetRowTo - _offsetRowFrom + 1];
		for (int i = 0; i < cols.length; i++) {
			cols[i] = _table.getCell(_offsetRowFrom + i, _offsetColFrom + colPos);
		}
		return cols;
	}


	/* (非 Javadoc)
	 * @see jp.co.nissy.jpicosheet.core.CellGroupReference#getCells()
	 */
	public Collection<Cell> getCells() {
		// Tableには同一セルが複数含まれている場合があるため、SetでなくListを使う必要がある。
		int rowSize = _offsetRowTo - _offsetRowFrom + 1;
		int colSize = _offsetColTo - _offsetColFrom + 1;
		List<Cell> cells = new ArrayList<Cell>(rowSize * colSize);
		for (int i = 0; i < rowSize; i++) {
			for (int o = 0; o < colSize; o++) {
				cells.add(_table.getCell(_offsetRowFrom + i, _offsetColFrom + o));
			}
		}
		return cells;
	}

	/**
	 * このレンジの行数を返します
	 * @return このレンジの行数
	 */
	public int getRowSize() {
		return _offsetRowTo - _offsetRowFrom + 1;
	}

	/**
	 * このレンジの列数を返します
	 * @return このレンジの列数
	 */
	public int getColSize() {
		return _offsetColTo - _offsetColFrom + 1;
	}

}
