/*
 * Copyright (c) 2008,2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.nukimas3;

import java.nio.FloatBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import javax.media.opengl.GL2;
import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.BlendMode;
import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableInteger;
import ch.kuramo.javie.api.IAnimatableLayerReference;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.IObjectArray;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.ShaderType;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.IVideoBuffer.TextureWrapMode;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IBlendSupport;
import ch.kuramo.javie.api.services.IBlurSupport;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.api.services.IBlurSupport.BlurDimensions;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.nukimas3.Imas2StageDifferenceKey2",
		category=Nukimas3Plugin.CATEGORY_NUKIMAS3)
public class Imas2StageDifferenceKey2 {

	public enum Output {
		RESULT_FROM_STAGE1,
		RESULT_FROM_STAGE2,
		RESULT_FROM_STAGE3,
		RESULT_FROM_STAGE4,

		RESULT_FROM_STAGE1_UM,
		RESULT_FROM_STAGE2_UM,
		RESULT_FROM_STAGE3_UM,
		RESULT_FROM_STAGE4_UM,

		MATTE_ONLY,

		STAGE1,
		STAGE2,
		STAGE3,
		STAGE4,

		STAGE1_UM,
		STAGE2_UM,
		STAGE3_UM,
		STAGE4_UM,

		DIFF,
		EDGE,
		DIFF_AND_EDGE_1,
		DIFF_AND_EDGE_2
	}

	public enum UseFleshColor {
		FLESH_COLOR_1,
		FLESH_COLOR_2,
		TWO_FLESH_COLORS
	}

	@Property
	private IAnimatableLayerReference stage2;

	@Property
	private IAnimatableLayerReference stage3;

	@Property
	private IAnimatableLayerReference stage4;

	@Property(value="40", min="0", max="100")
	private IAnimatableDouble unsharpMaskAmount;

	@Property(value="15", min="0", max="50")
	private IAnimatableDouble unsharpMaskRadius;

//	@Property("true")
//	private IAnimatableBoolean unsharpMaskFast;

	@Property(value="100", min="0")
	private IAnimatableDouble differenceThreshold;

	@Property(value="30", min="0")
	private IAnimatableDouble differenceCutoff;

	@Property(value="20", min="0", max="100")
	private IAnimatableDouble edgeThreshold;

	@Property(value="50", min="0", max="100")
	private IAnimatableDouble minEdgeCoverage;

	@Property(value="10", min="0")
	private IAnimatableInteger minRegionSize;

	@Property(value="0")
	private IAnimatableColor pupilColor;

	@Property(value="1")
	private IAnimatableColor scleraColor;

	@Property(value="0.9,0.8,0.66")
	private IAnimatableColor fleshColor1;

	@Property(value="0.9,0.74,0.55")
	private IAnimatableColor fleshColor2;

	@Property("FLESH_COLOR_1")
	private IAnimatableEnum<UseFleshColor> useFleshColor;

	@Property(value="0", min="0")
	private IAnimatableDouble detectEyeColor;

	@Property(value="0", min="0")
	private IAnimatableInteger maxEyeSize;

	@Property(value="50", min="0"/*, max="100"*/)
	private IAnimatableDouble blinkThresholdEdge;

	@Property(value="30", min="0", max="100")
	private IAnimatableDouble blinkThresholdColor;

	@Property
	private IAnimatableLayerReference blinkMask;

	@Property("DIFF_AND_EDGE_1")
	private IAnimatableEnum<Output> output;

//	@Property("false")
//	private IAnimatableBoolean changeBounds;
//
//	@Property(min="0")
//	private IAnimatableInteger changeLeft;
//
//	@Property(min="0")
//	private IAnimatableInteger changeTop;
//
//	@Property(min="0")
//	private IAnimatableInteger changeRight;
//
//	@Property(min="0")
//	private IAnimatableInteger changeBottom;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IBlurSupport blurSupport;

	private final IBlendSupport blendSupport;

	private final IArrayPools arrayPools;

	private final IShaderRegistry shaders;

	private final IShaderProgram unsharpMaskProgram;

	private final IShaderProgram luminosityProgram;

	private final IShaderProgram gradientProgram;

	private final IShaderProgram toMatteProgram;

	@Inject
	public Imas2StageDifferenceKey2(
			IVideoEffectContext context, IVideoRenderSupport support,
			IBlurSupport blurSupport, IBlendSupport blendSupport,
			IArrayPools arrayPools, IShaderRegistry shaders) {

		this.context = context;
		this.support = support;
		this.blurSupport = blurSupport;
		this.blendSupport = blendSupport;
		this.arrayPools = arrayPools;
		this.shaders = shaders;

		unsharpMaskProgram = shaders.getProgram(Imas2StageDifferenceKey2.class, "UNSHARP_MASK");
		luminosityProgram = shaders.getProgram(Imas2StageDifferenceKey2.class, "LUMINOSITY");
		gradientProgram = shaders.getProgram(Imas2StageDifferenceKey2.class, "GRADIENT");
		toMatteProgram = shaders.getProgram(Imas2StageDifferenceKey2.class, "TO_MATTE");
	}

//	private VideoBounds calcNewBounds(VideoBounds oldBounds) {
//		Resolution resolution = context.getVideoResolution();
//		int left = (int) resolution.scale(-context.value(this.changeLeft));
//		int top = (int) resolution.scale(-context.value(this.changeTop));
//		int right = (int) resolution.scale(-context.value(this.changeRight));
//		int bottom = (int) resolution.scale(-context.value(this.changeBottom));
//
//		VideoBounds newBounds = new VideoBounds(
//				oldBounds.x - left,
//				oldBounds.y - top,
//				Math.max(0, oldBounds.width + left + right),
//				Math.max(0, oldBounds.height + top + bottom));
//
//		return newBounds;
//	}
//
//	public VideoBounds getVideoBounds() {
//		VideoBounds bounds = context.getPreviousBounds();
//		if (context.value(this.changeBounds)) {
//			return calcNewBounds(bounds);
//		} else {
//			return bounds;
//		}
//	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer source1 = context.doPreviousEffect();
		VideoBounds bounds = source1.getBounds();

//		VideoBounds oldBounds = null;
//		if (context.value(this.changeBounds)) {
//			VideoBounds newBounds = calcNewBounds(bounds);
//			if (!newBounds.equals(bounds)) {
//				oldBounds = bounds;
//				bounds = newBounds;
//			}
//		}

		// RegionFilterが正常に動作するためには2x2以上のサイズが必要。
		if (bounds.width < 2 || bounds.height < 2) {
			return source1;
		}

		List<IVideoBuffer> buffers = new ArrayList<IVideoBuffer>();
		buffers.add(source1);
		try {
			// srcAndSrp[0][*] : ソース
			// srcAndSrp[1][*] : アンシャープマスク適用後
			IVideoBuffer[][] srcAndSrp = new IVideoBuffer[2][4];
			srcAndSrp[0][0] = source1;

			boolean srcGotten = getSourceBuffers(srcAndSrp[0], buffers);

//			if (oldBounds != null) {
//				changeBounds(oldBounds, bounds, srcAndSrp[0], buffers);
//			}

			if (!srcGotten) {
				buffers.remove(srcAndSrp[0][0]);
				return srcAndSrp[0][0];
			}

			Output output = context.value(this.output);

			if (isStage(output)) {
				IVideoBuffer outbuf = srcAndSrp[0][output.ordinal()-Output.STAGE1.ordinal()];
				return (outbuf != null) ? quit(outbuf, buffers) : blank(bounds);
			}

			if (doUnsharpMask(srcAndSrp, buffers)) {
				disposeSourceBuffers(srcAndSrp[0], buffers, output);
			}

			if (isStageUM(output)) {
				IVideoBuffer outbuf = srcAndSrp[1][output.ordinal()-Output.STAGE1_UM.ordinal()];
				return (outbuf != null) ? quit(outbuf, buffers) : blank(bounds);
			}

			if (output == Output.DIFF) {
				IVideoBuffer diff = createDiff(srcAndSrp[1], buffers);
				return quit(diff, buffers);
			}

			if (output == Output.EDGE) {
				IVideoBuffer edge = createEdge(bounds, srcAndSrp[1], buffers);
				return quit(edge, buffers);
			}

			IVideoBuffer diffAndEdge = createDiff(srcAndSrp[1], buffers);

			if (output == Output.DIFF_AND_EDGE_1) {
				addEdgeAndBlinkMask(srcAndSrp[1], false, diffAndEdge);
				return quit(diffAndEdge, buffers);
			}

			boolean addBlinkMask = (output != Output.DIFF_AND_EDGE_2);
			boolean hasBlinkMask = addEdgeAndBlinkMask(srcAndSrp[1], addBlinkMask, diffAndEdge);

			disposeSharpedBuffers(srcAndSrp[1], buffers, output);

			if (output == Output.DIFF_AND_EDGE_2) {
				doRegionFilters(diffAndEdge, hasBlinkMask, buffers, true);
				return quit(diffAndEdge, buffers);
			} else {
				doRegionFilters(diffAndEdge, hasBlinkMask, buffers, false);
			}

			IVideoBuffer matteBuffer = toMatteBuffer(diffAndEdge, buffers);

			if (output == Output.MATTE_ONLY) {
				return quit(matteBuffer, buffers);
			}

			IVideoBuffer stage = isResult(output) ? srcAndSrp[0][output.ordinal()-Output.RESULT_FROM_STAGE1.ordinal()]
							   : isResultUM(output) ? srcAndSrp[1][output.ordinal()-Output.RESULT_FROM_STAGE1_UM.ordinal()]
							   : null;
			if (stage != null) {
				return blendSupport.blend(matteBuffer, stage, null, BlendMode.STENCIL_ALPHA, 1.0);
			} else {
				return blank(bounds);
			}

		} finally {
			for (IVideoBuffer buf : buffers) {
				buf.dispose();
			}
		}
	}

	private boolean getSourceBuffers(IVideoBuffer[] sources, List<IVideoBuffer> buffers) {
		boolean gotten = false;

		IAnimatableLayerReference[] refs = new IAnimatableLayerReference[] { null, stage2, stage3, stage4 };
		for (int i = 1; i < 4; ++i) {
			IVideoBuffer buf = context.getLayerVideoFrame(refs[i]);
			if (buf != null) {
				sources[i] = buf;
				buffers.add(buf);
				gotten = true;
			}
		}

		return gotten;
	}

//	private void changeBounds(
//			VideoBounds oldBounds, VideoBounds newBounds,
//			IVideoBuffer[] sources, List<IVideoBuffer> buffers) {
//
//		for (int i = 0; i < 4; ++i) {
//			final IVideoBuffer src = sources[i];
//			if (src != null) {
//				IVideoBuffer buffer = null;
//				try {
//					buffer = context.createVideoBuffer(newBounds);
//
//					final IVideoBuffer buf = buffer;
//					Runnable operation = new Runnable() {
//						public void run() {
//							support.ortho2D(buf);
//							support.quad2D(buf, src);
//						}
//					};
//
//					support.useFramebuffer(operation, 0, buffer, src);
//
//					src.dispose();
//					buffers.remove(src);
//
//					sources[i] = buffer;
//					buffers.add(buffer);
//					buffer = null;
//
//				} finally {
//					if (buffer != null) buffer.dispose();
//				}
//			}
//		}
//	}

	private boolean doUnsharpMask(IVideoBuffer[][] srcAndSrp, List<IVideoBuffer> buffers) {
		Resolution resolution = context.getVideoResolution();
		double amount = context.value(this.unsharpMaskAmount) / 100;
		double radius = resolution.scale(context.value(this.unsharpMaskRadius));
		boolean fast = true; //context.value(this.unsharpMaskFast);

		if (amount == 0 || radius == 0) {
			srcAndSrp[1] = srcAndSrp[0];
			return false;
		} else {
			for (int i = 0; i < 4; ++i) {
				IVideoBuffer src = srcAndSrp[0][i];
				if (src != null) {
					IVideoBuffer blur = blurSupport.gaussianBlur(
							src, radius, BlurDimensions.BOTH, true, fast);
					try {
						Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
						uniforms.add(new GLUniformData("srcTex", 0));
						uniforms.add(new GLUniformData("blrTex", 1));
						uniforms.add(new GLUniformData("amount", (float)amount));
						srcAndSrp[1][i] = support.useShaderProgram(unsharpMaskProgram, uniforms, null, src, blur);
						buffers.add(srcAndSrp[1][i]);
					} finally {
						blur.dispose();
					}
				}
			}
			return true;
		}
	}

	private void disposeSourceBuffers(IVideoBuffer[] sources, List<IVideoBuffer> buffers, Output output) {
		int o = output.ordinal() - Output.RESULT_FROM_STAGE1.ordinal();
		for (int i = 0; i < 4; ++i) {
			IVideoBuffer src;
			if (i != o && (src = sources[i]) != null) {
				src.dispose();
				sources[i] = null;
				buffers.remove(src);
			}
		}
	}

	private void disposeSharpedBuffers(IVideoBuffer[] sharped, List<IVideoBuffer> buffers, Output output) {
		int o = output.ordinal() - Output.RESULT_FROM_STAGE1_UM.ordinal();
		for (int i = 0; i < 4; ++i) {
			IVideoBuffer srp;
			if (i != o && (srp = sharped[i]) != null) {
				srp.dispose();
				sharped[i] = null;
				buffers.remove(srp);
			}
		}
	}

	private IVideoBuffer createDiff(IVideoBuffer[] sharped, List<IVideoBuffer> buffers) {
		double threshold = context.value(this.differenceThreshold) / 1000;
		double cutoff = context.value(this.differenceCutoff) / 1000;

		Color pupilColor = context.value(this.pupilColor);
		Color scleraColor = context.value(this.scleraColor);
		Color fleshColor1 = null;
		Color fleshColor2 = null;

		UseFleshColor useFleshColor = context.value(this.useFleshColor);
		switch (useFleshColor) {
			case FLESH_COLOR_1:
				fleshColor1 = context.value(this.fleshColor1);
				break;
			case FLESH_COLOR_2:
				fleshColor1 = context.value(this.fleshColor2);
				break;
			default:
				fleshColor1 = context.value(this.fleshColor1);
				fleshColor2 = context.value(this.fleshColor2);
				break;
		}

		double detectEyeColor = context.value(this.detectEyeColor) / 1000;

		List<IVideoBuffer> srpList = new ArrayList<IVideoBuffer>();
		Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
		for (int i = 0; i < 4; ++i) {
			if (sharped[i] != null) {
				srpList.add(sharped[i]);
				int j = srpList.size();
				uniforms.add(new GLUniformData("texture"+j, j-1));
			}
		}
		uniforms.add(new GLUniformData("threshold", (float)threshold));
		uniforms.add(new GLUniformData("t_minus_c", (float)(threshold-cutoff)));
		uniforms.add(new GLUniformData("pupilColor", 3, toFloatBuffer(pupilColor.r, pupilColor.g, pupilColor.b)));
		uniforms.add(new GLUniformData("scleraColor", 3, toFloatBuffer(scleraColor.r, scleraColor.g, scleraColor.b)));
		uniforms.add(new GLUniformData("fleshColor1", 3, toFloatBuffer(fleshColor1.r, fleshColor1.g, fleshColor1.b)));
		uniforms.add(new GLUniformData("detectEyeColor", (float)detectEyeColor));
		if (fleshColor2 != null) {
			uniforms.add(new GLUniformData("fleshColor2", 3, toFloatBuffer(fleshColor2.r, fleshColor2.g, fleshColor2.b)));
		}

		IShaderProgram program = getDiffProgram(srpList.size(), fleshColor2 != null);
		IVideoBuffer diff = support.useShaderProgram(
				program, uniforms, null, srpList.toArray(new IVideoBuffer[srpList.size()]));
		buffers.add(diff);

		return diff;
	}

	private IVideoBuffer createEdge(VideoBounds bounds, IVideoBuffer[] sharped, List<IVideoBuffer> buffers) {
		IVideoBuffer edge = context.createVideoBuffer(bounds);
		buffers.add(edge);
		edge.clear(Color.BLACK);
		addEdgeAndBlinkMask(sharped, false, edge);
		return edge;
	}

	private boolean addEdgeAndBlinkMask(IVideoBuffer[] sharped, boolean addBlinkMask, IVideoBuffer dstbuf) {
		final VideoBounds bounds = dstbuf.getBounds();
		double dx = 1.0/bounds.width;
		double dy = 1.0/bounds.height;
		FloatBuffer offset = toFloatBuffer(new double[] {
				-dx,-dy, 0,-dy, dx,-dy,
				-dx,  0,        dx,  0,
				-dx, dy, 0, dy, dx, dy
		});

		List<IVideoBuffer> buffers = new ArrayList<IVideoBuffer>();
		try {
			for (int i = 0; i < 4; ++i) {
				IVideoBuffer srp = sharped[i];
				if (srp != null) {
					Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
					uniforms.add(new GLUniformData("texture", 0));
					IVideoBuffer luma = support.useShaderProgram(
							luminosityProgram, uniforms, null, srp);
					try {
						luma.setTextureWrapMode(TextureWrapMode.CLAMP_TO_EDGE);
						uniforms.clear();
						uniforms.add(new GLUniformData("texture", 0));
						uniforms.add(new GLUniformData("offset[0]", 2, offset));
						buffers.add(support.useShaderProgram(gradientProgram, uniforms, null, luma));
					} finally {
						luma.dispose();
					}
				}
			}

			double threshold = context.value(this.edgeThreshold) / 100;

			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			for (int i = 0; i < buffers.size(); ++i) {
				uniforms.add(new GLUniformData("texture"+(i+1), i));
			}
			uniforms.add(new GLUniformData("threshold", (float)threshold));
			uniforms.add(new GLUniformData("offset[0]", 2, offset));

			IVideoBuffer blinkMask = addBlinkMask ? context.getLayerVideoFrame(this.blinkMask) : null;
			if (blinkMask != null) {
				buffers.add(blinkMask);
				uniforms.add(new GLUniformData("blinkMask", buffers.size()-1));
			} else {
				addBlinkMask = false;
			}

			Runnable operation = new Runnable() {
				public void run() {
					GL2 gl = context.getGL().getGL2();
					gl.glEnable(GL2.GL_BLEND);
					gl.glBlendFuncSeparate(GL2.GL_ONE, GL2.GL_ONE, GL2.GL_ONE, GL2.GL_ZERO);
					gl.glBlendEquation(GL2.GL_FUNC_ADD);
					support.ortho2D(bounds);
					support.quad2D(bounds, new double[][]{{0,0},{1,0},{1,1},{0,1}});
				}
			};

			IShaderProgram program = addBlinkMask ? getEdgeProgram(buffers.size()-1, true)
												  : getEdgeProgram(buffers.size(), false);
			int pushAttribs = GL2.GL_COLOR_BUFFER_BIT | GL2.GL_ENABLE_BIT;
			support.useShaderProgram(
					program, uniforms, operation, pushAttribs,
					dstbuf, buffers.toArray(new IVideoBuffer[buffers.size()]));

			return addBlinkMask;

		} finally {
			for (IVideoBuffer buf : buffers) {
				buf.dispose();
			}
		}
	}

	private FloatBuffer toFloatBuffer(double...values) {
		float[] farray = new float[values.length];
		for (int i = 0; i < values.length; ++i) {
			farray[i] = (float)values[i];
		}
		return FloatBuffer.wrap(farray);
	}

	private void doRegionFilters(
			final IVideoBuffer diffAndEdge, final boolean hasBlinkMask,
			List<IVideoBuffer> buffers, boolean preview) {

		// minRegionSize は面積なので2回scaleする。
		Resolution resolution = context.getVideoResolution();
		double minEdgeCoverage = context.value(this.minEdgeCoverage) / 100;
		double minRegionSize = resolution.scale(resolution.scale(
								context.value(this.minRegionSize)));
		double maxEyeSize = resolution.scale(context.value(this.maxEyeSize));
		double blinkThresholdEdge = context.value(this.blinkThresholdEdge) / 100;
		double blinkThresholdColor = context.value(this.blinkThresholdColor) / 100;

		boolean filter1Enabled = (minEdgeCoverage > 0 || minRegionSize > 0);
		boolean filter2Enabled = (maxEyeSize > 0);

		if (filter1Enabled || filter2Enabled) {
			MatteSetter ms;
			switch (diffAndEdge.getColorMode()) {
				case RGBA8:
					class MatteSetter8 implements MatteSetter {
						final byte[] b = (byte[]) diffAndEdge.getArray();
						public void set(int i)		{ b[i*4+2] = b[i*4+3]; }
						public void set2(int i)		{ b[i*4] = b[i*4+1] = b[i*4+2] = (byte)255; }
						public void clear(int i)	{ b[i*4+2] = 0; }
					}
					ms = hasBlinkMask ? new MatteSetter8() : new MatteSetter8() {
						public void set(int i)		{ b[i*4+2] = (byte)255; }
					};
					break;

				case RGBA16:
					class MatteSetter16 implements MatteSetter {
						final short[] s = (short[]) diffAndEdge.getArray();
						public void set(int i)		{ s[i*4+2] = s[i*4+3]; }
						public void set2(int i)		{ s[i*4] = s[i*4+1] = s[i*4+2] = (short)65535; }
						public void clear(int i)	{ s[i*4+2] = 0; }
					}
					ms = hasBlinkMask ? new MatteSetter16() : new MatteSetter16() {
						public void set(int i)		{ s[i*4+2] = (short)65535; }
					};
					break;

				default:
					class MatteSetterF implements MatteSetter {
						final float[] f = (float[]) diffAndEdge.getArray();
						public void set(int i)		{ f[i*4+2] = f[i*4+3]; }
						public void set2(int i)		{ f[i*4] = f[i*4+1] = f[i*4+2] = 1; }
						public void clear(int i)	{ f[i*4+2] = 0; }
					}
					ms = hasBlinkMask ? new MatteSetterF() : new MatteSetterF() {
						public void set(int i)		{ f[i*4+2] = 1; }
					};
					break;
			}

			RegionData data = new RegionData(diffAndEdge, hasBlinkMask);
			try {
				if (filter1Enabled) {
					RegionFilter1 filter1 = new RegionFilter1(data);
					try {
						filter1.filter(minEdgeCoverage, minRegionSize, ms);
					} finally {
						filter1.dispose();
					}
				}

				if (filter2Enabled) {
					RegionFilter2 filter2 = new RegionFilter2(data);
					try {
						filter2.filter(maxEyeSize, blinkThresholdEdge, blinkThresholdColor,
										hasBlinkMask, ms, preview);
					} finally {
						filter2.dispose();
					}
				}
			} finally {
				data.dispose();
			}
		}
	}

	private IVideoBuffer toMatteBuffer(IVideoBuffer diffAndEdge, List<IVideoBuffer> buffers) {
		Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
		uniforms.add(new GLUniformData("texture", 0));
		IVideoBuffer matteBuffer = support.useShaderProgram(toMatteProgram, uniforms, null, diffAndEdge);
		buffers.add(matteBuffer);
		return matteBuffer;
	}

	private boolean isStage(Output output) {
		int o = output.ordinal();
		return o >= Output.STAGE1.ordinal()
			&& o <= Output.STAGE4.ordinal();
	}

	private boolean isStageUM(Output output) {
		int o = output.ordinal();
		return o >= Output.STAGE1_UM.ordinal()
			&& o <= Output.STAGE4_UM.ordinal();
	}

	private boolean isResult(Output output) {
		int o = output.ordinal();
		return o >= Output.RESULT_FROM_STAGE1.ordinal()
			&& o <= Output.RESULT_FROM_STAGE4.ordinal();
	}

	private boolean isResultUM(Output output) {
		int o = output.ordinal();
		return o >= Output.RESULT_FROM_STAGE1_UM.ordinal()
			&& o <= Output.RESULT_FROM_STAGE4_UM.ordinal();
	}

	private IVideoBuffer quit(IVideoBuffer outbuf, List<IVideoBuffer> buffers) {
		buffers.remove(outbuf);
		return outbuf;
	}

	private IVideoBuffer blank(VideoBounds bounds) {
		IVideoBuffer buffer = null;
		try {
			buffer = context.createVideoBuffer(bounds);
			buffer.clear();

			IVideoBuffer result = buffer;
			buffer = null;
			return result;

		} finally {
			if (buffer != null) buffer.dispose();
		}
	}

	@ShaderSource
	public static final String[] UNSHARP_MASK = {
		"uniform sampler2D srcTex;",
		"uniform sampler2D blrTex;",
		"uniform float amount;",
		"",
		"void main(void)",
		"{",
		"	vec2 tc = gl_TexCoord[0].st;",
		"	vec4 srcColor = texture2D(srcTex, tc);",
		"	vec4 blrColor = texture2D(blrTex, tc);",
		"",
		"	srcColor.rgb = (srcColor.a > 0.0) ? srcColor.rgb/srcColor.a : vec3(0.0);",
		"	blrColor.rgb = (blrColor.a > 0.0) ? blrColor.rgb/blrColor.a : vec3(0.0);",
		"",
		"	vec3 diff = srcColor.rgb - blrColor.rgb;",
		"",
		"	gl_FragColor = vec4((srcColor.rgb+diff*amount)*srcColor.a, srcColor.a);",
		"}"
	};

	private IShaderProgram getDiffProgram(int n, boolean twoFleshColors) {
		String programName = Imas2StageDifferenceKey2.class.getName()
								+ ".DIFF_" + n + (twoFleshColors ? "_TWO_FLESH_COLORS" : "");
		IShaderProgram program = shaders.getProgram(programName);
		if (program == null) {
			String[] source = createDiffProgramSource(n, twoFleshColors);
			program = shaders.registerProgram(programName, ShaderType.FRAGMENT_SHADER, null, source);
		}
		return program;
	}

	private String[] createDiffProgramSource(int n, boolean twoFleshColors) {
		if (n < 2 || n > 4) throw new IllegalArgumentException();
		boolean tfc = twoFleshColors;

		return new String[] {
				"#define N " + n,
		  tfc ? "#define TWO_FLESH_COLORS" : "",
				"",
				"uniform sampler2D texture1;",
				"uniform sampler2D texture2;",
				"",
				"#if N >= 3",
				"	uniform sampler2D texture3;",
				"#endif",
				"",
				"#if N >= 4",
				"	uniform sampler2D texture4;",
				"#endif",
				"",
				"uniform float threshold;",
				"uniform float t_minus_c;",
				"",
				"uniform vec3 pupilColor;",
				"uniform vec3 scleraColor;",
				"uniform vec3 fleshColor1;",
				"uniform float detectEyeColor;",
				"",
				"#ifdef TWO_FLESH_COLORS",
				"	uniform vec3 fleshColor2;",
				"#endif",
				"",
				"float eyeColorDiff(vec3 color)",
				"{",
				"	float pupil  = distance(color, pupilColor);",
				"	float sclera = distance(color, scleraColor);",
				"	float flesh1 = distance(color, fleshColor1);",
				"",
				"#ifdef TWO_FLESH_COLORS",
				"	float flesh2 = distance(color, fleshColor2);",
				"	return min(min(pupil, sclera), min(flesh1, flesh2));",
				"#else",
				"	return min(min(pupil, sclera), flesh1);",
				"#endif",
				"}",
				"",
				"void main(void)",
				"{",
				"	vec2 tc = gl_TexCoord[0].st;",
				"	vec3 sum;",
				"	float sum2;",
				"	float sum3;",
				"",
				"	vec4 color1 = texture2D(texture1, tc);",
				"	color1.rgb = (color1.a > 0.0) ? color1.rgb/color1.a : vec3(0.0);",
				"	sum = color1.rgb;",
				"	sum2 = dot(color1.rgb, color1.rgb);",
				"	sum3 = eyeColorDiff(color1.rgb);",
				"",
				"	vec4 color2 = texture2D(texture2, tc);",
				"	color2.rgb = (color2.a > 0.0) ? color2.rgb/color2.a : vec3(0.0);",
				"	sum += color2.rgb;",
				"	sum2 += dot(color2.rgb, color2.rgb);",
				"	sum3 += eyeColorDiff(color2.rgb);",
				"",
				"#if N >= 3",
				"	vec4 color3 = texture2D(texture3, tc);",
				"	color3.rgb = (color3.a > 0.0) ? color3.rgb/color3.a : vec3(0.0);",
				"	sum += color3.rgb;",
				"	sum2 += dot(color3.rgb, color3.rgb);",
				"	sum3 += eyeColorDiff(color3.rgb);",
				"#endif",
				"",
				"#if N >= 4",
				"	vec4 color4 = texture2D(texture4, tc);",
				"	color4.rgb = (color4.a > 0.0) ? color4.rgb/color4.a : vec3(0.0);",
				"	sum += color4.rgb;",
				"	sum2 += dot(color4.rgb, color4.rgb);",
				"	sum3 += eyeColorDiff(color4.rgb);",
				"#endif",
				"",
				"	vec3 avg = sum / float(N);",
				"	float variance = max(sum2 / float(N) - dot(avg, avg), 0.0);",
				"	float red = clamp((threshold-sqrt(variance))/t_minus_c, 0.0, 1.0);",
				"",
				"	float blue = 1.0 - step(detectEyeColor, sum3/float(N));",
				"",
				"	gl_FragColor = vec4(red, 0.0, blue, 1.0);",
				"}"
		};
	}

	@ShaderSource
	public static final String[] LUMINOSITY = {
		"uniform sampler2D texture;",
		"",
		"const vec3 yvec = vec3(0.299, 0.587, 0.114);",
		"",
		"void main(void)",
		"{",
		"	vec4 color = texture2D(texture, gl_TexCoord[0].st);",
		"	float y = (color.a > 0.0) ? dot(color.rgb/color.a, yvec) : 0.0;",
		"	gl_FragColor = vec4(y);",
		"}"
	};

	@ShaderSource
	public static final String[] GRADIENT = {
		"uniform sampler2D texture;",
		"uniform vec2 offset[8];",
		"",
		"void main(void)",
		"{",
		"	vec2 tc = gl_TexCoord[0].st;",
		"	float y00 = texture2D(texture, tc+offset[0]).r;",
		"	float y10 = texture2D(texture, tc+offset[1]).r;",
		"	float y20 = texture2D(texture, tc+offset[2]).r;",
		"	float y01 = texture2D(texture, tc+offset[3]).r;",
		"	float y11 = texture2D(texture, tc          ).r;",
		"	float y21 = texture2D(texture, tc+offset[4]).r;",
		"	float y02 = texture2D(texture, tc+offset[5]).r;",
		"	float y12 = texture2D(texture, tc+offset[6]).r;",
		"	float y22 = texture2D(texture, tc+offset[7]).r;",
		"",
		"	float grad = max(max(abs(y10-y11), abs(y01-y11)),",
		"					 max(abs(y21-y11), abs(y12-y11)));",
		"",
		"	float lap = y00+y10+y20+y01-8.0*y11+y21+y02+y12+y22;",
		"",
		"	gl_FragColor = (lap > 0.0) ? vec4(grad, 0.0, 0.0, 1.0)",
		"							   : vec4(0.0, grad, 0.0, 1.0);",
		"}"
	};

	private IShaderProgram getEdgeProgram(int n, boolean withBlinkMask) {
		String programName = Imas2StageDifferenceKey2.class.getName()
							+ ".EDGE_" + n + (withBlinkMask ? "_BLINKMASK" : "");
		IShaderProgram program = shaders.getProgram(programName);
		if (program == null) {
			String[] source = createEdgeProgramSource(n, withBlinkMask);
			program = shaders.registerProgram(programName, ShaderType.FRAGMENT_SHADER, null, source);
		}
		return program;
	}

	private static String[] createEdgeProgramSource(int n, boolean withBlinkMask) {
		if (n < 2 || n > 4) throw new IllegalArgumentException();

		boolean b = withBlinkMask;
		return new String[] {
				"#define N " + n,
			b ? "#define BLINKMASK" : "",
				"",
				"uniform sampler2D texture1;",
				"uniform sampler2D texture2;",
				"",
				"#if N >= 3",
				"	uniform sampler2D texture3;",
				"#endif",
				"",
				"#if N >= 4",
				"	uniform sampler2D texture4;",
				"#endif",
				"",
				"uniform float threshold;",
				"uniform vec2 offset[8];",
				"",
				"#ifdef BLINKMASK",
				"	uniform sampler2D blinkMask;",
				"#endif",
				"",
				"float grad(sampler2D texture, vec2 tc)",
				"{",
				"	float grad = texture2D(texture, tc).r;",
				"	if (grad != 0.0 && (",
				"			   texture2D(texture, tc+offset[0]).g != 0.0",
				"			|| texture2D(texture, tc+offset[1]).g != 0.0",
				"			|| texture2D(texture, tc+offset[2]).g != 0.0",
				"			|| texture2D(texture, tc+offset[3]).g != 0.0",
				"			|| texture2D(texture, tc+offset[4]).g != 0.0",
				"			|| texture2D(texture, tc+offset[5]).g != 0.0",
				"			|| texture2D(texture, tc+offset[6]).g != 0.0",
				"			|| texture2D(texture, tc+offset[7]).g != 0.0)) {",
				"		return grad;",
				"	} else {",
				"		return 0.0;",
				"	}",
				"}",
				"",
				"void main(void)",
				"{",
				"	vec2 tc = gl_TexCoord[0].st;",
				"	float sum = grad(texture1, tc)",
				"			  + grad(texture2, tc)",
				"#if N >= 3",
				"			  + grad(texture3, tc)",
				"#endif",
				"#if N >= 4",
				"			  + grad(texture4, tc)",
				"#endif",
				"		;",
				"",
				"	float green = step(threshold, sum/float(N));",
				"#ifdef BLINKMASK",
				"	float alpha = texture2D(blinkMask, tc).a;",
				"	gl_FragColor = vec4(0.0, green, 0.0, alpha);",
				"#else",
				"	gl_FragColor = vec4(0.0, green, 0.0, 1.0);",
				"#endif",
				"}"
		};
	}

	@ShaderSource
	public static final String[] TO_MATTE = {
		"uniform sampler2D texture;",
		"",
		"void main(void)",
		"{",
		"	gl_FragColor = vec4(texture2D(texture, gl_TexCoord[0].st).r);",
		"}"
	};

	private class RegionData {

		private final VideoBounds bounds;

		private final IArray<byte[]> dataIA;

		private final byte[] data;


		RegionData(IVideoBuffer diffAndEdge, boolean hasBlinkMask) {
			bounds = diffAndEdge.getBounds();
			int len = bounds.width * bounds.height;
			dataIA = arrayPools.getByteArray(len);
			data = dataIA.getArray();
			initData(diffAndEdge, hasBlinkMask);
		}

		private void initData(IVideoBuffer buffer, boolean hasBlinkMask) {
			// dataの値は次のフラグのビット和
			//   bit0: 領域の内側
			//   bit1: エッジ上
			//   bit2: 黒/白/肌色（まばたき候補）
			//   bit3: (未使用)
			//   bit4: まばたき修正用マスク
			//   bit5: RegionFilter1で選択された領域

			abstract class ArrayToData {
				abstract byte toData(int i);
				byte toData(int r, int g, int b, int a)			{ return (byte)((r != 0 ? 1+32 : 0) + (g != 0 ? 2 : 0) + (b != 0 ? 4 : 0) + (a != 0 ? 16 : 0)); }
				byte toData(int r, int g, int b)				{ return (byte)((r != 0 ? 1+32 : 0) + (g != 0 ? 2 : 0) + (b != 0 ? 4 : 0) + 16); }
				byte toData(float r, float g, float b, float a)	{ return (byte)((r != 0 ? 1+32 : 0) + (g != 0 ? 2 : 0) + (b != 0 ? 4 : 0) + (a != 0 ? 16 : 0)); }
				byte toData(float r, float g, float b)			{ return (byte)((r != 0 ? 1+32 : 0) + (g != 0 ? 2 : 0) + (b != 0 ? 4 : 0) + 16); }
			}

			ArrayToData a2d;

			switch (buffer.getColorMode()) {
				case RGBA8: {
					final byte[] b = (byte[]) buffer.getArray();
					a2d = hasBlinkMask ? new ArrayToData() { byte toData(int i) { return toData(b[i*4+2], b[i*4+1], b[i*4], b[i*4+3]); } }
									   : new ArrayToData() { byte toData(int i) { return toData(b[i*4+2], b[i*4+1], b[i*4]); } };
					break;
				}
				case RGBA16: {
					final short[] s = (short[]) buffer.getArray();
					a2d = hasBlinkMask ? new ArrayToData() { byte toData(int i) { return toData(s[i*4+2], s[i*4+1], s[i*4], s[i*4+3]); } }
									   : new ArrayToData() { byte toData(int i) { return toData(s[i*4+2], s[i*4+1], s[i*4]); } };
					break;
				}
				default: {
					final float[] f = (float[]) buffer.getArray();
					a2d = hasBlinkMask ? new ArrayToData() { byte toData(int i) { return toData(f[i*4+2], f[i*4+1], f[i*4], f[i*4+3]); } }
									   : new ArrayToData() { byte toData(int i) { return toData(f[i*4+2], f[i*4+1], f[i*4]); } };
					break;
				}
			}

			int len = dataIA.getLength();
			for (int i = 0; i < len; ++i) {
				data[i] = a2d.toData(i);
			}
		}

		void dispose() {
			dataIA.release();
		}
	}

	private interface MatteSetter {
		void set(int i);
		void set2(int i);
		void clear(int i);
	}

	private class RegionFilter1 {

		private final VideoBounds bounds;

		private final byte[] data;

		// ラベル (regionLabels の要素値であり regionSizes のインデックスでもある) は 1 から始まる。
		// (0 は regionLabels において、ラベルが張られていないことをあらわす)
		// そこで、regionSizes のインデックスの扱いを簡単にするために regionSizes[0]は使用しない。
		private final IArray<int[]> regionLabelsIA;
		private final int[] regionLabels;

		private int regionCount;

		private IArray<int[]> regionSizesIA;
		private int[] regionSizes;

		private IArray<int[]> regionBoundsIA;
		private int[] regionBounds;

		private IArray<int[]> edgeLengthsIA;
		private int[] edgeLengths;

		private IArray<int[]> edgeCoveragesIA;
		private int[] edgeCoverages;


		RegionFilter1(RegionData data) {
			bounds = data.bounds;
			this.data = data.data;

			regionLabelsIA = arrayPools.getIntArray(data.dataIA.getLength());
			regionLabels = regionLabelsIA.getArray();

			regionSizesIA = arrayPools.getIntArray(4096);
			regionSizes = regionSizesIA.getArray();
			regionSizes[0] = 0;

			regionBoundsIA = arrayPools.getIntArray(4096*4);
			regionBounds = regionBoundsIA.getArray();

			edgeLengthsIA = arrayPools.getIntArray(4096);
			edgeLengths = edgeLengthsIA.getArray();

			edgeCoveragesIA = arrayPools.getIntArray(4096);
			edgeCoverages = edgeCoveragesIA.getArray();
		}

		void dispose() {
			regionLabelsIA.release();
			regionSizesIA.release();
			regionBoundsIA.release();
			edgeLengthsIA.release();
			edgeCoveragesIA.release();
		}

		void filter(double minEdgeCoverage, double minRegionSize, MatteSetter matteSetter) {
			int w = bounds.width;
			int w1 = bounds.width - 1;
			int h1 = bounds.height - 1;
			int p;
			final byte Z = 0;

			// 左上、上、右上
			add(0, 0, 0, Z, Z, data[0], data[1], data[w]);
			for (int x = 1; x < w1; ++x) {
				add(x, 0, regionLabels[x-1], Z, data[x-1], data[x], data[x+1], data[x+w]);
			}
			add(w1, 0, regionLabels[w1-1], Z, data[w1-1], data[w1], Z, data[w1+w]);

			// 左、中央、右
			for (int y = 1; y < h1; ++y) {
				add(p=y*w, regionLabels[p-w], 0, data[p-w], Z, data[p], data[p+1], data[p+w]);
				for (int x = 1; x < w1; ++x) {
					add(p=y*w+x, regionLabels[p-w], regionLabels[p-1],
							data[p-w], data[p-1], data[p], data[p+1], data[p+w]);
				}
				add(++p, regionLabels[p-w], regionLabels[p-1], data[p-w], data[p-1], data[p], Z, data[p+w]);
			}

			// 左下、下、右下
			add(p=h1*w, regionLabels[p-w], 0, data[p-w], Z, data[p], data[p+1], Z);
			for (int x = 1; x < w1; ++x) {
				add(p=h1*w+x, regionLabels[p-w], regionLabels[p-1], data[p-w], data[p-1], data[p], data[p+1], Z);
			}
			add(++p, regionLabels[p-w], regionLabels[p-1], data[p-w], data[p-1], data[p], Z, Z);


			for (int i = 1; i <= regionCount; ++i) {
				int label = i;
				int regionSize;
				while ((regionSize = regionSizes[label]) < 0) {
					label = -regionSize;
				}
				if (label != i) {
					regionSizes[i] = -label;
				}
			}

			for (int i = 1; i <= regionCount; ++i) {
				int regionSize = regionSizes[i];
				if (regionSize > 0 && (regionSize < minRegionSize
						|| edgeCoverages[i] < edgeLengths[i]*minEdgeCoverage)) {

					int left   = regionBounds[i*4  ];
					int top    = regionBounds[i*4+1];
					int right  = regionBounds[i*4+2];
					int bottom = regionBounds[i*4+3];

					for (int y = top; y < bottom; ++y) {
						for (int x = left; x < right; ++x) {
							int j = y*bounds.width + x;
							int label = regionLabels[j];
							int size = regionSizes[label];
							if (size < 0) {
								label = -size;
							}
							if (label == i) {
								data[j] &= ~32;
								matteSetter.clear(j);
							}
						}
					}
				}
			}
		}

		private void add(int pxIndex, int upperLabel, int leftLabel,
				byte upper, byte left, byte center, byte right, byte lower) {

			if ((center & 1) == 0) {
				regionLabels[pxIndex] = 0;
				return;
			}

			int x = pxIndex % bounds.width;
			int y = pxIndex / bounds.width;

			int label = 0;
			if (upperLabel > 0 && leftLabel > 0 && upperLabel != leftLabel) {
				while (regionSizes[upperLabel] < 0) {
					upperLabel = -regionSizes[upperLabel];
				}
				while (regionSizes[leftLabel] < 0) {
					leftLabel = -regionSizes[leftLabel];
				}
				if (leftLabel != upperLabel) {
					regionSizes[upperLabel] += regionSizes[leftLabel];
					regionSizes[leftLabel] = -upperLabel;

					regionBounds[upperLabel*4  ] = Math.min(regionBounds[upperLabel*4  ], regionBounds[leftLabel*4  ]);
					regionBounds[upperLabel*4+1] = Math.min(regionBounds[upperLabel*4+1], regionBounds[leftLabel*4+1]);
					regionBounds[upperLabel*4+2] = Math.max(regionBounds[upperLabel*4+2], regionBounds[leftLabel*4+2]);
					regionBounds[upperLabel*4+3] = Math.max(regionBounds[upperLabel*4+3], regionBounds[leftLabel*4+3]);

					edgeLengths[upperLabel] += edgeLengths[leftLabel];
					edgeCoverages[upperLabel] += edgeCoverages[leftLabel];

					// 次のは更新しなくても問題ない。
					//regionBounds[leftLabel*4] = -upperLabel;
					//edgeLengths[leftLabel] = -upperLabel;
					//edgeCoverages[leftLabel] = -upperLabel;
				}
				label = upperLabel;
			} else if (upperLabel > 0) {
				label = upperLabel;
			} else if (leftLabel > 0) {
				label = leftLabel;
			}

			if (label > 0) {
				while (regionSizes[label] < 0) {
					label = -regionSizes[label];
				}
				regionLabels[pxIndex] = label;
				++regionSizes[label];

				regionBounds[label*4  ] = Math.min(regionBounds[label*4  ], x);
				regionBounds[label*4+1] = Math.min(regionBounds[label*4+1], y);
				regionBounds[label*4+2] = Math.max(regionBounds[label*4+2], x+1);
				regionBounds[label*4+3] = Math.max(regionBounds[label*4+3], y+1);

			} else {
				// regionSizes[0] を使用していないため、格納できる個数は配列の長さより 1 小さい。
				if (regionSizesIA.getLength()-1 == regionCount) {
					regionSizesIA = expandArray(regionSizesIA);
					regionSizes = regionSizesIA.getArray();

					regionBoundsIA = expandArray(regionBoundsIA);
					regionBounds = regionBoundsIA.getArray();

					edgeLengthsIA = expandArray(edgeLengthsIA);
					edgeLengths = edgeLengthsIA.getArray();

					edgeCoveragesIA = expandArray(edgeCoveragesIA);
					edgeCoverages = edgeCoveragesIA.getArray();
				}

				label = ++regionCount;
				regionLabels[pxIndex] = label;
				regionSizes[label] = 1;

				regionBounds[label*4  ] = x;
				regionBounds[label*4+1] = y;
				regionBounds[label*4+2] = x+1;
				regionBounds[label*4+3] = y+1;

				edgeLengths[label] = 0;
				edgeCoverages[label] = 0;
			}

			boolean edge = ((center & 2) != 0);

			if ((upper & 1) == 0) {
				++edgeLengths[label];
				if (edge || (upper & 2) != 0) {
					++edgeCoverages[label];
				}
			}
			if ((left & 1) == 0) {
				++edgeLengths[label];
				if (edge || (left & 2) != 0) {
					++edgeCoverages[label];
				}
			}
			if ((right & 1) == 0) {
				++edgeLengths[label];
				if (edge || (right & 2) != 0) {
					++edgeCoverages[label];
				}
			}
			if ((lower & 1) == 0) {
				++edgeLengths[label];
				if (edge || (lower & 2) != 0) {
					++edgeCoverages[label];
				}
			}
		}
	}

	private class RegionFilter2 {

		private final VideoBounds bounds;

		private final byte[] data;


		private final IArray<int[]> regionLabels1IA;
		private final int[] regionLabels1;

		private int regionCount1;
		private int actualRegionCount1;

		private IArray<int[]> regionSizes1IA;
		private int[] regionSizes1;

		private IArray<int[]> regionBounds1IA;
		private int[] regionBounds1;


		private final IArray<int[]> regionLabels2IA;
		private final int[] regionLabels2;

		private int regionCount2;
		private int actualRegionCount2;

		private IArray<int[]> regionSizes2IA;
		private int[] regionSizes2;

		private IArray<int[]> regionBounds2IA;
		private int[] regionBounds2;


		RegionFilter2(RegionData data) {
			bounds = data.bounds;
			this.data = data.data;

			regionLabels1IA = arrayPools.getIntArray(data.dataIA.getLength());
			regionLabels1 = regionLabels1IA.getArray();

			regionSizes1IA = arrayPools.getIntArray(4096);
			regionSizes1 = regionSizes1IA.getArray();
			regionSizes1[0] = 0;

			regionBounds1IA = arrayPools.getIntArray(4096*5);
			regionBounds1 = regionBounds1IA.getArray();

			regionLabels2IA = arrayPools.getIntArray(data.dataIA.getLength());
			regionLabels2 = regionLabels2IA.getArray();

			regionSizes2IA = arrayPools.getIntArray(4096);
			regionSizes2 = regionSizes2IA.getArray();
			regionSizes2[0] = 0;

			regionBounds2IA = arrayPools.getIntArray(4096*5);
			regionBounds2 = regionBounds2IA.getArray();
		}

		void dispose() {
			regionLabels1IA.release();
			regionSizes1IA.release();
			regionBounds1IA.release();
			regionLabels2IA.release();
			regionSizes2IA.release();
			regionBounds2IA.release();
		}

		void filter(double maxEyeSize, double blinkThresholdEdge, double blinkThresholdColor,
				boolean hasBlinkMask, MatteSetter matteSetter, boolean preview) {

			int w = bounds.width;
			int w1 = bounds.width - 1;
			int h1 = bounds.height - 1;
			int p;

			// 左上、上、右上
			add1(0, 0, 0, data[0]);
			add2(0, 0, 0, data[0]);
			for (int x = 1; x < w1; ++x) {
				add1(x, 0, regionLabels1[x-1], data[x]);
				add2(x, 0, regionLabels2[x-1], data[x]);
			}
			add1(w1, 0, regionLabels1[w1-1], data[w1]);
			add2(w1, 0, regionLabels2[w1-1], data[w1]);

			// 左、中央、右
			for (int y = 1; y < h1; ++y) {
				add1(p=y*w, regionLabels1[p-w], 0, data[p]);
				add2(p    , regionLabels2[p-w], 0, data[p]);
				for (int x = 1; x < w1; ++x) {
					add1(p=y*w+x, regionLabels1[p-w], regionLabels1[p-1], data[p]);
					add2(p      , regionLabels2[p-w], regionLabels2[p-1], data[p]);
				}
				add1(++p, regionLabels1[p-w], regionLabels1[p-1], data[p]);
				add2(  p, regionLabels2[p-w], regionLabels2[p-1], data[p]);
			}

			// 左下、下、右下
			add1(p=h1*w, regionLabels1[p-w], 0, data[p]);
			add2(p     , regionLabels2[p-w], 0, data[p]);
			for (int x = 1; x < w1; ++x) {
				add1(p=h1*w+x, regionLabels1[p-w], regionLabels1[p-1], data[p]);
				add2(p       , regionLabels2[p-w], regionLabels2[p-1], data[p]);
			}
			add1(++p, regionLabels1[p-w], regionLabels1[p-1], data[p]);
			add2(  p, regionLabels2[p-w], regionLabels2[p-1], data[p]);


			IObjectArray<Integer[]> regionLabelsIA = null;
			try {
				regionLabelsIA = arrayPools.getObjectArray(actualRegionCount1 + actualRegionCount2);

				Object[] regionLabels = regionLabelsIA.getArray();
				int arCount = 0;

				for (int i = 1; i <= regionCount1; ++i) {
					int label = i;
					int regionSize;
					while ((regionSize = regionSizes1[label]) < 0) {
						label = -regionSize;
					}
					if (label == i) {
						int rw = regionBounds1[i*5+2] - regionBounds1[i*5  ];
						int rh = regionBounds1[i*5+3] - regionBounds1[i*5+1];
						regionBounds1[i*5+4] = Math.max(rw, (int)Math.ceil(rh*1.5));
						regionLabels[arCount++] = label;
					} else {
						regionSizes1[i] = -label;
					}
				}

				for (int i = 1; i <= regionCount2; ++i) {
					int label = i;
					int regionSize;
					while ((regionSize = regionSizes2[label]) < 0) {
						label = -regionSize;
					}
					if (label == i) {
						int rw = regionBounds2[i*5+2] - regionBounds2[i*5  ];
						int rh = regionBounds2[i*5+3] - regionBounds2[i*5+1];
						regionBounds2[i*5+4] = Math.max(rw, rh);
						regionLabels[arCount++] = -label;
					} else {
						regionSizes2[i] = -label;
					}
				}

				Arrays.sort(regionLabels, 0, arCount, new Comparator<Object>() {
					public int compare(Object o1, Object o2) {
						int label1 = (Integer) o1;
						int label2 = (Integer) o2;
						int size1 = (label1 > 0) ? regionBounds1[label1*5+4] : regionBounds2[-label1*5+4];
						int size2 = (label2 > 0) ? regionBounds1[label2*5+4] : regionBounds2[-label2*5+4];
						return size2 - size1;
					}
				});

				double maxEyeSize2 = maxEyeSize * 0.5;
				Set<Integer> exclude = new HashSet<Integer>();
				Set<Integer> regions = new LinkedHashSet<Integer>();

				for (int k = 0; k < arCount; ++k) {
					Integer label = (Integer) regionLabels[k];

					int size = (label > 0) ? regionBounds1[label*5+4] : regionBounds2[-label*5+4];
					if ((label > 0 && size > maxEyeSize) || (label < 0 && size > maxEyeSize2)) {
						continue;
					}

					int left, top, right, bottom;
					if (label > 0) {
						left   = regionBounds1[label*5  ];
						top    = regionBounds1[label*5+1];
						right  = regionBounds1[label*5+2];
						bottom = regionBounds1[label*5+3];
					} else {
						left   = regionBounds2[-label*5  ];
						top    = regionBounds2[-label*5+1];
						right  = regionBounds2[-label*5+2];
						bottom = regionBounds2[-label*5+3];
					}

					if ((right-left) + (bottom-top) < 4) {
						continue;
					}

					exclude.clear();

					boolean loop;
					do {
						loop = false;
						for (int i = 1; ; ++i) {
							int rw = right - left + i;
							int rh = bottom - top + i;
							if (Math.max(rw, rh*1.5) > maxEyeSize) break;

							int x0 = Math.max(left-i, 0);
							int y0 = Math.max(top-i, 0);
							int x1 = Math.min(right+i, bounds.width);
							int y1 = Math.min(bottom+i, bounds.height);

							regions.clear();

							if (x0 == left-i)   findRegionsV(x0  , y0  , y1  , i <= 2, exclude, regions);
							if (y0 == top-i)    findRegionsH(y0  , x0+1, x1-1, i <= 2, exclude, regions);
							if (x1 == right+i)  findRegionsV(x1-1, y0  , y1  , i <= 2, exclude, regions);
							if (y1 == bottom+i) findRegionsH(y1-1, x0+1, x1-1, i <= 2, exclude, regions);

							if (!regions.isEmpty()) {
								for (Integer label2 : regions) {
									int left2, top2, right2, bottom2;
									if (label2 > 0) {
										left2   = Math.min(regionBounds1[label2*5  ], left);
										top2    = Math.min(regionBounds1[label2*5+1], top);
										right2  = Math.max(regionBounds1[label2*5+2], right);
										bottom2 = Math.max(regionBounds1[label2*5+3], bottom);
									} else {
										left2   = Math.min(regionBounds2[-label2*5  ], left);
										top2    = Math.min(regionBounds2[-label2*5+1], top);
										right2  = Math.max(regionBounds2[-label2*5+2], right);
										bottom2 = Math.max(regionBounds2[-label2*5+3], bottom);
									}

									int rw2 = right2 - left2;
									int rh2 = bottom2 - top2;
									if (Math.max(rw2, rh2*1.5) <= maxEyeSize) {
										if (left2 != left || top2 != top || right2 != right || bottom2 != bottom) {
											left = left2;
											top = top2;
											right = right2;
											bottom = bottom2;
											loop = true;
										}
									} else {
										exclude.add(label2);
									}
								}
								break;
							}
						}
					} while (loop);

					filter(left, top, right, bottom,
							blinkThresholdEdge, blinkThresholdColor,
							hasBlinkMask, matteSetter, preview);
				}

			} finally {
				if (regionLabelsIA != null) regionLabelsIA.release();
			}
		}

		private void findRegionsH(int y, int fromX, int toX, boolean sensitive,
								Set<Integer> exclude, Set<Integer> regions) {

			int i0 = y*bounds.width;
			for (int x = fromX; x < toX; ++x) {
				findRegions(i0 + x, sensitive, exclude, regions);
			}
		}

		private void findRegionsV(int x, int fromY, int toY, boolean sensitive,
								Set<Integer> exclude, Set<Integer> regions) {

			int i = fromY*bounds.width + x;
			for (int y = fromY; y < toY; ++y, i += bounds.width) {
				findRegions(i, sensitive, exclude, regions);
			}
		}

		private void findRegions(int i, boolean sensitive, Set<Integer> exclude, Set<Integer> regions) {
			Integer label = regionLabels1[i];
			int size = regionSizes1[label];
			if (size < 0) {
				label = -size;
			}
			if (label > 0 && !exclude.contains(label) && (sensitive || regionBounds1[label*5+4] > 5)) {
				regions.add(label);
			}

			label = regionLabels2[i];
			size = regionSizes2[label];
			if (size < 0) {
				label = -size;
			}
			if (label > 0 && !exclude.contains(-label) && (sensitive || regionBounds2[label*5+4] > 5)) {
				regions.add(-label);
			}
		}

		private void filter(int left, int top, int right, int bottom,
				double blinkThresholdEdge, double blinkThresholdColor,
				boolean hasBlinkMask, MatteSetter matteSetter, boolean preview) {

			int outside = 0;
			int edge = 0;
			int eye = 0;
			int complexity = 0;

			Set<Integer> regions = new HashSet<Integer>();
			Set<Integer> exclude = new HashSet<Integer>();

			for (int y = top; y < bottom; ++y) {
				for (int x = left; x < right; ++x) {
					int j = y*bounds.width + x;

					int label1 = regionLabels1[j];
					int size1 = regionSizes1[label1];
					if (size1 < 0) {
						label1 = -size1;
					}
					if (label1 > 0 && !regions.contains(label1) && !exclude.contains(label1)) {
						int left2   = regionBounds1[label1*5  ];
						int top2    = regionBounds1[label1*5+1];
						int right2  = regionBounds1[label1*5+2];
						int bottom2 = regionBounds1[label1*5+3];
						if (left2 >= left && top2 >= top && right2 <= right && bottom2 <= bottom) {
							regions.add(label1);
						} else {
							exclude.add(label1);
						}
					}

					int label2 = regionLabels2[j];
					int size2 = regionSizes2[label2];
					if (size2 < 0) {
						label2 = -size2;
					}
					if (label2 > 0 && !regions.contains(-label2) && !exclude.contains(-label2)) {
						int left2   = regionBounds2[label2*5  ];
						int top2    = regionBounds2[label2*5+1];
						int right2  = regionBounds2[label2*5+2];
						int bottom2 = regionBounds2[label2*5+3];
						if (left2 >= left && top2 >= top && right2 <= right && bottom2 <= bottom) {
							++complexity;
							regions.add(-label2);
						} else {
							exclude.add(-label2);
						}
					}

					if ((data[j] & 32) == 0 && !(exclude.contains(label1) && exclude.contains(-label2))) {
						++outside;

						if ((data[j] & 2) != 0) {
							++edge;
						}
						if ((data[j] & 4) != 0) {
							++eye;
						}
					}
				}
			}

			double logOutside = 100*Math.log(outside/100d+1);
			if (!hasBlinkMask && complexity <= logOutside * 0.1 + 1) return;
			if (edge <= blinkThresholdEdge  * logOutside) return;
			if (eye  <= blinkThresholdColor * outside) return;

			for (int y = top; y < bottom; ++y) {
				for (int x = left; x < right; ++x) {
					int j = y*bounds.width + x;

					if ((data[j] & 16) == 0) {
						continue;
					}

					int label1 = regionLabels1[j];
					int size1 = regionSizes1[label1];
					if (size1 < 0) {
						label1 = -size1;
					}
					if (regions.contains(label1)) {
						matteSetter.set(j);
						continue;
					}

					int label2 = regionLabels2[j];
					int size2 = regionSizes2[label2];
					if (size2 < 0) {
						label2 = -size2;
					}
					if (regions.contains(-label2)) {
						matteSetter.set(j);
						continue;
					}

					// FIXME これは余計なところまで残してしまうかも。
					if ((data[j] & 2) != 0) {
						matteSetter.set(j);
					}
				}
			}

			if (preview) {
				int j0 = top*bounds.width;
				int j1 = (bottom-1)*bounds.width;
				for (int x = left; x < right; ++x) {
					matteSetter.set2(j0 + x);
					matteSetter.set2(j1 + x);
				}
				j0 = top*bounds.width + left;
				j1 = top*bounds.width + right - 1;
				for (int y = top+1; y < bottom-1; ++y) {
					matteSetter.set2(j0 += bounds.width);
					matteSetter.set2(j1 += bounds.width);
				}
			}
		}

		private void add1(int pxIndex, int upperLabel, int leftLabel, byte data) {
			if ((data & 32) != 0) {
				regionLabels1[pxIndex] = 0;
				return;
			}

			int x = pxIndex % bounds.width;
			int y = pxIndex / bounds.width;

			int label = 0;
			if (upperLabel > 0 && leftLabel > 0 && upperLabel != leftLabel) {
				while (regionSizes1[upperLabel] < 0) {
					upperLabel = -regionSizes1[upperLabel];
				}
				while (regionSizes1[leftLabel] < 0) {
					leftLabel = -regionSizes1[leftLabel];
				}
				if (leftLabel != upperLabel) {
					--actualRegionCount1;
					regionSizes1[upperLabel] += regionSizes1[leftLabel];
					regionSizes1[leftLabel] = -upperLabel;

					regionBounds1[upperLabel*5  ] = Math.min(regionBounds1[upperLabel*5  ], regionBounds1[leftLabel*5  ]);
					regionBounds1[upperLabel*5+1] = Math.min(regionBounds1[upperLabel*5+1], regionBounds1[leftLabel*5+1]);
					regionBounds1[upperLabel*5+2] = Math.max(regionBounds1[upperLabel*5+2], regionBounds1[leftLabel*5+2]);
					regionBounds1[upperLabel*5+3] = Math.max(regionBounds1[upperLabel*5+3], regionBounds1[leftLabel*5+3]);
				}
				label = upperLabel;
			} else if (upperLabel > 0) {
				label = upperLabel;
			} else if (leftLabel > 0) {
				label = leftLabel;
			}

			if (label > 0) {
				while (regionSizes1[label] < 0) {
					label = -regionSizes1[label];
				}
				regionLabels1[pxIndex] = label;
				++regionSizes1[label];

				regionBounds1[label*5  ] = Math.min(regionBounds1[label*5  ], x);
				regionBounds1[label*5+1] = Math.min(regionBounds1[label*5+1], y);
				regionBounds1[label*5+2] = Math.max(regionBounds1[label*5+2], x+1);
				regionBounds1[label*5+3] = Math.max(regionBounds1[label*5+3], y+1);

			} else {
				// regionSizes1[0] を使用していないため、格納できる個数は配列の長さより 1 小さい。
				if (regionSizes1IA.getLength()-1 == regionCount1) {
					regionSizes1IA = expandArray(regionSizes1IA);
					regionSizes1 = regionSizes1IA.getArray();

					regionBounds1IA = expandArray(regionBounds1IA);
					regionBounds1 = regionBounds1IA.getArray();
				}

				label = ++regionCount1;
				++actualRegionCount1;
				regionLabels1[pxIndex] = label;
				regionSizes1[label] = 1;

				regionBounds1[label*5  ] = x;
				regionBounds1[label*5+1] = y;
				regionBounds1[label*5+2] = x+1;
				regionBounds1[label*5+3] = y+1;
			}
		}

		private void add2(int pxIndex, int upperLabel, int leftLabel, byte data) {
			if ((data & (2 | 32)) != 0) {
				regionLabels2[pxIndex] = 0;
				return;
			}

			int x = pxIndex % bounds.width;
			int y = pxIndex / bounds.width;

			int label = 0;
			if (upperLabel > 0 && leftLabel > 0 && upperLabel != leftLabel) {
				while (regionSizes2[upperLabel] < 0) {
					upperLabel = -regionSizes2[upperLabel];
				}
				while (regionSizes2[leftLabel] < 0) {
					leftLabel = -regionSizes2[leftLabel];
				}
				if (leftLabel != upperLabel) {
					--actualRegionCount2;
					regionSizes2[upperLabel] += regionSizes2[leftLabel];
					regionSizes2[leftLabel] = -upperLabel;

					regionBounds2[upperLabel*5  ] = Math.min(regionBounds2[upperLabel*5  ], regionBounds2[leftLabel*5  ]);
					regionBounds2[upperLabel*5+1] = Math.min(regionBounds2[upperLabel*5+1], regionBounds2[leftLabel*5+1]);
					regionBounds2[upperLabel*5+2] = Math.max(regionBounds2[upperLabel*5+2], regionBounds2[leftLabel*5+2]);
					regionBounds2[upperLabel*5+3] = Math.max(regionBounds2[upperLabel*5+3], regionBounds2[leftLabel*5+3]);
				}
				label = upperLabel;
			} else if (upperLabel > 0) {
				label = upperLabel;
			} else if (leftLabel > 0) {
				label = leftLabel;
			}

			if (label > 0) {
				while (regionSizes2[label] < 0) {
					label = -regionSizes2[label];
				}
				regionLabels2[pxIndex] = label;
				++regionSizes2[label];

				regionBounds2[label*5  ] = Math.min(regionBounds2[label*5  ], x);
				regionBounds2[label*5+1] = Math.min(regionBounds2[label*5+1], y);
				regionBounds2[label*5+2] = Math.max(regionBounds2[label*5+2], x+1);
				regionBounds2[label*5+3] = Math.max(regionBounds2[label*5+3], y+1);

			} else {
				// regionSizes2[0] を使用していないため、格納できる個数は配列の長さより 1 小さい。
				if (regionSizes2IA.getLength()-1 == regionCount2) {
					regionSizes2IA = expandArray(regionSizes2IA);
					regionSizes2 = regionSizes2IA.getArray();

					regionBounds2IA = expandArray(regionBounds2IA);
					regionBounds2 = regionBounds2IA.getArray();
				}

				label = ++regionCount2;
				++actualRegionCount2;
				regionLabels2[pxIndex] = label;
				regionSizes2[label] = 1;

				regionBounds2[label*5  ] = x;
				regionBounds2[label*5+1] = y;
				regionBounds2[label*5+2] = x+1;
				regionBounds2[label*5+3] = y+1;
			}
		}
	}

	private IArray<int[]> expandArray(IArray<int[]> oldArray) {
		try {
			IArray<int[]> newArray = arrayPools.getIntArray(oldArray.getLength()*2);
			System.arraycopy(oldArray.getArray(), 0, newArray.getArray(), 0, oldArray.getLength());
			return newArray;
		} finally {
			oldArray.release();
		}
	}

}
