/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.nukimas3;

import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableLayerReference;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.ShaderType;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.services.IBlurSupport;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.api.services.IBlurSupport.BlurDimensions;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.nukimas3.Imas2DifferenceMatte",
		category=Nukimas3Plugin.CATEGORY_NUKIMAS3)
public class Imas2DifferenceMatte {

	public enum Output {
		RESULT,
//		RESULT_UM,
		SOURCE_ONLY,
		MATTE_ONLY
	}

	@Property
	private IAnimatableLayerReference differenceLayer;

	@Property(value="40", min="0", max="100")
	private IAnimatableDouble unsharpMaskAmount;

	@Property(value="15", min="0", max="50")
	private IAnimatableDouble unsharpMaskRadius;

	@Property(value="0", min="0", max="100")
	private IAnimatableDouble matchingTolerance;

	@Property(value="0", min="0", max="100")
	private IAnimatableDouble matchingSoftness;

	@Property("RESULT")
	private IAnimatableEnum<Output> output;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IBlurSupport blurSupport;

	private final IShaderRegistry shaders;

	private final IShaderProgram unsharpMaskProgram;

	@Inject
	public Imas2DifferenceMatte(
			IVideoEffectContext context, IVideoRenderSupport support,
			IBlurSupport blurSupport, IShaderRegistry shaders) {

		this.context = context;
		this.support = support;
		this.blurSupport = blurSupport;
		this.shaders = shaders;

		unsharpMaskProgram = shaders.getProgram(Imas2StageDifferenceKey2.class, "UNSHARP_MASK");
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer source = context.doPreviousEffect();
		VideoBounds bounds = source.getBounds();
		if (bounds.isEmpty()) {
			return source;
		}

		IVideoBuffer diffLayer = null;
		IVideoBuffer[] sharped = null;
		try {
			Output output = context.value(this.output);
			if (output == Output.SOURCE_ONLY) {
				IVideoBuffer result = source;
				source = null;
				return result;
			}

			diffLayer = context.getLayerVideoFrame(this.differenceLayer);
			if (diffLayer == null) {
				IVideoBuffer result = source;
				source = null;
				return result;
			}

			sharped = unsharpMask(source, diffLayer);

			double tolerance = context.value(this.matchingTolerance) * Math.sqrt(3) / 100;
			double softness = context.value(this.matchingSoftness) / 100;

			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("texture1", 0));
			uniforms.add(new GLUniformData("texture2", 1));
			uniforms.add(new GLUniformData("t", (float)tolerance));
			uniforms.add(new GLUniformData("s", (float)((Math.sqrt(3)-tolerance)*softness)));

			if (output != Output.MATTE_ONLY) {
				uniforms.add(new GLUniformData("source", 2));
			}

			switch (output) {
				case RESULT:
					return support.useShaderProgram(getProgram(false), uniforms, null, sharped[0], sharped[1], source);
//				case RESULT_UM:
//					return support.useShaderProgram(getProgram(false), uniforms, null, sharped[0], sharped[1], sharped[0]);
				case MATTE_ONLY:
					return support.useShaderProgram(getProgram(true), uniforms, null, sharped[0], sharped[1]);
				default:
					// ここには到達しない。
					throw new Error();
			}

		} finally {
			if (source != null) source.dispose();
			if (diffLayer != null) diffLayer.dispose();
			if (sharped != null) {
				sharped[0].dispose();
				sharped[1].dispose();
			}
		}
	}

	private IVideoBuffer[] unsharpMask(IVideoBuffer buffer1, IVideoBuffer buffer2) {
		Resolution resolution = context.getVideoResolution();
		double amount = context.value(this.unsharpMaskAmount) / 100;
		double radius = resolution.scale(context.value(this.unsharpMaskRadius));
		boolean fast = true; //context.value(this.unsharpMaskFast);

		IVideoBuffer[] sharped = null;
		try {
			sharped = new IVideoBuffer[2];
			sharped[0] = unsharpMask(buffer1, amount, radius, fast);
			sharped[1] = unsharpMask(buffer2, amount, radius, fast);

			IVideoBuffer[] result = sharped;
			sharped = null;
			return result;

		} finally {
			if (sharped != null) {
				if (sharped[0] != null) sharped[0].dispose();
				if (sharped[1] != null) sharped[1].dispose();
			}
		}
	}

	private IVideoBuffer unsharpMask(IVideoBuffer buffer, double amount, double radius, boolean fast) {
		IVideoBuffer blur = blurSupport.gaussianBlur(
				buffer, radius, BlurDimensions.BOTH, true, fast);
		try {
			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("srcTex", 0));
			uniforms.add(new GLUniformData("blrTex", 1));
			uniforms.add(new GLUniformData("amount", (float)amount));
			return support.useShaderProgram(unsharpMaskProgram, uniforms, null, buffer, blur);
		} finally {
			blur.dispose();
		}
	}


	// 以下は ch.kuramo.javie.effects.keying.DifferenceMatte にほぼ同じものがある。

	private IShaderProgram getProgram(boolean matteOnly) {
		String programName = Imas2DifferenceMatte.class.getName()
								+ (matteOnly ? ".MATTE_ONLY" : ".DIFFERENCE_MATTE");
		IShaderProgram program = shaders.getProgram(programName);
		if (program == null) {
			String[] source = createProgramSource(matteOnly);
			program = shaders.registerProgram(programName, ShaderType.FRAGMENT_SHADER, null, source);
		}
		return program;
	}

	private String[] createProgramSource(boolean matteOnly) {
		boolean mo = matteOnly;
		return new String[] {
		   mo ? "#define MATTE_ONLY" : "",
				"",
				"uniform sampler2D texture1;",
				"uniform sampler2D texture2;",
				"uniform float t;",
				"uniform float s;",
				"",
				"#ifndef MATTE_ONLY",
				"	uniform sampler2D source;",
				"#endif",
				"",
				"void main(void)",
				"{",
				"	vec2 tc = gl_TexCoord[0].st;",
				"",
				"	vec4 color1 = texture2D(texture1, tc);",
				"	color1.rgb = (color1.a > 0.0) ? color1.rgb/color1.a : vec3(0.0);",
				"",
				"	vec4 color2 = texture2D(texture2, tc);",
				"	color2.rgb = (color2.a > 0.0) ? color2.rgb/color2.a : vec3(0.0);",
				"",
				"	float d = distance(color1.rgb, color2.rgb);",
				"	d = clamp((d-t)/s, 0.0, 1.0);",
				"",
				"#ifdef MATTE_ONLY",
				"	gl_FragColor = vec4(d);",
				"#else",
				"	gl_FragColor = texture2D(source, tc) * d;",
				"#endif",
				"}"
		};
	}

}
