/*
 * Copyright (c) 2008,2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.nukimas3;

import java.nio.FloatBuffer;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.media.opengl.GL2;
import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.BlendMode;
import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableInteger;
import ch.kuramo.javie.api.IAnimatableLayerReference;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.IVideoBuffer.TextureWrapMode;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IBlendSupport;
import ch.kuramo.javie.api.services.IBlurSupport;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.api.services.IBlurSupport.BlurDimensions;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.nukimas3.Imas2StageDifferenceKey",
		category=Nukimas3Plugin.CATEGORY_NUKIMAS3)
public class Imas2StageDifferenceKey {

	public enum Output {
		RESULT_FROM_STAGE1,
		RESULT_FROM_STAGE2,
		RESULT_FROM_STAGE3,
		RESULT_FROM_STAGE4,

		RESULT_FROM_STAGE1_UM,
		RESULT_FROM_STAGE2_UM,
		RESULT_FROM_STAGE3_UM,
		RESULT_FROM_STAGE4_UM,

		MASK,

		STAGE1,
		STAGE2,
		STAGE3,
		STAGE4,

		STAGE1_UM,
		STAGE2_UM,
		STAGE3_UM,
		STAGE4_UM,

		DIFFERENCE,
		EDGE,
		DIFFERENCE_AND_EDGE
	}

	@Property
	private IAnimatableLayerReference stage2;

	@Property
	private IAnimatableLayerReference stage3;

	@Property
	private IAnimatableLayerReference stage4;

	@Property(value="40", min="0", max="100")
	private IAnimatableDouble unsharpMaskAmount;

	@Property(value="15", min="0", max="50")
	private IAnimatableDouble unsharpMaskRadius;

//	@Property("true")
//	private IAnimatableBoolean unsharpMaskFast;

	@Property(value="100", min="0")
	private IAnimatableDouble differenceThreshold;

	@Property(value="30", min="0")
	private IAnimatableDouble differenceCutoff;

	@Property(value="20", min="0", max="100")
	private IAnimatableDouble edgeThreshold;

	@Property(value="50", min="0", max="100")
	private IAnimatableDouble minEdgeCoverage;

	@Property(value="10", min="0")
	private IAnimatableInteger minRegionSize;

	@Property("DIFFERENCE_AND_EDGE")
	private IAnimatableEnum<Output> output;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IBlurSupport blurSupport;

	private final IBlendSupport blendSupport;

	private final IArrayPools arrayPools;

	private final IShaderProgram unsharpMaskProgram;

	private final IShaderProgram[] differencePrograms;

	private final IShaderProgram luminosityProgram;

	private final IShaderProgram gradientProgram;

	private final IShaderProgram[] edgePrograms;

	@Inject
	public Imas2StageDifferenceKey(
			IVideoEffectContext context, IVideoRenderSupport support,
			IBlurSupport blurSupport, IBlendSupport blendSupport,
			IArrayPools arrayPools, IShaderRegistry shaders) {

		this.context = context;
		this.support = support;
		this.blurSupport = blurSupport;
		this.blendSupport = blendSupport;
		this.arrayPools = arrayPools;

		unsharpMaskProgram = shaders.getProgram(Imas2StageDifferenceKey.class, "UNSHARP_MASK");
		differencePrograms = new IShaderProgram[] {
				shaders.getProgram(Imas2StageDifferenceKey.class, "DIFFERENCE_2"),
				shaders.getProgram(Imas2StageDifferenceKey.class, "DIFFERENCE_3"),
				shaders.getProgram(Imas2StageDifferenceKey.class, "DIFFERENCE_4")
		};
		luminosityProgram = shaders.getProgram(Imas2StageDifferenceKey.class, "LUMINOSITY");
		gradientProgram = shaders.getProgram(Imas2StageDifferenceKey.class, "GRADIENT");
		edgePrograms = new IShaderProgram[] {
				shaders.getProgram(Imas2StageDifferenceKey.class, "EDGE_2"),
				shaders.getProgram(Imas2StageDifferenceKey.class, "EDGE_3"),
				shaders.getProgram(Imas2StageDifferenceKey.class, "EDGE_4")
		};
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer source1 = context.doPreviousEffect();
		VideoBounds bounds = source1.getBounds();

		// RegionFilterが正常に動作するためには2x2以上のサイズが必要。
		if (bounds.width < 2 || bounds.height < 2) {
			return source1;
		}

		List<IVideoBuffer> buffers = new ArrayList<IVideoBuffer>();
		buffers.add(source1);
		try {
			// srcAndSrp[0][*] : ソース
			// srcAndSrp[1][*] : アンシャープマスク適用後
			IVideoBuffer[][] srcAndSrp = new IVideoBuffer[2][4];
			srcAndSrp[0][0] = source1;

			if (!getSourceBuffers(srcAndSrp[0], buffers)) {
				buffers.remove(source1);
				return source1;
			}

			Output output = context.value(this.output);

			if (isStage(output)) {
				IVideoBuffer outbuf = srcAndSrp[0][output.ordinal()-Output.STAGE1.ordinal()];
				return (outbuf != null) ? quit(outbuf, buffers) : blank(bounds);
			}

			if (doUnsharpMask(srcAndSrp, buffers)) {
				disposeSourceBuffers(srcAndSrp[0], buffers, output);
			}

			if (isStageUM(output)) {
				IVideoBuffer outbuf = srcAndSrp[1][output.ordinal()-Output.STAGE1_UM.ordinal()];
				return (outbuf != null) ? quit(outbuf, buffers) : blank(bounds);
			}

			if (output == Output.DIFFERENCE) {
				IVideoBuffer difference = createDifference(srcAndSrp[1], buffers);
				return quit(difference, buffers);
			}

			if (output == Output.EDGE) {
				IVideoBuffer edge = createEdge(bounds, srcAndSrp[1], buffers);
				return quit(edge, buffers);
			}

			IVideoBuffer diffAndEdge = createDifference(srcAndSrp[1], buffers);
			addEdge(srcAndSrp[1], diffAndEdge);

			if (output == Output.DIFFERENCE_AND_EDGE) {
				return quit(diffAndEdge, buffers);
			}

			disposeSharpedBuffers(srcAndSrp[1], buffers, output);

			filterGarbage(diffAndEdge);

			if (output == Output.MASK) {
				return quit(diffAndEdge, buffers);
			}

			IVideoBuffer stage = isResult(output) ? srcAndSrp[0][output.ordinal()-Output.RESULT_FROM_STAGE1.ordinal()]
							   : isResultUM(output) ? srcAndSrp[1][output.ordinal()-Output.RESULT_FROM_STAGE1_UM.ordinal()]
							   : null;
			if (stage != null) {
				return blendSupport.blend(diffAndEdge, stage, null, BlendMode.STENCIL_ALPHA, 1.0);
			} else {
				return blank(bounds);
			}

		} finally {
			for (IVideoBuffer buf : buffers) {
				buf.dispose();
			}
		}
	}

	private boolean getSourceBuffers(IVideoBuffer[] sources, List<IVideoBuffer> buffers) {
		boolean gotten = false;

		IAnimatableLayerReference[] refs = new IAnimatableLayerReference[] { null, stage2, stage3, stage4 };
		for (int i = 1; i < 4; ++i) {
			IVideoBuffer buf = context.getLayerVideoFrame(refs[i]);
			if (buf != null) {
				sources[i] = buf;
				buffers.add(buf);
				gotten = true;
			}
		}

		return gotten;
	}

	private boolean doUnsharpMask(IVideoBuffer[][] srcAndSrp, List<IVideoBuffer> buffers) {
		Resolution resolution = context.getVideoResolution();
		double amount = context.value(this.unsharpMaskAmount) / 100;
		double radius = resolution.scale(context.value(this.unsharpMaskRadius));
		boolean fast = true; //context.value(this.unsharpMaskFast);

		if (amount == 0 || radius == 0) {
			srcAndSrp[1] = srcAndSrp[0];
			return false;
		} else {
			for (int i = 0; i < 4; ++i) {
				IVideoBuffer src = srcAndSrp[0][i];
				if (src != null) {
					IVideoBuffer blur = blurSupport.gaussianBlur(
							src, radius, BlurDimensions.BOTH, true, fast);
					try {
						Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
						uniforms.add(new GLUniformData("srcTex", 0));
						uniforms.add(new GLUniformData("blrTex", 1));
						uniforms.add(new GLUniformData("amount", (float)amount));
						srcAndSrp[1][i] = support.useShaderProgram(unsharpMaskProgram, uniforms, null, src, blur);
						buffers.add(srcAndSrp[1][i]);
					} finally {
						blur.dispose();
					}
				}
			}
			return true;
		}
	}

	private void disposeSourceBuffers(IVideoBuffer[] sources, List<IVideoBuffer> buffers, Output output) {
		int o = output.ordinal() - Output.RESULT_FROM_STAGE1.ordinal();
		for (int i = 0; i < 4; ++i) {
			IVideoBuffer src;
			if (i != o && (src = sources[i]) != null) {
				src.dispose();
				sources[i] = null;
				buffers.remove(src);
			}
		}
	}

	private void disposeSharpedBuffers(IVideoBuffer[] sharped, List<IVideoBuffer> buffers, Output output) {
		int o = output.ordinal() - Output.RESULT_FROM_STAGE1_UM.ordinal();
		for (int i = 0; i < 4; ++i) {
			IVideoBuffer srp;
			if (i != o && (srp = sharped[i]) != null) {
				srp.dispose();
				sharped[i] = null;
				buffers.remove(srp);
			}
		}
	}

	private IVideoBuffer createDifference(IVideoBuffer[] sharped, List<IVideoBuffer> buffers) {
		double threshold = context.value(this.differenceThreshold) / 1000;
		double cutoff = context.value(this.differenceCutoff) / 1000;

		List<IVideoBuffer> srpList = new ArrayList<IVideoBuffer>();
		Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
		for (int i = 0; i < 4; ++i) {
			if (sharped[i] != null) {
				srpList.add(sharped[i]);
				int j = srpList.size();
				uniforms.add(new GLUniformData("texture"+j, j-1));
			}
		}
		uniforms.add(new GLUniformData("threshold", (float)threshold));
		uniforms.add(new GLUniformData("t_minus_c", (float)(threshold-cutoff)));

		IVideoBuffer difference = support.useShaderProgram(
				differencePrograms[srpList.size()-2], uniforms, null,
				srpList.toArray(new IVideoBuffer[srpList.size()]));
		buffers.add(difference);

		return difference;
	}

	private IVideoBuffer createEdge(VideoBounds bounds, IVideoBuffer[] sharped, List<IVideoBuffer> buffers) {
		IVideoBuffer edge = context.createVideoBuffer(bounds);
		buffers.add(edge);
		edge.clear(Color.BLACK);
		addEdge(sharped, edge);
		return edge;
	}

	private void addEdge(IVideoBuffer[] sharped, IVideoBuffer dstbuf) {
		final VideoBounds bounds = dstbuf.getBounds();
		double dx = 1.0/bounds.width;
		double dy = 1.0/bounds.height;
		FloatBuffer offset = toFloatBuffer(new double[] {
				-dx,-dy, 0,-dy, dx,-dy,
				-dx,  0,        dx,  0,
				-dx, dy, 0, dy, dx, dy
		});

		List<IVideoBuffer> grad = new ArrayList<IVideoBuffer>();
		try {
			for (int i = 0; i < 4; ++i) {
				IVideoBuffer srp = sharped[i];
				if (srp != null) {
					Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
					uniforms.add(new GLUniformData("texture", 0));
					IVideoBuffer luma = support.useShaderProgram(
							luminosityProgram, uniforms, null, srp);
					try {
						luma.setTextureWrapMode(TextureWrapMode.CLAMP_TO_EDGE);
						uniforms.clear();
						uniforms.add(new GLUniformData("texture", 0));
						uniforms.add(new GLUniformData("offset[0]", 2, offset));
						grad.add(support.useShaderProgram(gradientProgram, uniforms, null, luma));
					} finally {
						luma.dispose();
					}
				}
			}

			double threshold = context.value(this.edgeThreshold) / 100;

			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			for (int i = 0; i < grad.size(); ++i) {
				uniforms.add(new GLUniformData("texture"+(i+1), i));
			}
			uniforms.add(new GLUniformData("threshold", (float)threshold));
			uniforms.add(new GLUniformData("offset[0]", 2, offset));

			Runnable operation = new Runnable() {
				public void run() {
					GL2 gl = context.getGL().getGL2();
					gl.glEnable(GL2.GL_BLEND);
					gl.glBlendFuncSeparate(GL2.GL_ONE, GL2.GL_ONE, GL2.GL_ZERO, GL2.GL_ONE);
					gl.glBlendEquation(GL2.GL_FUNC_ADD);
					support.ortho2D(bounds);
					support.quad2D(bounds, new double[][]{{0,0},{1,0},{1,1},{0,1}});
				}
			};

			int pushAttribs = GL2.GL_COLOR_BUFFER_BIT | GL2.GL_ENABLE_BIT;
			support.useShaderProgram(
					edgePrograms[grad.size()-2], uniforms, operation, pushAttribs,
					dstbuf, grad.toArray(new IVideoBuffer[grad.size()]));

		} finally {
			for (IVideoBuffer buf : grad) {
				buf.dispose();
			}
		}
	}

	private FloatBuffer toFloatBuffer(double...values) {
		float[] farray = new float[values.length];
		for (int i = 0; i < values.length; ++i) {
			farray[i] = (float)values[i];
		}
		return FloatBuffer.wrap(farray);
	}

	private void filterGarbage(IVideoBuffer diffAndEdge) {
		Resolution resolution = context.getVideoResolution();
		double minEdgeCoverage = context.value(this.minEdgeCoverage) / 100;
		double minRegionSize = resolution.scale(resolution.scale(	// 面積なので2回scaleする。
								context.value(this.minRegionSize)));

		RegionFilter filter = new RegionFilter(diffAndEdge);
		try {
			filter.filterGarbage(minEdgeCoverage, minRegionSize);
		} finally {
			filter.dispose();
		}
	}

	private boolean isStage(Output output) {
		int o = output.ordinal();
		return o >= Output.STAGE1.ordinal()
			&& o <= Output.STAGE4.ordinal();
	}

	private boolean isStageUM(Output output) {
		int o = output.ordinal();
		return o >= Output.STAGE1_UM.ordinal()
			&& o <= Output.STAGE4_UM.ordinal();
	}

	private boolean isResult(Output output) {
		int o = output.ordinal();
		return o >= Output.RESULT_FROM_STAGE1.ordinal()
			&& o <= Output.RESULT_FROM_STAGE4.ordinal();
	}

	private boolean isResultUM(Output output) {
		int o = output.ordinal();
		return o >= Output.RESULT_FROM_STAGE1_UM.ordinal()
			&& o <= Output.RESULT_FROM_STAGE4_UM.ordinal();
	}

	private IVideoBuffer quit(IVideoBuffer outbuf, List<IVideoBuffer> buffers) {
		buffers.remove(outbuf);
		return outbuf;
	}

	private IVideoBuffer blank(VideoBounds bounds) {
		IVideoBuffer buffer = null;
		try {
			buffer = context.createVideoBuffer(bounds);
			buffer.clear();

			IVideoBuffer result = buffer;
			buffer = null;
			return result;

		} finally {
			if (buffer != null) buffer.dispose();
		}
	}

	@ShaderSource
	public static final String[] UNSHARP_MASK = {
		"uniform sampler2D srcTex;",
		"uniform sampler2D blrTex;",
		"uniform float amount;",
		"",
		"void main(void)",
		"{",
		"	vec2 tc = gl_TexCoord[0].st;",
		"	vec4 srcColor = texture2D(srcTex, tc);",
		"	vec4 blrColor = texture2D(blrTex, tc);",
		"",
		"	srcColor.rgb = (srcColor.a > 0.0) ? srcColor.rgb/srcColor.a : vec3(0.0);",
		"	blrColor.rgb = (blrColor.a > 0.0) ? blrColor.rgb/blrColor.a : vec3(0.0);",
		"",
		"	vec3 diff = srcColor.rgb - blrColor.rgb;",
		"",
		"	gl_FragColor = vec4((srcColor.rgb+diff*amount)*srcColor.a, srcColor.a);",
		"}"
	};

	@ShaderSource
	public static final String[] DIFFERENCE_2 = createDifferenceProgramSource(2);

	@ShaderSource
	public static final String[] DIFFERENCE_3 = createDifferenceProgramSource(3);

	@ShaderSource
	public static final String[] DIFFERENCE_4 = createDifferenceProgramSource(4);


	private static String[] createDifferenceProgramSource(int n) {
		if (n < 2 || n > 4) throw new IllegalArgumentException();

		return new String[] {
				"#define N " + n,
				"",
				"uniform sampler2D texture1;",
				"uniform sampler2D texture2;",
				"",
				"#if N >= 3",
				"	uniform sampler2D texture3;",
				"#endif",
				"",
				"#if N >= 4",
				"	uniform sampler2D texture4;",
				"#endif",
				"",
				"uniform float threshold;",
				"uniform float t_minus_c;",
				"",
				"void main(void)",
				"{",
				"	vec2 tc = gl_TexCoord[0].st;",
				"	vec3 sum = vec3(0.0);",
				"	float sum2 = 0.0;",
				"",
				"	vec4 color1 = texture2D(texture1, tc);",
				"	color1.rgb = (color1.a > 0.0) ? color1.rgb/color1.a : vec3(0.0);",
				"	sum += color1.rgb;",
				"	sum2 += dot(color1.rgb, color1.rgb);",
				"",
				"	vec4 color2 = texture2D(texture2, tc);",
				"	color2.rgb = (color2.a > 0.0) ? color2.rgb/color2.a : vec3(0.0);",
				"	sum += color2.rgb;",
				"	sum2 += dot(color2.rgb, color2.rgb);",
				"",
				"#if N >= 3",
				"	vec4 color3 = texture2D(texture3, tc);",
				"	color3.rgb = (color3.a > 0.0) ? color3.rgb/color3.a : vec3(0.0);",
				"	sum += color3.rgb;",
				"	sum2 += dot(color3.rgb, color3.rgb);",
				"#endif",
				"",
				"#if N >= 4",
				"	vec4 color4 = texture2D(texture4, tc);",
				"	color4.rgb = (color4.a > 0.0) ? color4.rgb/color4.a : vec3(0.0);",
				"	sum += color4.rgb;",
				"	sum2 += dot(color4.rgb, color4.rgb);",
				"#endif",
				"",
				"	vec3 avg = sum / float(N);",
				"	float variance = max(sum2 / float(N) - dot(avg, avg), 0.0);",
				"",
				"	float result = clamp((threshold-sqrt(variance))/t_minus_c, 0.0, 1.0);",
				"	gl_FragColor = vec4(result, 0.0, 0.0, 1.0);",
				"}"
		};
	}

	@ShaderSource
	public static final String[] LUMINOSITY = {
		"uniform sampler2D texture;",
		"",
		"const vec3 yvec = vec3(0.299, 0.587, 0.114);",
		"",
		"void main(void)",
		"{",
		"	vec4 color = texture2D(texture, gl_TexCoord[0].st);",
		"	float y = (color.a > 0.0) ? dot(color.rgb/color.a, yvec) : 0.0;",
		"	gl_FragColor = vec4(y);",
		"}"
	};

	@ShaderSource
	public static final String[] GRADIENT = {
		"uniform sampler2D texture;",
		"uniform vec2 offset[8];",
		"",
		"void main(void)",
		"{",
		"	vec2 tc = gl_TexCoord[0].st;",
		"	float y00 = texture2D(texture, tc+offset[0]).r;",
		"	float y10 = texture2D(texture, tc+offset[1]).r;",
		"	float y20 = texture2D(texture, tc+offset[2]).r;",
		"	float y01 = texture2D(texture, tc+offset[3]).r;",
		"	float y11 = texture2D(texture, tc          ).r;",
		"	float y21 = texture2D(texture, tc+offset[4]).r;",
		"	float y02 = texture2D(texture, tc+offset[5]).r;",
		"	float y12 = texture2D(texture, tc+offset[6]).r;",
		"	float y22 = texture2D(texture, tc+offset[7]).r;",
		"",
		"	float grad = max(max(abs(y10-y11), abs(y01-y11)),",
		"					 max(abs(y21-y11), abs(y12-y11)));",
		"",
		"	float lap = y00+y10+y20+y01-8.0*y11+y21+y02+y12+y22;",
		"",
		"	gl_FragColor = (lap > 0.0) ? vec4(grad, 0.0, 0.0, 1.0)",
		"							   : vec4(0.0, grad, 0.0, 1.0);",
		"}"
	};

	@ShaderSource
	public static final String[] EDGE_2 = createEdgeProgramSource(2);

	@ShaderSource
	public static final String[] EDGE_3 = createEdgeProgramSource(3);

	@ShaderSource
	public static final String[] EDGE_4 = createEdgeProgramSource(4);


	private static String[] createEdgeProgramSource(int n) {
		if (n < 2 || n > 4) throw new IllegalArgumentException();

		return new String[] {
				"#define N " + n,
				"",
				"uniform sampler2D texture1;",
				"uniform sampler2D texture2;",
				"",
				"#if N >= 3",
				"	uniform sampler2D texture3;",
				"#endif",
				"",
				"#if N >= 4",
				"	uniform sampler2D texture4;",
				"#endif",
				"",
				"uniform float threshold;",
				"uniform vec2 offset[8];",
				"",
				"float grad(sampler2D texture, vec2 tc)",
				"{",
				"	float grad = texture2D(texture, tc).r;",
				"	if (grad != 0.0 && (",
				"			   texture2D(texture, tc+offset[0]).g != 0.0",
				"			|| texture2D(texture, tc+offset[1]).g != 0.0",
				"			|| texture2D(texture, tc+offset[2]).g != 0.0",
				"			|| texture2D(texture, tc+offset[3]).g != 0.0",
				"			|| texture2D(texture, tc+offset[4]).g != 0.0",
				"			|| texture2D(texture, tc+offset[5]).g != 0.0",
				"			|| texture2D(texture, tc+offset[6]).g != 0.0",
				"			|| texture2D(texture, tc+offset[7]).g != 0.0)) {",
				"		return grad;",
				"	} else {",
				"		return 0.0;",
				"	}",
				"}",
				"",
				"void main(void)",
				"{",
				"	vec2 tc = gl_TexCoord[0].st;",
				"	float sum = grad(texture1, tc)",
				"			  + grad(texture2, tc)",
				"#if N >= 3",
				"			  + grad(texture3, tc)",
				"#endif",
				"#if N >= 4",
				"			  + grad(texture4, tc)",
				"#endif",
				"		;",
				"",
				"	gl_FragColor = vec4(0.0, step(threshold, sum/float(N)), 0.0, 1.0);",
				"}"
		};
	}

	private class RegionFilter {

		private final IVideoBuffer buffer;

		private final IArray<byte[]> diffAndEdgeIA;
		private final byte[] diffAndEdge;

		// ラベル (regionLabels の要素値であり regionSizes のインデックスでもある) は 1 から始まる。
		// (0 は regionLabels において、ラベルが張られていないことをあらわす)
		// そこで、regionSizes のインデックスの扱いを簡単にするために regionSizes[0]は使用しない。
		private final IArray<int[]> regionLabelsIA;
		private final int[] regionLabels;

		private int regionCount;

		private IArray<int[]> regionSizesIA;
		private int[] regionSizes;

		private IArray<int[]> edgeLengthsIA;
		private int[] edgeLengths;

		private IArray<int[]> edgeCoveragesIA;
		private int[] edgeCoverages;


		RegionFilter(IVideoBuffer diffAndEdgeBuffer) {
			buffer = diffAndEdgeBuffer;

			VideoBounds bounds = buffer.getBounds();
			int len = bounds.width * bounds.height;

			diffAndEdgeIA = arrayPools.getByteArray(len);
			diffAndEdge = diffAndEdgeIA.getArray();
			initDiffAndEdge();

			regionLabelsIA = arrayPools.getIntArray(len);
			regionLabels = regionLabelsIA.getArray();

			regionSizesIA = arrayPools.getIntArray(4096);
			regionSizes = regionSizesIA.getArray();

			edgeLengthsIA = arrayPools.getIntArray(4096);
			edgeLengths = edgeLengthsIA.getArray();

			edgeCoveragesIA = arrayPools.getIntArray(4096);
			edgeCoverages = edgeCoveragesIA.getArray();
		}

		private void initDiffAndEdge() {
			// diffAndEdgeの値は次のフラグのビット和
			//   bit0: エッジ上
			//   bit1: 領域の内側

			int len = diffAndEdgeIA.getLength();

			switch (buffer.getColorMode()) {
				case RGBA8: {
					byte[] b = (byte[]) buffer.getArray();
					for (int i = 0; i < len; ++i) {
						diffAndEdge[i] = (byte)((b[i*4+1] != 0 ? 1 : 0) + (b[i*4+2] != 0 ? 2 : 0));
					}
					break;
				}
				case RGBA16: {
					short[] s = (short[]) buffer.getArray();
					for (int i = 0; i < len; ++i) {
						diffAndEdge[i] = (byte)((s[i*4+1] != 0 ? 1 : 0) + (s[i*4+2] != 0 ? 2 : 0));
					}
					break;
				}
				default: {
					float[] f = (float[]) buffer.getArray();
					for (int i = 0; i < len; ++i) {
						diffAndEdge[i] = (byte)((f[i*4+1] != 0 ? 1 : 0) + (f[i*4+2] != 0 ? 2 : 0));
					}
					break;
				}
			}
		}

		void dispose() {
			diffAndEdgeIA.release();
			regionLabelsIA.release();
			regionSizesIA.release();
			edgeLengthsIA.release();
			edgeCoveragesIA.release();
		}

		void filterGarbage(double minEdgeCoverage, double minRegionSize) {
			VideoBounds bounds = buffer.getBounds();
			int w = bounds.width;
			int w1 = bounds.width - 1;
			int h1 = bounds.height - 1;
			int p;
			final byte Z = 0;

			// 左上、上、右上
			add(0, 0, 0, Z, Z, diffAndEdge[0], diffAndEdge[1], diffAndEdge[w]);
			for (int x = 1; x < w1; ++x) {
				add(x, 0, regionLabels[x-1], Z, diffAndEdge[x-1], diffAndEdge[x], diffAndEdge[x+1], diffAndEdge[x+w]);
			}
			add(w1, 0, regionLabels[w1-1], Z, diffAndEdge[w1-1], diffAndEdge[w1], Z, diffAndEdge[w1+w]);

			// 左、中央、右
			for (int y = 1; y < h1; ++y) {
				add(p=y*w, regionLabels[p-w], 0, diffAndEdge[p-w], Z, diffAndEdge[p], diffAndEdge[p+1], diffAndEdge[p+w]);
				for (int x = 1; x < w1; ++x) {
					add(p=y*w+x, regionLabels[p-w], regionLabels[p-1],
							diffAndEdge[p-w], diffAndEdge[p-1], diffAndEdge[p], diffAndEdge[p+1], diffAndEdge[p+w]);
				}
				add(++p, regionLabels[p-w], regionLabels[p-1], diffAndEdge[p-w], diffAndEdge[p-1], diffAndEdge[p], Z, diffAndEdge[p+w]);
			}

			// 左下、下、右下
			add(p=h1*w, regionLabels[p-w], 0, diffAndEdge[p-w], Z, diffAndEdge[p], diffAndEdge[p+1], Z);
			for (int x = 1; x < w1; ++x) {
				add(p=h1*w+x, regionLabels[p-w], regionLabels[p-1], diffAndEdge[p-w], diffAndEdge[p-1], diffAndEdge[p], diffAndEdge[p+1], Z);
			}
			add(++p, regionLabels[p-w], regionLabels[p-1], diffAndEdge[p-w], diffAndEdge[p-1], diffAndEdge[p], Z, Z);


			abstract class MaskSetter {
				abstract void set(int i);
				abstract void clear(int i);
			}

			MaskSetter ms;

			switch (buffer.getColorMode()) {
				case RGBA8:
					ms = new MaskSetter() {
						final byte[] b = (byte[]) buffer.getArray();
						void set(int i)		{ b[i*4] = b[i*4+1] = b[i*4+3] = b[i*4+2]; }
						void clear(int i)	{ b[i*4] = b[i*4+1] = b[i*4+2] = b[i*4+3] = 0; }
					};
					break;

				case RGBA16:
					ms = new MaskSetter() {
						final short[] s = (short[]) buffer.getArray();
						void set(int i)		{ s[i*4] = s[i*4+1] = s[i*4+3] = s[i*4+2]; }
						void clear(int i)	{ s[i*4] = s[i*4+1] = s[i*4+2] = s[i*4+3] = 0; }
					};
					break;

				default:
					ms = new MaskSetter() {
						final float[] f = (float[]) buffer.getArray();
						void set(int i)		{ f[i*4] = f[i*4+1] = f[i*4+3] = f[i*4+2]; }
						void clear(int i)	{ f[i*4] = f[i*4+1] = f[i*4+2] = f[i*4+3] = 0; }
					};
					break;
			}

			int len = regionLabelsIA.getLength();
			for (int i = 0; i < len; ++i) {
				int label = regionLabels[i];
				int regionSize;

				while ((regionSize = regionSizes[label]) < 0) {
					label = -regionSizes[label];
				}

				if (regionSize < minRegionSize || edgeCoverages[label] < edgeLengths[label]*minEdgeCoverage) {
					ms.clear(i);
				} else {
					ms.set(i);
				}
			}
		}

		private void add(int pxIndex, int upperLabel, int leftLabel,
				byte upper, byte left, byte center, byte right, byte lower) {

			if ((center & 2) == 0) {
				regionLabels[pxIndex] = 0;
				return;
			}

			int label = 0;
			if (upperLabel > 0 && leftLabel > 0 && upperLabel != leftLabel) {
				while (regionSizes[upperLabel] < 0) {
					upperLabel = -regionSizes[upperLabel];
				}
				while (regionSizes[leftLabel] < 0) {
					leftLabel = -regionSizes[leftLabel];
				}
				if (leftLabel != upperLabel) {
					regionSizes[upperLabel] += regionSizes[leftLabel];
					edgeLengths[upperLabel] += edgeLengths[leftLabel];
					edgeCoverages[upperLabel] += edgeCoverages[leftLabel];
					regionSizes[leftLabel] = -upperLabel;
					// 次のふたつは更新しなくても問題ない。
					//edgeLengths[leftLabel] = -upperLabel;
					//edgeCoverages[leftLabel] = -upperLabel;
				}
				label = upperLabel;
			} else if (upperLabel > 0) {
				label = upperLabel;
			} else if (leftLabel > 0) {
				label = leftLabel;
			}

			if (label > 0) {
				while (regionSizes[label] < 0) {
					label = -regionSizes[label];
				}
				regionLabels[pxIndex] = label;
				++regionSizes[label];
			} else {
				// regionSizes[0] を使用していないため、格納できる個数は配列の長さより 1 小さい。
				if (regionSizesIA.getLength()-1 == regionCount) {
					regionSizesIA = expandArray(regionSizesIA);
					regionSizes = regionSizesIA.getArray();

					edgeLengthsIA = expandArray(edgeLengthsIA);
					edgeLengths = edgeLengthsIA.getArray();

					edgeCoveragesIA = expandArray(edgeCoveragesIA);
					edgeCoverages = edgeCoveragesIA.getArray();
				}

				label = ++regionCount;
				regionLabels[pxIndex] = label;
				regionSizes[label] = 1;
				edgeLengths[label] = 0;
				edgeCoverages[label] = 0;
			}

			boolean edge = ((center & 1) != 0);

			if ((upper & 2) == 0) {
				++edgeLengths[label];
				if (edge || (upper & 1) != 0) {
					++edgeCoverages[label];
				}
			}
			if ((left & 2) == 0) {
				++edgeLengths[label];
				if (edge || (left & 1) != 0) {
					++edgeCoverages[label];
				}
			}
			if ((right & 2) == 0) {
				++edgeLengths[label];
				if (edge || (right & 1) != 0) {
					++edgeCoverages[label];
				}
			}
			if ((lower & 2) == 0) {
				++edgeLengths[label];
				if (edge || (lower & 1) != 0) {
					++edgeCoverages[label];
				}
			}
		}

		private IArray<int[]> expandArray(IArray<int[]> oldArray) {
			try {
				IArray<int[]> newArray = arrayPools.getIntArray(oldArray.getLength()*2);
				System.arraycopy(oldArray.getArray(), 0, newArray.getArray(), 0, oldArray.getLength());
				return newArray;
			} finally {
				oldArray.release();
			}
		}
	}

}
