#/*
# *  Copyright 2007 hkrn <hikarin@users.sourceforge.jp>
# *
# *  Licensed under the Apache License, Version 2.0 (the "License");
# *  you may not use this file except in compliance with the License.
# *  You may obtain a copy of the License at
# *
# *      http://www.apache.org/licenses/LICENSE-2.0
# *
# *  Unless required by applicable law or agreed to in writing, software
# *  distributed under the License is distributed on an "AS IS" BASIS,
# *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# *  See the License for the specific language governing permissions and
# *  limitations under the License.
# */
#
# $Id: UFM.pm 1285 2007-10-05 14:47:29Z hikarin $
#

package Img0ch::App::UFM;

use strict;
use Img0ch::BBS qw();
use Img0ch::Template qw();
use Img0ch::Upload qw();

my $TAG = 'BAD';

sub new {
    my ( $iClass, $iKernel, @args ) = @_;
    bless {
        __bbs     => undef,
        __config  => $iKernel->get_config(),
        __kernel  => $iKernel,
        __key     => 0,
        __resno   => 0,
        __request => Img0ch::Request->new(@args),
        __setting => undef,
        __upload  => undef,
    }, $iClass;
}

sub run {
    my ($iApp) = @_;
    $iApp->init() or return $iApp->redirect_error('init');

    my $key   = $iApp->{__key};
    my $resno = $iApp->{__resno};
    my ( $result, $iTemplate, $ctype, $body );
    if ( $key and $resno ) {
        ( $result, $iTemplate, $ctype, $body )
            = $iApp->process( $key, $resno );
    }
    else {
        ( $result, $iTemplate, $ctype, $body ) = $iApp->render();
    }

    $result or return $iApp->redirect_error('remove');
    my $iRequest = $iApp->{__request};
    $iRequest->send_http_header( $ctype || 'text/html' );
    $iTemplate ? $iTemplate->flush() : $iRequest->print($body);
    return 1;
}

sub init {
    my ($iApp) = @_;
    my $iRequest = $iApp->{__request};
    $iRequest->init( $iApp->{__config} );

    my $bbs   = $iRequest->bbs();
    my $key   = $iRequest->param_int('key');
    my $resno = $iRequest->param_int('resno');
    if ( my $path_info = $iRequest->path_info() ) {
        my @path = split '/', $path_info;
        ( $path[1] || 'test' ) =~ /([\w\-]+)/xms;
        $bbs = $1;
        ( $path[2] || '0000000000' ) =~ /(\d{9,10})/xms;
        $key = $1 eq '0000000000' ? $key : $1;
        ( $path[3] || '0' ) =~ /(\d+)/xms;
        $resno = $1 || $resno;
    }

    my $iBBS = Img0ch::BBS->new( $iApp->{__kernel}, { bbs => $bbs, } );
    if ( !$iBBS->get_id() ) {
        $iApp->{__error} = 'INVALID_POST';
        return 0;
    }

    $iApp->{__bbs}     = $iBBS;
    $iApp->{__key}     = $key;
    $iApp->{__resno}   = $resno;
    $iApp->{__setting} = $iBBS->get_setting_instance();

    return 1;
}

sub process {
    my ( $iApp, $key, $resno ) = @_;
    my $iBBS     = $iApp->{__bbs};
    my $iRequest = $iApp->{__request};
    my $iUpload  = $iApp->{__upload} = $iBBS->get_upload_instance($key);
    my $pass     = $iRequest->param('pass');
    my $type     = $iRequest->param('type');
    my $process  = {};

    if ($pass) {
        my $js = {};
        if (    $type eq 'remove'
            and $iUpload->can_remove( $resno, $pass ) )
        {
            $iUpload->remove($resno);
            $iUpload->save();
            $process->{removed} = 1;
            $js->{ok}           = 1;
        }
        elsif ( $type eq 'download'
            and $iUpload->can_download( $resno, $pass ) )
        {
            my $iKernel = $iApp->{__kernel};
            my ($ext)   = @{ $iUpload->get($resno) };
            my $path    = $iUpload->path( $resno, $ext );

            my $fh = $iKernel->get_read_file_handle($path);
            binmode $fh;
            my $buffer = do { local $/ = undef; <$fh> };
            close $fh or $iKernel->throw_io_exception($path);

            $iRequest->set_header( 'content-disposition',
                'attachment; filename="' . "${resno}.${ext}" . '"' );
            $iRequest->set_header( 'content-length', length($buffer) );

            return ( 1, undef, 'application/octet-stream', \$buffer, );
        }
        elsif ( $type eq 'confirm' ) {
            $js->{remove}   = $iUpload->can_remove( $resno,   $pass );
            $js->{download} = $iUpload->can_download( $resno, $pass );
            $process->{ok}  = 1;
        }
        if ( $iRequest->param('js') ) {
            return ( ( $process->{ok} || $process->{removed} || 0 ),
                undef, 'text/javascript', $iRequest->get_json($js) );
        }
    }
    elsif ( $type eq 'poll' ) {
        defined &{'Img0ch::Upload::Poll::new'}
            or require Img0ch::Upload::Poll;
        my $iSetting = $iApp->{__setting};
        my $iUPoll   = Img0ch::Upload::Poll->new( $iBBS, $key );
        my $agent    = $iRequest->agent();
        my $comment  = $iRequest->param('comment');
        my $ip       = $iRequest->ip();
        my $maxlen   = $iSetting->get_int('BBS_IMG_POLL_COMMENT_COUNT');
        if (    $iUpload->get($resno)->[0]
            and $iApp->validate_poll( $ip, $agent, $comment, $maxlen )
            and $iUPoll->can_poll(
                $TAG,
                $iRequest->ip(),
                $iSetting->get_int('BBS_IMG_POLL_TIME_TO_POLLABLE'),
                $iSetting->get_int('BBS_IMG_POLL_POLLABLE_COUNT')
            )
            and $iUPoll->add( $resno, $TAG, $ip, $comment )
            )
        {
            $iUPoll->save();
            $process->{polled} = 1;
            if ( $iUPoll->count( $resno, $TAG )
                >= $iSetting->get('BBS_IMG_POLL_REQUIRE_TO_STOP') )
            {
                $iUpload->freeze($resno);
                $process->{freezed} = 1;
            }
        }
        else {
            $process->{polled} = 0;
        }
    }

    return $iApp->render( $process, $key, $resno, $pass, );
}

sub load_poll {
    my ( $iApp, $key, $resno ) = @_;

    defined &{'Img0ch::Upload::Poll::new'} or require Img0ch::Upload::Poll;
    my $iUPoll = Img0ch::Upload::Poll->new( $iApp->{__bbs}, $key );
    my $ret = $iUPoll->get( $resno, $TAG );
    for my $one (@$ret) {
        my @d = localtime( $one->{polled} );
        $d[5] += 1900;
        $d[4] += 1;
        $one->{polled} = sprintf '%02d/%02d/%02d %02d:%02d:%02d', $d[5],
            $d[4], $d[3], $d[2], $d[1], $d[0];
    }

    return {
        PollForBad      => $ret,
        PollForBadCount => scalar(@$ret),
        PollableCount   => $iUPoll->get_pollable_count(
            $TAG,
            $iApp->{__request}->ip(),
            $iApp->{__setting}->get_int('BBS_IMG_POLL_POLLABLE_COUNT')
        ),
    };
}

sub validate_poll {
    my ( $iApp, $ip, $agent, $comment, $maxlen ) = @_;
    !$comment                 and return 0;
    length $comment > $maxlen and return 0;
    $agent                    and return 0;

    my $iConfig = $iApp->{__config};
    if ( !$iConfig->get('ufm.skip_check_dnsbl') ) {
        defined &{'Img0ch::Plugin::Module::DNSBL::is_in_dnsbl'}
            or require Img0ch::Plugin::Module::DNSBL;
        my $dnsbl = $iConfig->get('ufm.dnsbl_list') || 'dsbl,rbl,bbq,bbx';
        my @dnsbl_list = split ',', $dnsbl;
        map {
            Img0ch::Plugin::Module::DNSBL->is_in_dnsbl( $_, undef, $ip )
                and return 0
        } @dnsbl_list;
    }

    if ( !$iConfig->get('ufm.skip_check_ip') ) {
        defined &{'Img0ch::Filter::IP::new'} or require Img0ch::Filter::IP;
        my $iFIP = Img0ch::Filter::IP->new( $iApp->{__bbs} );
        $iFIP->load();
        $iFIP->write_ok($ip) or return 0;
    }

    return 1;
}

sub render {
    my ( $iApp, $process, $key_param, $resno_param, $pass_param ) = @_;
    my $iBBS     = $iApp->{__bbs};
    my $iConfig  = $iApp->{__config};
    my $iRequest = $iApp->{__request};
    $key_param   ||= $iApp->{__key};
    $resno_param ||= $iApp->{__resno};

    my $iSubject = $iBBS->get_subject_instance();
    my $i        = 1;
    my $stack    = [];
    for my $key ( @{ $iSubject->to_array() } ) {
        my $subj = $iSubject->get($key);
        push @{$stack},
            {
            res      => $subj->[1],
            key      => $key,
            subject  => $subj->[0],
            count    => $i,
            index    => $i,
            selected => ( $key eq $key_param ),
            };
        $i++;
    }

    my $poll = {};
    if ( $key_param and $resno_param ) {
        %$poll = %{ $iApp->load_poll( $key_param, $resno_param ) };
    }

    my $iMeta     = $iBBS->get_metadata_instance();
    my $iTemplate = $iBBS->get_template_instance(
        {   file    => 'ufm',
            request => $iRequest,
            setting => $iApp->{__setting},
        }
    );

    my $removed       = $process->{removed};
    my $really_polled = exists $process->{polled};
    my $polled_result = $process->{polled};
    $iTemplate->param(
        {   Banner          => $iMeta->main_banner(),
            DisplayPolledIP => ( $iConfig->get('ufm.no_polled_ip') ? 0 : 1 ),
            Freezed         => $process->{freezed},
            KEY             => $key_param,
            META            => $iMeta->meta(),
            Password        => $pass_param,
            PolledOK => ( $really_polled and $polled_result ),
            PolledNG => ( $really_polled and !$polled_result ),
            Removed  => $removed,
            Failed   => ( $pass_param    and !$removed ),
            Resno    => $resno_param,
            Threads  => $stack,
            %$poll,
        }
    );

    return ( 1, $iTemplate, 'text/html', undef, );
}

sub redirect_error {
    my ( $iApp, $reason ) = @_;
    my $iKernel  = $iApp->{__kernel};
    my $iRequest = $iApp->{__request};
    my $key      = $iRequest->key();
    my $ip       = $iRequest->ip_int();

    defined &{'Img0ch::Error::new'} or require Img0ch::Error;
    my $iError = Img0ch::Error->new( $iKernel, $key, $ip );
    my $long = $iError->get( $iApp->{__error} || 'INVALID_POST' );
    $iError->record();

    if ( $iRequest->param('js') ) {
        $iRequest->send_http_header( 'text/javascript', 'UTF-8' );
        $iRequest->print(
            $iRequest->get_json( { ok => 0, error => $reason } ) );
    }
    else {
        my $iTemplate = Img0ch::Template->new(
            ( $iApp->{__bbs} || $iKernel ),
            {   file    => ('error'),
                request => $iRequest,
                setting => $iApp->{__setting},
            }
        );
        $iTemplate->param(
            {   Banner       => '',
                Flag         => 0,
                LongErrorStr => $long,
            }
        );
        my $iConfig = $iApp->{__config};
        my $charset = $iConfig->get('DefaultCharset');
        $iRequest->send_http_header( 'text/html', $charset );
        $iTemplate->flush();
    }
}

1;
__END__
